/*
 * Copyright (c) 2002-2005 Endace Technology Ltd, Hamilton, New Zealand.
 * All rights reserved.
 *
 * This source code is proprietary to Endace Technology Limited and no part
 * of it may be redistributed, published or disclosed except as outlined in
 * the written contract supplied with this product.
 *
 * $Id: dagrndcap.c 11595 2009-07-21 09:02:13Z jomi $
 */

/* Endace headers. */
#include "dag_config_api.h"
#include "dag_platform.h"
#include "dagapi.h"
#include "dagclarg.h"
#include <string.h>


#define FLOW_SIZE 13
#define BUFFER_SIZE 256
#define POLYNOMIAL (0x04C11DB7L)
#define MASK_23 (0x7FFFFFL)
#define MASK_19 (0x7FFFFL)
#define MODULES 6

static void enable_disable_modules(dag_card_ref_t card, dag_component_t pc, const char* enable_options, bool enable);
static void build_hash_table(dag_card_ref_t card, dag_component_t pc, const char* filename);
static void build_flow_table(dag_card_ref_t card, dag_component_t pc, const char* filename);
static void reset_module(dag_card_ref_t card, dag_component_t pc);
static void test_get_bit_from_flow();
static uint8_t get_bit_from_flow(const uint8_t* flow, int bit_index);
static void reverse_flow(const uint8_t* in, uint8_t* out);
static void order_flow(const uint8_t* in, uint8_t* out);

char* device_name = "/dev/dag0"; /*Device name */

uint32_t register_val;
uint32_t key23 = 0x0002B;
uint32_t key19 = 0x00027;
uint8_t mask_value = 23;
//static uint32_t ethernet_crc_table[256];
static int dagstream = 0;
static char dagname[BUFFER_SIZE];
static char dagname_buf[BUFFER_SIZE] = "dag0";
char module_names[MODULES][BUFFER_SIZE] = {"all", "global", "host_hash", "host_flow", "prob_sampler", "prob_flow"};

/* CVS Header. */
static const char* const kCvsHeader __attribute__ ((unused)) = "$Id: dagrndcap.c 11595 2009-07-21 09:02:13Z jomi $";
static const char* const kRevisionString = "$Revision: 11595 $";


/* Commandline argument codes. */
enum
{
    CLA_DEVICE,
    CLA_HELP,
    CLA_VERBOSE,
    CLA_SET_HASH,
    CLA_UNSET_HASH,
    CLA_HASH_TABLE,
    CLA_FLOW_TABLE,
    CLA_RESET,
    CLA_ENABLE,
    CLA_DISABLE,
    CLA_PACKET_THRESHOLD,
    CLA_FLOW_THRESHOLD,
    CLA_DP_RESET,
    CLA_STATUS,
    CLA_VERSION
};

typedef struct
{
    bool hash_table;
    bool enable;
    bool disable;
    bool flow_table;
    bool status;
    bool flow_threshold;
    bool packet_threshold;
    bool dp_reset;
    bool reset;
} options_t;

static void
print_version(void)
{
	printf("dagrndcap (DAG %s) %s\n", kDagReleaseVersion, kRevisionString);
}

int
dagrndcap_main(int argc, char* argv[])
{
    dag_card_ref_t card_ref = NULL;
    dag_component_t root_component = NULL;
    dag_component_t pc = NULL;
    ClArgPtr clarg = NULL;
    int result = 0;
    int argindex = 0;
    int code = 0;
    char output_filename[BUFFER_SIZE];
    char hash_table_filename[BUFFER_SIZE];
    char flow_table_filename[BUFFER_SIZE];
    char enable_options[BUFFER_SIZE];
    char disable_options[BUFFER_SIZE];
	dag_reg_t reg_result[DAG_REG_MAX_ENTRIES];
    uint32_t idt_base;
    volatile uint8_t* dagiom = NULL;
    int v_level = 0;
    int k_threshold = 0;
    int l_threshold = 0;
    options_t options;
    attr_uuid_t attr = 0;
                    
    memset(output_filename, 0, BUFFER_SIZE);
    memset(&options, 0, sizeof(options));
	/* Set up default DAG device. */
	clarg = dagclarg_init(argc, (const char** const)argv);
	if (-1 == dag_parse_name(dagname_buf, dagname, BUFFER_SIZE, &dagstream))
	{
        dagutil_panic("dag_parse_name(%s): %s\n", dagname_buf, strerror(errno));
	}
    dagclarg_add_string(clarg, "DAG device to use.", "--device", 'd', "device", dagname_buf, DAGNAME_BUFSIZE, CLA_DEVICE);
	dagclarg_add(clarg, "This page.", "--help", 'h', CLA_HELP);
    dagclarg_add(clarg, "Module status. Displays which modules are enabled/disabled.", "--status", 's', CLA_STATUS);
	dagclarg_add_int(clarg, "Verbosity", "--verbosity", 'v', "level", &v_level, CLA_VERBOSE);
	dagclarg_add_int(clarg, "k value. The packet capture probability sampler threshold.", "--k_threshold", 'k', "threshold", &k_threshold, CLA_PACKET_THRESHOLD);
	dagclarg_add_int(clarg, "l value. The flow capture probability sampler threshold.", "--l_threshold", 'l', "threshold", &l_threshold, CLA_FLOW_THRESHOLD);
    dagclarg_add(clarg, "Reset the random capture module", "--reset", 'r', CLA_RESET);
	dagclarg_add(clarg, "Display version information.", "--version", 'V', CLA_VERSION);
    dagclarg_add(clarg, "Perform a datapath reset.", "--datapath", 'p', CLA_DP_RESET);
    dagclarg_add_string(clarg, "File with TCP flows.", "--hash_table", 'H', "filename", hash_table_filename, BUFFER_SIZE, CLA_HASH_TABLE);
    dagclarg_add_string(clarg, "File with TCP flows.", "--flow_table", 'F', "filename", flow_table_filename, BUFFER_SIZE, CLA_FLOW_TABLE);
    dagclarg_add_string(clarg, "Enable the modules or all of them. Available options are: all global host_flow prob_sampler prob_flow host_hash.\n"
            "\t\t\t\t\tUsage example: dagrndcap -e \"global prob_sampler host_flow\" ", "--enable", 'e', "options", enable_options, BUFFER_SIZE, CLA_ENABLE);
    dagclarg_add_string(clarg, "Disable the modules or all of them. Available options are: all global host_flow prob_sampler prob_flow host_hash.\n"
            "\t\t\t\t\tUsage example: dagrndcap -D \"global prob_sampler host_flow\" ", "--disable", 'D', "options", disable_options, BUFFER_SIZE, CLA_DISABLE);
    if(argc == 1)
    {
        dagclarg_display_usage(clarg, stdout);
        return EXIT_FAILURE;
    }
    
    dagutil_set_progname("dagrndcap");
    /* Parse the command line options. */
	result = dagclarg_parse(clarg, NULL, &argindex, &code);
    while (1 == result)
    {
        switch (code)
        {
            case CLA_DEVICE:
                if (-1 == dag_parse_name(dagname_buf, dagname, DAGNAME_BUFSIZE, &dagstream))
                {
                    dagutil_panic("dag_parse_name(%s): %s\n", dagname_buf, strerror(errno));
                }
                break;

            case CLA_HELP:
                dagclarg_display_usage(clarg, stdout);
                exit(EXIT_SUCCESS);

            case CLA_RESET:
                options.reset = true;
                break;

            case CLA_DP_RESET:
                options.dp_reset = true;
                break;


            case CLA_HASH_TABLE:
                options.hash_table = true;
                break;

            case CLA_ENABLE:
                options.enable = true;
                break;

            case CLA_DISABLE:
                options.disable = true;
                break;

            case CLA_FLOW_TABLE:
                options.flow_table = true;
                break;

            case CLA_STATUS:
                options.status = true;
                break;

            case CLA_FLOW_THRESHOLD:
                options.flow_threshold = true;
                break;

            case CLA_PACKET_THRESHOLD:
                options.packet_threshold = true;
                break;

            case CLA_VERSION:
                print_version();
                return EXIT_SUCCESS;
        }
        result = dagclarg_parse(clarg, NULL, &argindex, &code);
    }
    dagutil_set_verbosity(v_level);

    card_ref = dag_config_init(dagname);
    if (card_ref == NULL)
    {
        dagutil_panic("No card\n");
    }

    dagiom = dag_iom(dag_config_get_card_fd(card_ref));
	if ((dag_reg_find((char*)dagiom, DAG_REG_IDT_TCAM, reg_result)) == 0)
    {
        dagutil_panic("No IDT copro\n");
    }
    else
    {
		idt_base = DAG_REG_ADDR(*reg_result);
        /* init the copro */
        *(volatile uint32_t*)(dagiom + 0x20dc) = 0x40000000;
    }
    
    root_component = dag_config_get_root_component(card_ref);
    pc = dag_component_get_subcomponent(root_component, kComponentPacketCapture, 0);
    if(pc == kComponentInvalid)
    {
        /* No component exists */
        dagutil_panic("Packet Capture Module does not exist on card\n");
    }

    /*set the threshold values */
    if (options.packet_threshold)
    {
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableProbabilitySampler);
        if (dag_config_get_boolean_attribute(card_ref, attr) == 1)
            dagutil_panic("Disable the probablity sampler module before changing it's value\n");
        if (k_threshold < 1)
            k_threshold = 1;
        if (k_threshold > 100000)
            k_threshold = 100000;
        k_threshold = 0xffffffff/k_threshold;
        attr = dag_component_get_attribute_uuid(pc, kUint32AttributePacketCaptureThreshold);
        dag_config_set_uint32_attribute(card_ref, attr, k_threshold);
    }
    if (options.flow_threshold)
    {
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableProbabilityHashTable);
        if (dag_config_get_boolean_attribute(card_ref, attr) == 1)
            dagutil_panic("Disable the probablity flow sampler module before changing it's value\n");
        if (l_threshold < 1)
            l_threshold = 1;
        if (l_threshold > 100000)
            l_threshold = 100000;
        l_threshold = 0xffffffff/l_threshold;
        attr = dag_component_get_attribute_uuid(pc, kUint32AttributeFlowCaptureThreshold);
        dag_config_set_uint32_attribute(card_ref, attr, l_threshold);
    }

    if (options.reset == true)
    {
        reset_module(card_ref, pc);
    }
    
    if (options.enable)
    {
        /* process enable option string */
        enable_disable_modules(card_ref, pc, enable_options, true);
    }

    if (options.disable)
    {
        enable_disable_modules(card_ref, pc, disable_options, false);
    }
    
    if (options.dp_reset)
    {
        /* do a datapath reset */
        *(volatile uint32_t*)(dagiom + 0x88) = 0x80000000;
    }

    if (options.hash_table)
    {
        test_get_bit_from_flow();
        *(volatile uint32_t*)(dagiom + 0x20dc) = 0x20000000;
        build_hash_table(card_ref, pc, hash_table_filename);
        *(volatile uint32_t*)(dagiom + 0x20dc) = 0x40000000;
    }

    if (options.flow_table)
    {
        attr_uuid_t tcam_init;
        tcam_init = dag_component_get_attribute_uuid(pc, kBooleanAttributeTCAMInit);
        dag_config_set_boolean_attribute(card_ref, tcam_init, 0);
        *(volatile uint32_t*)(dagiom + 0x20dc) = 0x20000000;
        build_flow_table(card_ref, pc, flow_table_filename);
        *(volatile uint32_t*)(dagiom + 0x20dc) = 0x40000000;
        dag_config_set_boolean_attribute(card_ref, tcam_init, 1);
    }

    if (options.status)
    {
        attr_uuid_t attr = 0;
        bool val = 0;
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnablePacketCaptureModules);
        val = dag_config_get_boolean_attribute(card_ref, attr);
        printf("Global Enable: %s\n", val ? "enabled":"disabled");
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableHostHashTable);
        val = dag_config_get_boolean_attribute(card_ref, attr);
        printf("Host Hash Table: %s\n", val ? "enabled":"disabled");
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableHostFlowTable);
        val = dag_config_get_boolean_attribute(card_ref, attr);
        printf("Host Flow Table: %s\n", val ? "enabled":"disabled");
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableProbabilityHashTable);
        val = dag_config_get_boolean_attribute(card_ref, attr);
        printf("Probabilty Hash Table: %s\n", val ? "enabled":"disabled");
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableProbabilitySampler);
        val = dag_config_get_boolean_attribute(card_ref, attr);
        printf("Probabilty Sampler: %s\n", val ? "enabled":"disabled");
    }


    return  EXIT_SUCCESS;
}

static void test_get_bit_from_flow()
{
    uint8_t flow[FLOW_SIZE];
    int i = 0;

    memset(flow, 0xff, FLOW_SIZE);

    for (i = 0; i < 104; i++)
    {
        if (get_bit_from_flow(flow, i) != 1)
        {
            dagutil_panic("1: Testing bit extraction failed\n");
        }
    }

    memset(flow, 0xaa, FLOW_SIZE);
    for (i = 0; i < 104; i++)
    {
        if (i % 2 == 0)
        {
            if (get_bit_from_flow(flow, i) != 0)
            {
                dagutil_panic("2: Testing bit extraction failed: i = %d\n", i);
            }
        }
        else
        {
            if (get_bit_from_flow(flow, i) != 1)
            {
                dagutil_panic("3: Testing bit extraction failed: i = %d\n", i);
            }
        }
    }

    flow[0] = 1;
    if (get_bit_from_flow(flow, 0) != 1)
    {
        dagutil_panic("4: Testing bit extraction failed\n");
    }

    flow[0] = 0x80;
    if (get_bit_from_flow(flow, 7) != 1)
    {
        dagutil_panic("5: Testing bit extraction failed\n");
    }

    flow[1] = 0x80;
    if (get_bit_from_flow(flow, 15) != 1)
    {
        dagutil_panic("6: Testing bit extraction failed\n");
    }

    if (get_bit_from_flow(flow, 14) != 0)
    {
        dagutil_panic("7: Testing bit extraction failed\n");
    }
}


static uint8_t
get_bit_from_flow(const uint8_t* flow, int bit_index)
{
    /* bit_index can be from 0 to 103. flow must be 13 bytes long*/
    uint8_t ret_val = 0;
    if (bit_index < 0 || bit_index > 103)
    {
        dagutil_panic("Bad bit index\n");
    }
         
    ret_val =  flow[bit_index/8] & (1 << bit_index%8);
    ret_val >>= (bit_index%8);
    if (ret_val != 1 && ret_val != 0)
        dagutil_panic("Problem getting a bit from the flow: ret_val = 0x%08x\n", ret_val);
    return ret_val;
}

static char*
read_field_uint8(const char* start, const char separater, uint8_t* val, int base)
{
    char* p = NULL;

    p = strchr(start, (int)separater);
    if (!p)
    {
        dagutil_warning("Ignoring bad line\n.");
        return NULL;
    }
    *val = strtol(start, &p, base);
    p++;
    return p;
}

static char*
read_field_uint16(const char* start, const char separater, uint16_t* val, int base)
{
    char* p = NULL;

    p = strchr(start, (int)separater);
    if (!p)
    {
        dagutil_warning("Ignoring bad line\n.");
        return NULL;
    }
    *val = strtol(start, &p, base);
    p++;
    return p;
}

static void
build_flow_table(dag_card_ref_t card, dag_component_t pc, const char* filename)
{
    FILE* f = fopen(filename, "r");
    char line[BUFFER_SIZE];
    uint8_t flow[FLOW_SIZE];
    uint8_t flow_mask[FLOW_SIZE];
    char* start = NULL;
    uint16_t port;
    attr_uuid_t data_attr = 0;
    attr_uuid_t mask_attr = 0;
    attr_uuid_t address_attr = 0;
    attr_uuid_t mask_address_attr = 0;
    attr_uuid_t init_attr = 0;
    attr_uuid_t enable_attr = 0;
    uint32_t address = 0;
    uint32_t mask_address = 0;
    dag_component_t sc256 = 0;
    dag_component_t root = 0;
    sc256_72bit_data_t data;
    sc256_72bit_mask_t mask;

    if (!f)
        dagutil_panic("Could not open file %s\n", filename);

    enable_attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableHostFlowTable);
    if (dag_config_get_boolean_attribute(card, enable_attr) == 1)
    {
        dagutil_panic("Disable the module before loading the flow table.\n");
    }
    dagutil_verbose_level(1, "building flow table\n");
    memset(flow, 0, FLOW_SIZE);
    memset(&data, 0, sizeof(data));
    root = dag_config_get_root_component(card);
    sc256 = dag_component_get_subcomponent(root, kComponentSC256, 0);
    data_attr = dag_component_get_attribute_uuid(sc256, kStructAttributeSC25672BitData);
    address_attr = dag_component_get_attribute_uuid(sc256, kUint32AttributeSC256DataAddress);
    init_attr =  dag_component_get_attribute_uuid(sc256, kNullAttributeSC256Init);
    mask_address_attr = dag_component_get_attribute_uuid(sc256, kUint32AttributeSC256MaskAddress);
    mask_attr = dag_component_get_attribute_uuid(sc256, kStructAttributeSC25672BitMask);
    dag_config_set_null_attribute(card, init_attr);
    while (fgets(line, BUFFER_SIZE, f) != NULL)
    {
        /* ignore certain lines */
        dagutil_verbose_level(2, "Line: %s", line);
        if (line[0] == '#' || line[0] == '\n')
            continue;

        start = line;
        if (NULL == (start = read_field_uint8(start, '.',  &flow[5], 10)))
            continue;

        if (NULL == (start = read_field_uint8(start, '.',  &flow[6], 10)))
            continue;

        if (NULL == (start = read_field_uint8(start, '.',  &flow[7], 10)))
            continue;
            
        if (NULL == (start = read_field_uint8(start, ',', &flow[8], 10)))
            continue;

        if (NULL == (start = read_field_uint8(start, '.',  &flow[9], 10)))
            continue;

        if (NULL == (start = read_field_uint8(start, '.',  &flow[10], 10)))
            continue;

        if (NULL == (start = read_field_uint8(start, '.',  &flow[11], 10)))
            continue;

        if (NULL == (start = read_field_uint8(start, ',',  &flow[12], 10)))
            continue;

        if (NULL == (start = read_field_uint16(start, ',', &port, 10)))
            continue;
        flow[0] = (port >> 8);
        flow[1] = port & 0xffff;

        if (NULL == (start = read_field_uint16(start, ',', &port, 10)))
            continue;
        flow[2] = (port >> 8);
        flow[3] = port & 0xffff;

        if (NULL == (start = read_field_uint8(start, ',',  &flow[4], 10)))
            continue;

        /* read the mask information */
        if (NULL == (start = read_field_uint8(start, '.',  &flow_mask[5], 16)))
            continue;

        if (NULL == (start = read_field_uint8(start, '.',  &flow_mask[6], 16)))
            continue;

        if (NULL == (start = read_field_uint8(start, '.',  &flow_mask[7], 16)))
            continue;
            
        if (NULL == (start = read_field_uint8(start, ',',  &flow_mask[8], 16)))
            continue;

        if (NULL == (start = read_field_uint8(start, '.',  &flow_mask[9], 16)))
            continue;

        if (NULL == (start = read_field_uint8(start, '.',  &flow_mask[10], 16)))
            continue;

        if (NULL == (start = read_field_uint8(start, '.',  &flow_mask[11], 16)))
            continue;

        if (NULL == (start = read_field_uint8(start, ',',  &flow_mask[12], 16)))
            continue;

        if (NULL == (start = read_field_uint16(start, ',', &port, 16)))
            continue;
        flow_mask[0] = (port >> 8);
        flow_mask[1] = port & 0xffff;

        if (NULL == (start = read_field_uint16(start, ',', &port, 16)))
            continue;
        flow_mask[2] = (port >> 8);
        flow_mask[3] = port & 0xffff;

        if (NULL == (start = read_field_uint8(start, '\n',  &flow_mask[4], 16)))
            continue;
        
        /* sort based on higher ip address */
        order_flow(flow, flow);
        reverse_flow(flow, flow);
        dagutil_verbose_level(2, "Flow Id: 0x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\n",
                flow[0], flow[1], flow[2], flow[3], flow[4], flow[5], flow[6],
                flow[7], flow[8], flow[9], flow[10], flow[11], flow[12]);
    

        data.data0 = htonl(flow[0] << 24 | flow[1] << 16 | flow[2] << 8 | flow[3]);
        data.data1 = htonl(flow[4] << 24 | flow[5] << 16 | flow[6] << 8 | flow[7]);
        data.data2 = flow[8];

        dag_config_set_uint32_attribute(card, address_attr, address);
        address++;
        dag_config_set_struct_attribute(card, data_attr, (void*)&data);
        data.data0 = htonl(flow[9] << 24 | flow[10] << 16 | flow[11] << 8 | flow[12]);
        data.data1 = 0x0;
        data.data2 = 0x0;
        dag_config_set_uint32_attribute(card, address_attr, address);
        address++;
        dag_config_set_struct_attribute(card, data_attr, (void*)&data);

        reverse_flow(flow_mask, flow_mask);
        dagutil_verbose_level(2, "Flow Mask: 0x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\n",
                flow_mask[0], flow_mask[1], flow_mask[2], flow_mask[3], flow_mask[4], flow_mask[5], flow_mask[6],
                flow_mask[7], flow_mask[8], flow_mask[9], flow_mask[10], flow_mask[11], flow_mask[12]);

        mask.mask0 = htonl(flow_mask[0] << 24 | flow_mask[1] << 16 | flow_mask[2] << 8 | flow_mask[3]);
        mask.mask1 = htonl(flow_mask[4] << 24 | flow_mask[5] << 16 | flow_mask[6] << 8 | flow_mask[7]);
        mask.mask2 = flow_mask[8];

        dag_config_set_uint32_attribute(card, mask_attr, mask_address);
        mask_address++;
        dag_config_set_struct_attribute(card, mask_attr, (void*)&mask);
        mask.mask0 = htonl(flow_mask[9] << 24 | flow_mask[10] << 16 | flow_mask[11] << 8 | flow_mask[12]);
        mask.mask1 = 0x0;
        mask.mask2 = 0x0;
        dag_config_set_uint32_attribute(card, mask_address_attr, mask_address);
        mask_address++;
        dag_config_set_struct_attribute(card, mask_attr, (void*)&mask);
    }
    fclose(f);
}


static uint32_t
make_hash(const uint8_t* flow, uint32_t seed)
{
    /* Based on VHDL code from Erez. Apparently it does a crc32. */
    volatile uint32_t hash = 0;

    hash |=  (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 97) ^ 
             get_bit_from_flow(flow, 96) ^ get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 94) ^ get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 85) ^ get_bit_from_flow(flow, 84) ^ 
             get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 73) ^ get_bit_from_flow(flow, 72) ^ 
             get_bit_from_flow(flow, 68) ^ get_bit_from_flow(flow, 67) ^ get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 63) ^ get_bit_from_flow(flow, 61) ^ 
             get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 50) ^ 
             get_bit_from_flow(flow, 48) ^ get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 37) ^ get_bit_from_flow(flow, 34) ^ 
             get_bit_from_flow(flow, 32) ^ get_bit_from_flow(flow, 31) ^ get_bit_from_flow(flow, 30) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 26) ^ 
             get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 16) ^ get_bit_from_flow(flow, 12) ^ get_bit_from_flow(flow, 10) ^ get_bit_from_flow(flow, 9) ^ 
             get_bit_from_flow(flow, 6) ^ get_bit_from_flow(flow, 0) ^ ((seed & BIT0) >> 0) ^ ((seed & BIT1) >> 1) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT9) >> 9) ^ 
             ((seed & BIT10) >> 10) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT13) >> 13) ^ ((seed & BIT15) >> 15) ^ ((seed & BIT22) >> 22) ^ 
             ((seed & BIT23) >> 23) ^ ((seed & BIT24) >> 24) ^ ((seed & BIT25) >> 25) ^ ((seed & BIT26) >> 26) ^ ((seed & BIT27) >> 27) ^ ((seed & BIT29) >> 29) ^ 
             ((seed & BIT31) >> 31)); 

    hash |=  (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 94) ^ 
             get_bit_from_flow(flow, 88) ^ get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 86) ^ get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 79) ^ 
             get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 72) ^ get_bit_from_flow(flow, 69) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 64) ^ get_bit_from_flow(flow, 63) ^ 
             get_bit_from_flow(flow, 62) ^ get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 53) ^ 
             get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 46) ^ get_bit_from_flow(flow, 44) ^ 
             get_bit_from_flow(flow, 38) ^ get_bit_from_flow(flow, 37) ^ get_bit_from_flow(flow, 35) ^ get_bit_from_flow(flow, 34) ^ get_bit_from_flow(flow, 33) ^ get_bit_from_flow(flow, 28) ^ 
             get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 17) ^ get_bit_from_flow(flow, 16) ^ get_bit_from_flow(flow, 13) ^ get_bit_from_flow(flow, 12) ^ 
             get_bit_from_flow(flow, 11) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 7) ^ get_bit_from_flow(flow, 6) ^ get_bit_from_flow(flow, 1) ^ get_bit_from_flow(flow, 0) ^ 
             ((seed & BIT0) >> 0) ^ ((seed & BIT2) >> 2) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT9) >> 9) ^ ((seed & BIT14) >> 14) ^ 
             ((seed & BIT15) >> 15) ^ ((seed & BIT16) >> 16) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT28) >> 28) ^ ((seed & BIT29) >> 29) ^ ((seed & BIT30) >> 30) ^ 
             ((seed & BIT31) >> 31)) << 1;

    hash |=  (get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 97) ^ get_bit_from_flow(flow, 96) ^ 
             get_bit_from_flow(flow, 94) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 88) ^ get_bit_from_flow(flow, 85) ^ get_bit_from_flow(flow, 84) ^ get_bit_from_flow(flow, 83) ^ 
             get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 75) ^ get_bit_from_flow(flow, 72) ^ get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 68) ^ 
             get_bit_from_flow(flow, 67) ^ get_bit_from_flow(flow, 64) ^ get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 55) ^ 
             get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 39) ^ get_bit_from_flow(flow, 38) ^ 
             get_bit_from_flow(flow, 37) ^ get_bit_from_flow(flow, 36) ^ get_bit_from_flow(flow, 35) ^ get_bit_from_flow(flow, 32) ^ get_bit_from_flow(flow, 31) ^ get_bit_from_flow(flow, 30) ^ 
             get_bit_from_flow(flow, 26) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 18) ^ get_bit_from_flow(flow, 17) ^ get_bit_from_flow(flow, 16) ^ get_bit_from_flow(flow, 14) ^ 
             get_bit_from_flow(flow, 13) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 7) ^ get_bit_from_flow(flow, 6) ^ get_bit_from_flow(flow, 2) ^ 
             get_bit_from_flow(flow, 1) ^ get_bit_from_flow(flow, 0) ^ ((seed & BIT0) >> 0) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT8) >> 8) ^ 
             ((seed & BIT11) >> 11) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT13) >> 13) ^ ((seed & BIT16) >> 16) ^ ((seed & BIT17) >> 17) ^ ((seed & BIT22) >> 22) ^ 
             ((seed & BIT24) >> 24) ^ ((seed & BIT25) >> 25) ^ ((seed & BIT26) >> 26) ^ ((seed & BIT27) >> 27) ^ ((seed & BIT30) >> 30)) << 2;
    
    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 97) ^ 
            get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 86) ^ get_bit_from_flow(flow, 85) ^ get_bit_from_flow(flow, 84) ^ 
            get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 73) ^ get_bit_from_flow(flow, 71) ^ get_bit_from_flow(flow, 69) ^ 
            get_bit_from_flow(flow, 68) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 56) ^ 
            get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 40) ^ get_bit_from_flow(flow, 39) ^ 
            get_bit_from_flow(flow, 38) ^ get_bit_from_flow(flow, 37) ^ get_bit_from_flow(flow, 36) ^ get_bit_from_flow(flow, 33) ^ get_bit_from_flow(flow, 32) ^ get_bit_from_flow(flow, 31) ^ 
            get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 19) ^ get_bit_from_flow(flow, 18) ^ get_bit_from_flow(flow, 17) ^ get_bit_from_flow(flow, 15) ^ 
            get_bit_from_flow(flow, 14) ^ get_bit_from_flow(flow, 10) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 7) ^ get_bit_from_flow(flow, 3) ^ 
            get_bit_from_flow(flow, 2) ^ get_bit_from_flow(flow, 1) ^ ((seed & BIT1) >> 1) ^ ((seed & BIT4) >> 4) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT9) >> 9) ^ 
            ((seed & BIT12) >> 12) ^ ((seed & BIT13) >> 13) ^ ((seed & BIT14) >> 14) ^ ((seed & BIT17) >> 17) ^ ((seed & BIT18) >> 18) ^ ((seed & BIT23) >> 23) ^ 
            ((seed & BIT25) >> 25) ^ ((seed & BIT26) >> 26) ^ ((seed & BIT27) >> 27) ^ ((seed & BIT28) >> 28) ^ ((seed & BIT31) >> 31)) << 3;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 97) ^ get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 94) ^ 
                 get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 86) ^ get_bit_from_flow(flow, 84) ^ get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 79) ^ 
                 get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 73) ^ get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 69) ^ get_bit_from_flow(flow, 68) ^ 
                 get_bit_from_flow(flow, 67) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 63) ^ get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 57) ^ 
                 get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 48) ^ get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 46) ^ get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 44) ^ 
                 get_bit_from_flow(flow, 41) ^ get_bit_from_flow(flow, 40) ^ get_bit_from_flow(flow, 39) ^ get_bit_from_flow(flow, 38) ^ get_bit_from_flow(flow, 33) ^ get_bit_from_flow(flow, 31) ^ 
                 get_bit_from_flow(flow, 30) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 20) ^ get_bit_from_flow(flow, 19) ^ 
                 get_bit_from_flow(flow, 18) ^ get_bit_from_flow(flow, 15) ^ get_bit_from_flow(flow, 12) ^ get_bit_from_flow(flow, 11) ^ get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 6) ^ 
                 get_bit_from_flow(flow, 4) ^ get_bit_from_flow(flow, 3) ^ get_bit_from_flow(flow, 2) ^ get_bit_from_flow(flow, 0) ^ ((seed & BIT1) >> 1) ^ ((seed & BIT2) >> 2) ^ 
                 ((seed & BIT5) >> 5) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT14) >> 14) ^ ((seed & BIT18) >> 18) ^ 
                 ((seed & BIT19) >> 19) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT23) >> 23) ^ ((seed & BIT25) >> 25) ^ ((seed & BIT28) >> 28) ^ ((seed & BIT31) >> 31)) << 4;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 97) ^ get_bit_from_flow(flow, 94) ^ get_bit_from_flow(flow, 92) ^ 
                 get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 79) ^ 
                 get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 75) ^ get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 73) ^ get_bit_from_flow(flow, 72) ^ get_bit_from_flow(flow, 71) ^ 
                 get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 69) ^ get_bit_from_flow(flow, 67) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 64) ^ get_bit_from_flow(flow, 63) ^ 
                 get_bit_from_flow(flow, 61) ^ get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 51) ^ 
                 get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 46) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 42) ^ get_bit_from_flow(flow, 41) ^ 
                 get_bit_from_flow(flow, 40) ^ get_bit_from_flow(flow, 39) ^ get_bit_from_flow(flow, 37) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 24) ^ 
                 get_bit_from_flow(flow, 21) ^ get_bit_from_flow(flow, 20) ^ get_bit_from_flow(flow, 19) ^ get_bit_from_flow(flow, 13) ^ get_bit_from_flow(flow, 10) ^ get_bit_from_flow(flow, 7) ^ 
                 get_bit_from_flow(flow, 6) ^ get_bit_from_flow(flow, 5) ^ get_bit_from_flow(flow, 4) ^ get_bit_from_flow(flow, 3) ^ get_bit_from_flow(flow, 1) ^ get_bit_from_flow(flow, 0) ^ 
                 ((seed & BIT0) >> 0) ^ ((seed & BIT1) >> 1) ^ ((seed & BIT2) >> 2) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT6) >> 6) ^ ((seed & BIT7) >> 7) ^ 
                 ((seed & BIT8) >> 8) ^ ((seed & BIT9) >> 9) ^ ((seed & BIT10) >> 10) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT19) >> 19) ^ ((seed & BIT20) >> 20) ^ 
                 ((seed & BIT22) >> 22) ^ ((seed & BIT25) >> 25) ^ ((seed & BIT27) >> 27) ^ ((seed & BIT31) >> 31)) << 5;

    hash |= (get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 92) ^ 
                 get_bit_from_flow(flow, 84) ^ get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 79) ^ 
                 get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 75) ^ get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 73) ^ get_bit_from_flow(flow, 72) ^ get_bit_from_flow(flow, 71) ^ 
                 get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 68) ^ get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 64) ^ get_bit_from_flow(flow, 62) ^ 
                 get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 51) ^ 
                 get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 43) ^ get_bit_from_flow(flow, 42) ^ get_bit_from_flow(flow, 41) ^ 
                 get_bit_from_flow(flow, 40) ^ get_bit_from_flow(flow, 38) ^ get_bit_from_flow(flow, 30) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 22) ^ 
                 get_bit_from_flow(flow, 21) ^ get_bit_from_flow(flow, 20) ^ get_bit_from_flow(flow, 14) ^ get_bit_from_flow(flow, 11) ^ get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 7) ^ 
                 get_bit_from_flow(flow, 6) ^ get_bit_from_flow(flow, 5) ^ get_bit_from_flow(flow, 4) ^ get_bit_from_flow(flow, 2) ^ get_bit_from_flow(flow, 1) ^ ((seed & BIT0) >> 0) ^ 
                 ((seed & BIT1) >> 1) ^ ((seed & BIT2) >> 2) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT4) >> 4) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT8) >> 8) ^ 
                 ((seed & BIT9) >> 9) ^ ((seed & BIT10) >> 10) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT20) >> 20) ^ ((seed & BIT21) >> 21) ^ 
                 ((seed & BIT23) >> 23) ^ ((seed & BIT26) >> 26) ^ ((seed & BIT28) >> 28)) << 6;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 97) ^ get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 93) ^ 
                 get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 75) ^ 
                 get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 71) ^ get_bit_from_flow(flow, 69) ^ get_bit_from_flow(flow, 68) ^ get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 58) ^ 
                 get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 50) ^ 
                 get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 46) ^ get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 43) ^ get_bit_from_flow(flow, 42) ^ get_bit_from_flow(flow, 41) ^ 
                 get_bit_from_flow(flow, 39) ^ get_bit_from_flow(flow, 37) ^ get_bit_from_flow(flow, 34) ^ get_bit_from_flow(flow, 32) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 28) ^ 
                 get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 23) ^ get_bit_from_flow(flow, 22) ^ get_bit_from_flow(flow, 21) ^ get_bit_from_flow(flow, 16) ^ 
                 get_bit_from_flow(flow, 15) ^ get_bit_from_flow(flow, 10) ^ get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 7) ^ get_bit_from_flow(flow, 5) ^ get_bit_from_flow(flow, 3) ^ 
                 get_bit_from_flow(flow, 2) ^ get_bit_from_flow(flow, 0) ^ ((seed & BIT2) >> 2) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT4) >> 4) ^ ((seed & BIT5) >> 5) ^ 
                 ((seed & BIT7) >> 7) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT15) >> 15) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT23) >> 23) ^ ((seed & BIT25) >> 25) ^ 
                 ((seed & BIT26) >> 26) ^ ((seed & BIT31) >> 31)) << 7;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 97) ^ get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 88) ^ 
                 get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 85) ^ get_bit_from_flow(flow, 84) ^ get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 80) ^ 
                 get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 75) ^ get_bit_from_flow(flow, 73) ^ 
                 get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 69) ^ get_bit_from_flow(flow, 68) ^ get_bit_from_flow(flow, 67) ^ get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 65) ^ 
                 get_bit_from_flow(flow, 63) ^ get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 52) ^ 
                 get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 46) ^ get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 43) ^ get_bit_from_flow(flow, 42) ^ 
                 get_bit_from_flow(flow, 40) ^ get_bit_from_flow(flow, 38) ^ get_bit_from_flow(flow, 37) ^ get_bit_from_flow(flow, 35) ^ get_bit_from_flow(flow, 34) ^ get_bit_from_flow(flow, 33) ^ 
                 get_bit_from_flow(flow, 32) ^ get_bit_from_flow(flow, 31) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 23) ^ get_bit_from_flow(flow, 22) ^ get_bit_from_flow(flow, 17) ^ 
                 get_bit_from_flow(flow, 12) ^ get_bit_from_flow(flow, 11) ^ get_bit_from_flow(flow, 10) ^ get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 4) ^ get_bit_from_flow(flow, 3) ^ 
                 get_bit_from_flow(flow, 1) ^ get_bit_from_flow(flow, 0) ^ ((seed & BIT1) >> 1) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT4) >> 4) ^ ((seed & BIT5) >> 5) ^ 
                 ((seed & BIT6) >> 6) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT10) >> 10) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT12) >> 12) ^ 
                 ((seed & BIT13) >> 13) ^ ((seed & BIT15) >> 15) ^ ((seed & BIT16) >> 16) ^ ((seed & BIT23) >> 23) ^ ((seed & BIT25) >> 25) ^ ((seed & BIT29) >> 29) ^ 
                 ((seed & BIT31) >> 31)) << 8;

    hash |= (get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 96) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 88) ^ 
                 get_bit_from_flow(flow, 86) ^ get_bit_from_flow(flow, 85) ^ get_bit_from_flow(flow, 84) ^ get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 80) ^ 
                 get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 71) ^ 
                 get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 69) ^ get_bit_from_flow(flow, 68) ^ get_bit_from_flow(flow, 67) ^ get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 64) ^ 
                 get_bit_from_flow(flow, 61) ^ get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 52) ^ 
                 get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 46) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 43) ^ get_bit_from_flow(flow, 41) ^ 
                 get_bit_from_flow(flow, 39) ^ get_bit_from_flow(flow, 38) ^ get_bit_from_flow(flow, 36) ^ get_bit_from_flow(flow, 35) ^ get_bit_from_flow(flow, 34) ^ get_bit_from_flow(flow, 33) ^ 
                 get_bit_from_flow(flow, 32) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 23) ^ get_bit_from_flow(flow, 18) ^ get_bit_from_flow(flow, 13) ^ 
                 get_bit_from_flow(flow, 12) ^ get_bit_from_flow(flow, 11) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 5) ^ get_bit_from_flow(flow, 4) ^ get_bit_from_flow(flow, 2) ^ 
                 get_bit_from_flow(flow, 1) ^ ((seed & BIT2) >> 2) ^ ((seed & BIT4) >> 4) ^ ((seed & BIT5) >> 5) ^ ((seed & BIT6) >> 6) ^ ((seed & BIT7) >> 7) ^ 
                 ((seed & BIT8) >> 8) ^ ((seed & BIT9) >> 9) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT13) >> 13) ^ ((seed & BIT14) >> 14) ^ 
                 ((seed & BIT16) >> 16) ^ ((seed & BIT17) >> 17) ^ ((seed & BIT24) >> 24) ^ ((seed & BIT26) >> 26) ^ ((seed & BIT30) >> 30)) << 9;

    hash |= (get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 96) ^ get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 94) ^ 
                  get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 86) ^ get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 78) ^ 
                  get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 75) ^ get_bit_from_flow(flow, 73) ^ get_bit_from_flow(flow, 71) ^ get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 69) ^ 
                  get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 63) ^ get_bit_from_flow(flow, 62) ^ get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 58) ^ 
                  get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 42) ^ get_bit_from_flow(flow, 40) ^ 
                  get_bit_from_flow(flow, 39) ^ get_bit_from_flow(flow, 36) ^ get_bit_from_flow(flow, 35) ^ get_bit_from_flow(flow, 33) ^ get_bit_from_flow(flow, 32) ^ get_bit_from_flow(flow, 31) ^ 
                  get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 26) ^ get_bit_from_flow(flow, 19) ^ get_bit_from_flow(flow, 16) ^ get_bit_from_flow(flow, 14) ^ 
                  get_bit_from_flow(flow, 13) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 5) ^ get_bit_from_flow(flow, 3) ^ get_bit_from_flow(flow, 2) ^ get_bit_from_flow(flow, 0) ^ 
                  ((seed & BIT1) >> 1) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT5) >> 5) ^ ((seed & BIT6) >> 6) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT11) >> 11) ^ 
                  ((seed & BIT14) >> 14) ^ ((seed & BIT17) >> 17) ^ ((seed & BIT18) >> 18) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT23) >> 23) ^ ((seed & BIT24) >> 24) ^ 
                  ((seed & BIT26) >> 26) ^ ((seed & BIT29) >> 29)) << 10;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 94) ^ 
                  get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 85) ^ get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 78) ^ 
                  get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 73) ^ get_bit_from_flow(flow, 71) ^ get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 68) ^ 
                  get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 64) ^ get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 57) ^ 
                  get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 48) ^ 
                  get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 43) ^ get_bit_from_flow(flow, 41) ^ get_bit_from_flow(flow, 40) ^ 
                  get_bit_from_flow(flow, 36) ^ get_bit_from_flow(flow, 33) ^ get_bit_from_flow(flow, 31) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 26) ^ 
                  get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 20) ^ get_bit_from_flow(flow, 17) ^ get_bit_from_flow(flow, 16) ^ get_bit_from_flow(flow, 15) ^ 
                  get_bit_from_flow(flow, 14) ^ get_bit_from_flow(flow, 12) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 4) ^ get_bit_from_flow(flow, 3) ^ get_bit_from_flow(flow, 1) ^ 
                  get_bit_from_flow(flow, 0) ^ ((seed & BIT1) >> 1) ^ ((seed & BIT2) >> 2) ^ ((seed & BIT4) >> 4) ^ ((seed & BIT6) >> 6) ^ ((seed & BIT10) >> 10) ^ 
                  ((seed & BIT11) >> 11) ^ ((seed & BIT13) >> 13) ^ ((seed & BIT18) >> 18) ^ ((seed & BIT19) >> 19) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT26) >> 26) ^ 
                  ((seed & BIT29) >> 29) ^ ((seed & BIT30) >> 30) ^ ((seed & BIT31) >> 31)) << 11;

    hash |= (get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 97) ^ get_bit_from_flow(flow, 96) ^ 
                  get_bit_from_flow(flow, 94) ^ get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 86) ^ get_bit_from_flow(flow, 85) ^ 
                  get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 75) ^ get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 73) ^ 
                  get_bit_from_flow(flow, 71) ^ get_bit_from_flow(flow, 69) ^ get_bit_from_flow(flow, 68) ^ get_bit_from_flow(flow, 63) ^ get_bit_from_flow(flow, 61) ^ get_bit_from_flow(flow, 59) ^ 
                  get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 51) ^ 
                  get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 46) ^ get_bit_from_flow(flow, 42) ^ get_bit_from_flow(flow, 41) ^ 
                  get_bit_from_flow(flow, 31) ^ get_bit_from_flow(flow, 30) ^ get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 21) ^ get_bit_from_flow(flow, 18) ^ 
                  get_bit_from_flow(flow, 17) ^ get_bit_from_flow(flow, 15) ^ get_bit_from_flow(flow, 13) ^ get_bit_from_flow(flow, 12) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 6) ^ 
                  get_bit_from_flow(flow, 5) ^ get_bit_from_flow(flow, 4) ^ get_bit_from_flow(flow, 2) ^ get_bit_from_flow(flow, 1) ^ get_bit_from_flow(flow, 0) ^ ((seed & BIT1) >> 1) ^ 
                  ((seed & BIT2) >> 2) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT5) >> 5) ^ ((seed & BIT9) >> 9) ^ ((seed & BIT10) >> 10) ^ ((seed & BIT13) >> 13) ^ 
                  ((seed & BIT14) >> 14) ^ ((seed & BIT15) >> 15) ^ ((seed & BIT19) >> 19) ^ ((seed & BIT20) >> 20) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT24) >> 24) ^ 
                  ((seed & BIT25) >> 25) ^ ((seed & BIT26) >> 26) ^ ((seed & BIT29) >> 29) ^ ((seed & BIT30) >> 30)) << 12;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 97) ^ 
                  get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 88) ^ get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 86) ^ 
                  get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 75) ^ get_bit_from_flow(flow, 74) ^ 
                  get_bit_from_flow(flow, 72) ^ get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 69) ^ get_bit_from_flow(flow, 64) ^ get_bit_from_flow(flow, 62) ^ get_bit_from_flow(flow, 60) ^ 
                  get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 52) ^ 
                  get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 48) ^ get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 43) ^ get_bit_from_flow(flow, 42) ^ 
                  get_bit_from_flow(flow, 32) ^ get_bit_from_flow(flow, 31) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 22) ^ get_bit_from_flow(flow, 19) ^ 
                  get_bit_from_flow(flow, 18) ^ get_bit_from_flow(flow, 16) ^ get_bit_from_flow(flow, 14) ^ get_bit_from_flow(flow, 13) ^ get_bit_from_flow(flow, 10) ^ get_bit_from_flow(flow, 7) ^ 
                  get_bit_from_flow(flow, 6) ^ get_bit_from_flow(flow, 5) ^ get_bit_from_flow(flow, 3) ^ get_bit_from_flow(flow, 2) ^ get_bit_from_flow(flow, 1) ^ ((seed & BIT0) >> 0) ^ 
                  ((seed & BIT2) >> 2) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT4) >> 4) ^ ((seed & BIT6) >> 6) ^ ((seed & BIT10) >> 10) ^ ((seed & BIT11) >> 11) ^ 
                  ((seed & BIT14) >> 14) ^ ((seed & BIT15) >> 15) ^ ((seed & BIT16) >> 16) ^ ((seed & BIT20) >> 20) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT23) >> 23) ^ 
                  ((seed & BIT25) >> 25) ^ ((seed & BIT26) >> 26) ^ ((seed & BIT27) >> 27) ^ ((seed & BIT30) >> 30) ^ ((seed & BIT31) >> 31)) << 13;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 96) ^ 
                  get_bit_from_flow(flow, 94) ^ get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 88) ^ get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 84) ^ 
                  get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 75) ^ get_bit_from_flow(flow, 73) ^ 
                  get_bit_from_flow(flow, 71) ^ get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 63) ^ get_bit_from_flow(flow, 61) ^ get_bit_from_flow(flow, 59) ^ 
                  get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 52) ^ 
                  get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 48) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 43) ^ get_bit_from_flow(flow, 33) ^ 
                  get_bit_from_flow(flow, 32) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 26) ^ get_bit_from_flow(flow, 23) ^ get_bit_from_flow(flow, 20) ^ get_bit_from_flow(flow, 19) ^ 
                  get_bit_from_flow(flow, 17) ^ get_bit_from_flow(flow, 15) ^ get_bit_from_flow(flow, 14) ^ get_bit_from_flow(flow, 11) ^ get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 7) ^ 
                  get_bit_from_flow(flow, 6) ^ get_bit_from_flow(flow, 4) ^ get_bit_from_flow(flow, 3) ^ get_bit_from_flow(flow, 2) ^ ((seed & BIT1) >> 1) ^ ((seed & BIT3) >> 3) ^ 
                  ((seed & BIT4) >> 4) ^ ((seed & BIT5) >> 5) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT15) >> 15) ^ 
                  ((seed & BIT16) >> 16) ^ ((seed & BIT17) >> 17) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT24) >> 24) ^ ((seed & BIT26) >> 26) ^ 
                  ((seed & BIT27) >> 27) ^ ((seed & BIT28) >> 28) ^ ((seed & BIT31) >> 31)) << 14;

    hash |= (get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 97) ^ get_bit_from_flow(flow, 95) ^ 
                  get_bit_from_flow(flow, 94) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 88) ^ get_bit_from_flow(flow, 85) ^ get_bit_from_flow(flow, 84) ^ 
                  get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 72) ^ 
                  get_bit_from_flow(flow, 71) ^ get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 64) ^ get_bit_from_flow(flow, 62) ^ get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 59) ^ 
                  get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 52) ^ 
                  get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 34) ^ get_bit_from_flow(flow, 33) ^ 
                  get_bit_from_flow(flow, 30) ^ get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 21) ^ get_bit_from_flow(flow, 20) ^ get_bit_from_flow(flow, 18) ^ 
                  get_bit_from_flow(flow, 16) ^ get_bit_from_flow(flow, 15) ^ get_bit_from_flow(flow, 12) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 7) ^ 
                  get_bit_from_flow(flow, 5) ^ get_bit_from_flow(flow, 4) ^ get_bit_from_flow(flow, 3) ^ ((seed & BIT0) >> 0) ^ ((seed & BIT2) >> 2) ^ ((seed & BIT4) >> 4) ^ 
                  ((seed & BIT5) >> 5) ^ ((seed & BIT6) >> 6) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT13) >> 13) ^ ((seed & BIT16) >> 16) ^ 
                  ((seed & BIT17) >> 17) ^ ((seed & BIT18) >> 18) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT23) >> 23) ^ ((seed & BIT25) >> 25) ^ ((seed & BIT27) >> 27) ^ 
                  ((seed & BIT28) >> 28) ^ ((seed & BIT29) >> 29)) << 15;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 97) ^ 
                  get_bit_from_flow(flow, 94) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 86) ^ 
                  get_bit_from_flow(flow, 84) ^ get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 75) ^ 
                  get_bit_from_flow(flow, 68) ^ get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 48) ^ 
                  get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 46) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 37) ^ get_bit_from_flow(flow, 35) ^ get_bit_from_flow(flow, 32) ^ 
                  get_bit_from_flow(flow, 30) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 26) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 22) ^ get_bit_from_flow(flow, 21) ^ 
                  get_bit_from_flow(flow, 19) ^ get_bit_from_flow(flow, 17) ^ get_bit_from_flow(flow, 13) ^ get_bit_from_flow(flow, 12) ^ get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 5) ^ 
                  get_bit_from_flow(flow, 4) ^ get_bit_from_flow(flow, 0) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT5) >> 5) ^ ((seed & BIT6) >> 6) ^ ((seed & BIT10) >> 10) ^ 
                  ((seed & BIT11) >> 11) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT14) >> 14) ^ ((seed & BIT15) >> 15) ^ ((seed & BIT17) >> 17) ^ ((seed & BIT18) >> 18) ^ 
                  ((seed & BIT19) >> 19) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT25) >> 25) ^ ((seed & BIT27) >> 27) ^ ((seed & BIT28) >> 28) ^ ((seed & BIT30) >> 30) ^ 
                  ((seed & BIT31) >> 31)) << 16;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 95) ^ 
                  get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 88) ^ get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 85) ^ 
                  get_bit_from_flow(flow, 84) ^ get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 69) ^ 
                  get_bit_from_flow(flow, 67) ^ get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 48) ^ 
                  get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 38) ^ get_bit_from_flow(flow, 36) ^ get_bit_from_flow(flow, 33) ^ get_bit_from_flow(flow, 31) ^ 
                  get_bit_from_flow(flow, 30) ^ get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 23) ^ get_bit_from_flow(flow, 22) ^ get_bit_from_flow(flow, 20) ^ 
                  get_bit_from_flow(flow, 18) ^ get_bit_from_flow(flow, 14) ^ get_bit_from_flow(flow, 13) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 6) ^ get_bit_from_flow(flow, 5) ^ 
                  get_bit_from_flow(flow, 1) ^ ((seed & BIT4) >> 4) ^ ((seed & BIT6) >> 6) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT12) >> 12) ^ 
                  ((seed & BIT13) >> 13) ^ ((seed & BIT15) >> 15) ^ ((seed & BIT16) >> 16) ^ ((seed & BIT18) >> 18) ^ ((seed & BIT19) >> 19) ^ ((seed & BIT20) >> 20) ^ 
                  ((seed & BIT23) >> 23) ^ ((seed & BIT26) >> 26) ^ ((seed & BIT28) >> 28) ^ ((seed & BIT29) >> 29) ^ ((seed & BIT31) >> 31)) << 17;

    hash |= (get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 96) ^ get_bit_from_flow(flow, 93) ^ 
                  get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 88) ^ get_bit_from_flow(flow, 86) ^ get_bit_from_flow(flow, 85) ^ 
                  get_bit_from_flow(flow, 84) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 68) ^ 
                  get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 48) ^ 
                  get_bit_from_flow(flow, 46) ^ get_bit_from_flow(flow, 39) ^ get_bit_from_flow(flow, 37) ^ get_bit_from_flow(flow, 34) ^ get_bit_from_flow(flow, 32) ^ get_bit_from_flow(flow, 31) ^ 
                  get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 26) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 23) ^ get_bit_from_flow(flow, 21) ^ get_bit_from_flow(flow, 19) ^ 
                  get_bit_from_flow(flow, 15) ^ get_bit_from_flow(flow, 14) ^ get_bit_from_flow(flow, 10) ^ get_bit_from_flow(flow, 7) ^ get_bit_from_flow(flow, 6) ^ get_bit_from_flow(flow, 2) ^ 
                  ((seed & BIT5) >> 5) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT13) >> 13) ^ ((seed & BIT14) >> 14) ^ 
                  ((seed & BIT16) >> 16) ^ ((seed & BIT17) >> 17) ^ ((seed & BIT19) >> 19) ^ ((seed & BIT20) >> 20) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT24) >> 24) ^ 
                  ((seed & BIT27) >> 27) ^ ((seed & BIT29) >> 29) ^ ((seed & BIT30) >> 30)) << 18;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 97) ^ get_bit_from_flow(flow, 94) ^ 
                  get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 86) ^ 
                  get_bit_from_flow(flow, 85) ^ get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 71) ^ get_bit_from_flow(flow, 69) ^ 
                  get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 49) ^ 
                  get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 40) ^ get_bit_from_flow(flow, 38) ^ get_bit_from_flow(flow, 35) ^ get_bit_from_flow(flow, 33) ^ get_bit_from_flow(flow, 32) ^ 
                  get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 22) ^ get_bit_from_flow(flow, 20) ^ 
                  get_bit_from_flow(flow, 16) ^ get_bit_from_flow(flow, 15) ^ get_bit_from_flow(flow, 11) ^ get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 7) ^ get_bit_from_flow(flow, 3) ^ 
                  ((seed & BIT6) >> 6) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT9) >> 9) ^ ((seed & BIT13) >> 13) ^ ((seed & BIT14) >> 14) ^ ((seed & BIT15) >> 15) ^ 
                  ((seed & BIT17) >> 17) ^ ((seed & BIT18) >> 18) ^ ((seed & BIT20) >> 20) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT25) >> 25) ^ 
                  ((seed & BIT28) >> 28) ^ ((seed & BIT30) >> 30) ^ ((seed & BIT31) >> 31)) << 19;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 94) ^ 
                  get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 88) ^ get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 86) ^ 
                  get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 72) ^ get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 61) ^ 
                  get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 48) ^ 
                  get_bit_from_flow(flow, 41) ^ get_bit_from_flow(flow, 39) ^ get_bit_from_flow(flow, 36) ^ get_bit_from_flow(flow, 34) ^ get_bit_from_flow(flow, 33) ^ get_bit_from_flow(flow, 30) ^ 
                  get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 26) ^ get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 23) ^ get_bit_from_flow(flow, 21) ^ get_bit_from_flow(flow, 17) ^ 
                  get_bit_from_flow(flow, 16) ^ get_bit_from_flow(flow, 12) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 4) ^ ((seed & BIT0) >> 0) ^ 
                  ((seed & BIT7) >> 7) ^ ((seed & BIT9) >> 9) ^ ((seed & BIT10) >> 10) ^ ((seed & BIT14) >> 14) ^ ((seed & BIT15) >> 15) ^ ((seed & BIT16) >> 16) ^ 
                  ((seed & BIT18) >> 18) ^ ((seed & BIT19) >> 19) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT23) >> 23) ^ ((seed & BIT26) >> 26) ^ 
                  ((seed & BIT29) >> 29) ^ ((seed & BIT31) >> 31)) << 20;

    hash |= (get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 96) ^ get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 94) ^ 
                  get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 88) ^ get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 83) ^ 
                  get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 73) ^ get_bit_from_flow(flow, 71) ^ get_bit_from_flow(flow, 62) ^ get_bit_from_flow(flow, 61) ^ 
                  get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 42) ^ 
                  get_bit_from_flow(flow, 40) ^ get_bit_from_flow(flow, 37) ^ get_bit_from_flow(flow, 35) ^ get_bit_from_flow(flow, 34) ^ get_bit_from_flow(flow, 31) ^ get_bit_from_flow(flow, 29) ^ 
                  get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 26) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 22) ^ get_bit_from_flow(flow, 18) ^ get_bit_from_flow(flow, 17) ^ 
                  get_bit_from_flow(flow, 13) ^ get_bit_from_flow(flow, 10) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 5) ^ ((seed & BIT1) >> 1) ^ ((seed & BIT8) >> 8) ^ 
                  ((seed & BIT10) >> 10) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT15) >> 15) ^ ((seed & BIT16) >> 16) ^ ((seed & BIT17) >> 17) ^ ((seed & BIT19) >> 19) ^ 
                  ((seed & BIT20) >> 20) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT23) >> 23) ^ ((seed & BIT24) >> 24) ^ ((seed & BIT27) >> 27) ^ ((seed & BIT30) >> 30)) << 21;

    hash |= (get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 94) ^ 
                  get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 88) ^ get_bit_from_flow(flow, 87) ^ 
                  get_bit_from_flow(flow, 85) ^ get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 73) ^ get_bit_from_flow(flow, 68) ^ 
                  get_bit_from_flow(flow, 67) ^ get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 62) ^ get_bit_from_flow(flow, 61) ^ get_bit_from_flow(flow, 60) ^ 
                  get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 48) ^ get_bit_from_flow(flow, 47) ^ 
                  get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 43) ^ get_bit_from_flow(flow, 41) ^ get_bit_from_flow(flow, 38) ^ get_bit_from_flow(flow, 37) ^ 
                  get_bit_from_flow(flow, 36) ^ get_bit_from_flow(flow, 35) ^ get_bit_from_flow(flow, 34) ^ get_bit_from_flow(flow, 31) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 27) ^ 
                  get_bit_from_flow(flow, 26) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 23) ^ get_bit_from_flow(flow, 19) ^ get_bit_from_flow(flow, 18) ^ get_bit_from_flow(flow, 16) ^ 
                  get_bit_from_flow(flow, 14) ^ get_bit_from_flow(flow, 12) ^ get_bit_from_flow(flow, 11) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 0) ^ ((seed & BIT1) >> 1) ^ 
                  ((seed & BIT2) >> 2) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT10) >> 10) ^ ((seed & BIT13) >> 13) ^ ((seed & BIT15) >> 15) ^ ((seed & BIT16) >> 16) ^ 
                  ((seed & BIT17) >> 17) ^ ((seed & BIT18) >> 18) ^ ((seed & BIT20) >> 20) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT26) >> 26) ^ 
                  ((seed & BIT27) >> 27) ^ ((seed & BIT28) >> 28) ^ ((seed & BIT29) >> 29)) << 22;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 97) ^ 
                  get_bit_from_flow(flow, 96) ^ get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 88) ^ 
                  get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 86) ^ get_bit_from_flow(flow, 85) ^ get_bit_from_flow(flow, 84) ^ get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 81) ^ 
                  get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 75) ^ get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 73) ^ get_bit_from_flow(flow, 72) ^ 
                  get_bit_from_flow(flow, 69) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 62) ^ get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 56) ^ 
                  get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 46) ^ 
                  get_bit_from_flow(flow, 42) ^ get_bit_from_flow(flow, 39) ^ get_bit_from_flow(flow, 38) ^ get_bit_from_flow(flow, 36) ^ get_bit_from_flow(flow, 35) ^ get_bit_from_flow(flow, 34) ^ 
                  get_bit_from_flow(flow, 31) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 26) ^ get_bit_from_flow(flow, 20) ^ get_bit_from_flow(flow, 19) ^ 
                  get_bit_from_flow(flow, 17) ^ get_bit_from_flow(flow, 16) ^ get_bit_from_flow(flow, 15) ^ get_bit_from_flow(flow, 13) ^ get_bit_from_flow(flow, 9) ^ get_bit_from_flow(flow, 6) ^ 
                  get_bit_from_flow(flow, 1) ^ get_bit_from_flow(flow, 0) ^ ((seed & BIT0) >> 0) ^ ((seed & BIT1) >> 1) ^ ((seed & BIT2) >> 2) ^ ((seed & BIT3) >> 3) ^ 
                  ((seed & BIT7) >> 7) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT9) >> 9) ^ ((seed & BIT10) >> 10) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT13) >> 13) ^ 
                  ((seed & BIT14) >> 14) ^ ((seed & BIT15) >> 15) ^ ((seed & BIT16) >> 16) ^ ((seed & BIT17) >> 17) ^ ((seed & BIT18) >> 18) ^ ((seed & BIT19) >> 19) ^ 
                  ((seed & BIT21) >> 21) ^ ((seed & BIT24) >> 24) ^ ((seed & BIT25) >> 25) ^ ((seed & BIT26) >> 26) ^ ((seed & BIT28) >> 28) ^ ((seed & BIT30) >> 30) ^ 
                  ((seed & BIT31) >> 31)) << 23;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 97) ^ 
                  get_bit_from_flow(flow, 94) ^ get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 88) ^ 
                  get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 86) ^ get_bit_from_flow(flow, 85) ^ get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 81) ^ 
                  get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 75) ^ get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 73) ^ get_bit_from_flow(flow, 70) ^ 
                  get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 63) ^ get_bit_from_flow(flow, 61) ^ get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 56) ^ 
                  get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 48) ^ get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 43) ^ 
                  get_bit_from_flow(flow, 40) ^ get_bit_from_flow(flow, 39) ^ get_bit_from_flow(flow, 37) ^ get_bit_from_flow(flow, 36) ^ get_bit_from_flow(flow, 35) ^ get_bit_from_flow(flow, 32) ^ 
                  get_bit_from_flow(flow, 30) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 21) ^ get_bit_from_flow(flow, 20) ^ get_bit_from_flow(flow, 18) ^ 
                  get_bit_from_flow(flow, 17) ^ get_bit_from_flow(flow, 16) ^ get_bit_from_flow(flow, 14) ^ get_bit_from_flow(flow, 10) ^ get_bit_from_flow(flow, 7) ^ get_bit_from_flow(flow, 2) ^ 
                  get_bit_from_flow(flow, 1) ^ ((seed & BIT1) >> 1) ^ ((seed & BIT2) >> 2) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT4) >> 4) ^ ((seed & BIT8) >> 8) ^ 
                  ((seed & BIT9) >> 9) ^ ((seed & BIT10) >> 10) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT13) >> 13) ^ ((seed & BIT14) >> 14) ^ ((seed & BIT15) >> 15) ^ 
                  ((seed & BIT16) >> 16) ^ ((seed & BIT17) >> 17) ^ ((seed & BIT18) >> 18) ^ ((seed & BIT19) >> 19) ^ ((seed & BIT20) >> 20) ^ ((seed & BIT22) >> 22) ^ 
                  ((seed & BIT25) >> 25) ^ ((seed & BIT26) >> 26) ^ ((seed & BIT27) >> 27) ^ ((seed & BIT29) >> 29) ^ ((seed & BIT31) >> 31)) << 24;

    hash |= (get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 95) ^ 
                  get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 88) ^ 
                  get_bit_from_flow(flow, 87) ^ get_bit_from_flow(flow, 86) ^ get_bit_from_flow(flow, 84) ^ get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 81) ^ 
                  get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 75) ^ get_bit_from_flow(flow, 74) ^ get_bit_from_flow(flow, 71) ^ get_bit_from_flow(flow, 67) ^ 
                  get_bit_from_flow(flow, 64) ^ get_bit_from_flow(flow, 62) ^ get_bit_from_flow(flow, 61) ^ get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 56) ^ 
                  get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 48) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 41) ^ 
                  get_bit_from_flow(flow, 40) ^ get_bit_from_flow(flow, 38) ^ get_bit_from_flow(flow, 37) ^ get_bit_from_flow(flow, 36) ^ get_bit_from_flow(flow, 33) ^ get_bit_from_flow(flow, 31) ^ 
                  get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 22) ^ get_bit_from_flow(flow, 21) ^ get_bit_from_flow(flow, 19) ^ get_bit_from_flow(flow, 18) ^ 
                  get_bit_from_flow(flow, 17) ^ get_bit_from_flow(flow, 15) ^ get_bit_from_flow(flow, 11) ^ get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 3) ^ get_bit_from_flow(flow, 2) ^ 
                  ((seed & BIT2) >> 2) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT4) >> 4) ^ ((seed & BIT5) >> 5) ^ ((seed & BIT9) >> 9) ^ ((seed & BIT10) >> 10) ^ 
                  ((seed & BIT11) >> 11) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT14) >> 14) ^ ((seed & BIT15) >> 15) ^ ((seed & BIT16) >> 16) ^ ((seed & BIT17) >> 17) ^ 
                  ((seed & BIT18) >> 18) ^ ((seed & BIT19) >> 19) ^ ((seed & BIT20) >> 20) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT23) >> 23) ^ ((seed & BIT26) >> 26) ^ 
                  ((seed & BIT27) >> 27) ^ ((seed & BIT28) >> 28) ^ ((seed & BIT30) >> 30)) << 25;

    hash |= (get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 97) ^ get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 93) ^ 
                  get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 88) ^ get_bit_from_flow(flow, 81) ^ 
                  get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 75) ^ get_bit_from_flow(flow, 73) ^ 
                  get_bit_from_flow(flow, 67) ^ get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 62) ^ get_bit_from_flow(flow, 61) ^ get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 59) ^ 
                  get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 48) ^ 
                  get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 42) ^ get_bit_from_flow(flow, 41) ^ get_bit_from_flow(flow, 39) ^ get_bit_from_flow(flow, 38) ^ 
                  get_bit_from_flow(flow, 31) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 26) ^ get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 23) ^ 
                  get_bit_from_flow(flow, 22) ^ get_bit_from_flow(flow, 20) ^ get_bit_from_flow(flow, 19) ^ get_bit_from_flow(flow, 18) ^ get_bit_from_flow(flow, 10) ^ get_bit_from_flow(flow, 6) ^ 
                  get_bit_from_flow(flow, 4) ^ get_bit_from_flow(flow, 3) ^ get_bit_from_flow(flow, 0) ^ ((seed & BIT1) >> 1) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT4) >> 4) ^ 
                  ((seed & BIT5) >> 5) ^ ((seed & BIT6) >> 6) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT9) >> 9) ^ ((seed & BIT16) >> 16) ^ ((seed & BIT17) >> 17) ^ 
                  ((seed & BIT18) >> 18) ^ ((seed & BIT19) >> 19) ^ ((seed & BIT20) >> 20) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT23) >> 23) ^ ((seed & BIT25) >> 25) ^ 
                  ((seed & BIT26) >> 26) ^ ((seed & BIT28) >> 28)) << 26;

    hash |= (get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 96) ^ get_bit_from_flow(flow, 94) ^ 
                  get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 89) ^ get_bit_from_flow(flow, 82) ^ 
                  get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 76) ^ get_bit_from_flow(flow, 74) ^ 
                  get_bit_from_flow(flow, 68) ^ get_bit_from_flow(flow, 67) ^ get_bit_from_flow(flow, 63) ^ get_bit_from_flow(flow, 62) ^ get_bit_from_flow(flow, 61) ^ get_bit_from_flow(flow, 60) ^ 
                  get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 49) ^ 
                  get_bit_from_flow(flow, 48) ^ get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 43) ^ get_bit_from_flow(flow, 42) ^ get_bit_from_flow(flow, 40) ^ get_bit_from_flow(flow, 39) ^ 
                  get_bit_from_flow(flow, 32) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 26) ^ get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 24) ^ 
                  get_bit_from_flow(flow, 23) ^ get_bit_from_flow(flow, 21) ^ get_bit_from_flow(flow, 20) ^ get_bit_from_flow(flow, 19) ^ get_bit_from_flow(flow, 11) ^ get_bit_from_flow(flow, 7) ^ 
                  get_bit_from_flow(flow, 5) ^ get_bit_from_flow(flow, 4) ^ get_bit_from_flow(flow, 1) ^ ((seed & BIT2) >> 2) ^ ((seed & BIT4) >> 4) ^ ((seed & BIT5) >> 5) ^ 
                  ((seed & BIT6) >> 6) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT10) >> 10) ^ ((seed & BIT17) >> 17) ^ ((seed & BIT18) >> 18) ^ 
                  ((seed & BIT19) >> 19) ^ ((seed & BIT20) >> 20) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT24) >> 24) ^ ((seed & BIT26) >> 26) ^ 
                  ((seed & BIT27) >> 27) ^ ((seed & BIT29) >> 29)) << 27;

    hash |= (get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 97) ^ get_bit_from_flow(flow, 95) ^ 
                  get_bit_from_flow(flow, 94) ^ get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 90) ^ get_bit_from_flow(flow, 83) ^ 
                  get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 75) ^ 
                  get_bit_from_flow(flow, 69) ^ get_bit_from_flow(flow, 68) ^ get_bit_from_flow(flow, 64) ^ get_bit_from_flow(flow, 63) ^ get_bit_from_flow(flow, 62) ^ get_bit_from_flow(flow, 61) ^ 
                  get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 51) ^ get_bit_from_flow(flow, 50) ^ 
                  get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 46) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 43) ^ get_bit_from_flow(flow, 41) ^ get_bit_from_flow(flow, 40) ^ 
                  get_bit_from_flow(flow, 33) ^ get_bit_from_flow(flow, 30) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 26) ^ get_bit_from_flow(flow, 25) ^ 
                  get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 22) ^ get_bit_from_flow(flow, 21) ^ get_bit_from_flow(flow, 20) ^ get_bit_from_flow(flow, 12) ^ get_bit_from_flow(flow, 8) ^ 
                  get_bit_from_flow(flow, 6) ^ get_bit_from_flow(flow, 5) ^ get_bit_from_flow(flow, 2) ^ ((seed & BIT3) >> 3) ^ ((seed & BIT5) >> 5) ^ ((seed & BIT6) >> 6) ^ 
                  ((seed & BIT7) >> 7) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT9) >> 9) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT18) >> 18) ^ ((seed & BIT19) >> 19) ^ 
                  ((seed & BIT20) >> 20) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT23) >> 23) ^ ((seed & BIT25) >> 25) ^ ((seed & BIT27) >> 27) ^ 
                  ((seed & BIT28) >> 28) ^ ((seed & BIT30) >> 30)) << 28;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 96) ^ 
                  get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 94) ^ get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 91) ^ get_bit_from_flow(flow, 84) ^ 
                  get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 76) ^ 
                  get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 69) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 64) ^ get_bit_from_flow(flow, 63) ^ get_bit_from_flow(flow, 62) ^ 
                  get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 55) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 51) ^ 
                  get_bit_from_flow(flow, 50) ^ get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 42) ^ get_bit_from_flow(flow, 41) ^ 
                  get_bit_from_flow(flow, 34) ^ get_bit_from_flow(flow, 31) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 26) ^ 
                  get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 23) ^ get_bit_from_flow(flow, 22) ^ get_bit_from_flow(flow, 21) ^ get_bit_from_flow(flow, 13) ^ get_bit_from_flow(flow, 9) ^ 
                  get_bit_from_flow(flow, 7) ^ get_bit_from_flow(flow, 6) ^ get_bit_from_flow(flow, 3) ^ ((seed & BIT4) >> 4) ^ ((seed & BIT6) >> 6) ^ ((seed & BIT7) >> 7) ^ 
                  ((seed & BIT8) >> 8) ^ ((seed & BIT9) >> 9) ^ ((seed & BIT10) >> 10) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT19) >> 19) ^ ((seed & BIT20) >> 20) ^ 
                  ((seed & BIT21) >> 21) ^ ((seed & BIT22) >> 22) ^ ((seed & BIT23) >> 23) ^ ((seed & BIT24) >> 24) ^ ((seed & BIT26) >> 26) ^ ((seed & BIT28) >> 28) ^ 
                  ((seed & BIT29) >> 29) ^ ((seed & BIT31) >> 31)) << 29;

    hash |= (get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 101) ^ get_bit_from_flow(flow, 99) ^ get_bit_from_flow(flow, 97) ^ get_bit_from_flow(flow, 96) ^ 
                  get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 94) ^ get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 92) ^ get_bit_from_flow(flow, 85) ^ get_bit_from_flow(flow, 83) ^ 
                  get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 79) ^ get_bit_from_flow(flow, 77) ^ get_bit_from_flow(flow, 71) ^ 
                  get_bit_from_flow(flow, 70) ^ get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 64) ^ get_bit_from_flow(flow, 63) ^ get_bit_from_flow(flow, 61) ^ 
                  get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 58) ^ get_bit_from_flow(flow, 56) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 52) ^ get_bit_from_flow(flow, 51) ^ 
                  get_bit_from_flow(flow, 48) ^ get_bit_from_flow(flow, 46) ^ get_bit_from_flow(flow, 45) ^ get_bit_from_flow(flow, 43) ^ get_bit_from_flow(flow, 42) ^ get_bit_from_flow(flow, 35) ^ 
                  get_bit_from_flow(flow, 32) ^ get_bit_from_flow(flow, 30) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 27) ^ get_bit_from_flow(flow, 26) ^ 
                  get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 23) ^ get_bit_from_flow(flow, 22) ^ get_bit_from_flow(flow, 14) ^ get_bit_from_flow(flow, 10) ^ get_bit_from_flow(flow, 8) ^ 
                  get_bit_from_flow(flow, 7) ^ get_bit_from_flow(flow, 4) ^ ((seed & BIT5) >> 5) ^ ((seed & BIT7) >> 7) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT9) >> 9) ^ 
                  ((seed & BIT10) >> 10) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT13) >> 13) ^ ((seed & BIT20) >> 20) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT22) >> 22) ^ 
                  ((seed & BIT23) >> 23) ^ ((seed & BIT24) >> 24) ^ ((seed & BIT25) >> 25) ^ ((seed & BIT27) >> 27) ^ ((seed & BIT29) >> 29) ^ ((seed & BIT30) >> 30)) << 30;

    hash |= (get_bit_from_flow(flow, 103) ^ get_bit_from_flow(flow, 102) ^ get_bit_from_flow(flow, 100) ^ get_bit_from_flow(flow, 98) ^ get_bit_from_flow(flow, 97) ^ 
                  get_bit_from_flow(flow, 96) ^ get_bit_from_flow(flow, 95) ^ get_bit_from_flow(flow, 94) ^ get_bit_from_flow(flow, 93) ^ get_bit_from_flow(flow, 86) ^ get_bit_from_flow(flow, 84) ^ 
                  get_bit_from_flow(flow, 83) ^ get_bit_from_flow(flow, 82) ^ get_bit_from_flow(flow, 81) ^ get_bit_from_flow(flow, 80) ^ get_bit_from_flow(flow, 78) ^ get_bit_from_flow(flow, 72) ^ 
                  get_bit_from_flow(flow, 71) ^ get_bit_from_flow(flow, 67) ^ get_bit_from_flow(flow, 66) ^ get_bit_from_flow(flow, 65) ^ get_bit_from_flow(flow, 64) ^ get_bit_from_flow(flow, 62) ^ 
                  get_bit_from_flow(flow, 60) ^ get_bit_from_flow(flow, 59) ^ get_bit_from_flow(flow, 57) ^ get_bit_from_flow(flow, 54) ^ get_bit_from_flow(flow, 53) ^ get_bit_from_flow(flow, 52) ^ 
                  get_bit_from_flow(flow, 49) ^ get_bit_from_flow(flow, 47) ^ get_bit_from_flow(flow, 46) ^ get_bit_from_flow(flow, 44) ^ get_bit_from_flow(flow, 43) ^ get_bit_from_flow(flow, 36) ^ 
                  get_bit_from_flow(flow, 33) ^ get_bit_from_flow(flow, 31) ^ get_bit_from_flow(flow, 30) ^ get_bit_from_flow(flow, 29) ^ get_bit_from_flow(flow, 28) ^ get_bit_from_flow(flow, 27) ^ 
                  get_bit_from_flow(flow, 25) ^ get_bit_from_flow(flow, 24) ^ get_bit_from_flow(flow, 23) ^ get_bit_from_flow(flow, 15) ^ get_bit_from_flow(flow, 11) ^ get_bit_from_flow(flow, 9) ^ 
                  get_bit_from_flow(flow, 8) ^ get_bit_from_flow(flow, 5) ^ ((seed & BIT0) >> 0) ^ ((seed & BIT6) >> 6) ^ ((seed & BIT8) >> 8) ^ ((seed & BIT9) >> 9) ^ 
                  ((seed & BIT10) >> 10) ^ ((seed & BIT11) >> 11) ^ ((seed & BIT12) >> 12) ^ ((seed & BIT14) >> 14) ^ ((seed & BIT21) >> 21) ^ ((seed & BIT22) >> 22) ^ 
                  ((seed & BIT23) >> 23) ^ ((seed & BIT24) >> 24) ^ ((seed & BIT25) >> 25) ^ ((seed & BIT26) >> 26) ^ ((seed & BIT28) >> 28) ^ ((seed & BIT30) >> 30) ^ 
                  ((seed & BIT31) >> 31)) << 31; 

    return hash;
}

static bool
find_module_name(const char* suspect)
{
    bool found = false;
    int i = 0;
    for (i = 0; i < MODULES; i++)
    {
        if (strcmp(suspect, module_names[i]) == 0)
        {
            found = true;
            break;
        }
    }
    return found;
}

static void
enable_disable_modules(dag_card_ref_t card, dag_component_t pc, const char* enable_options, bool enable)
{
    attr_uuid_t attr = 0;
    char* p = NULL;
    char* start = NULL;
    char suspect[BUFFER_SIZE];

    dagutil_verbose_level(1, "%s the following modules: %s\n", enable ? "Enabling":"Disabling", enable_options);
    /* Check for bad options */
    start = (char*)enable_options;
    p = strstr(start, " ");
    memset(suspect, 0, BUFFER_SIZE);
    do
    {
        if (p != NULL)
        {
            memcpy(suspect, start, p - start);
            suspect[p - start] = 0;
            p++;
            start = p;
            p = strstr(start, " ");
            if (!find_module_name(suspect))
            {
                dagutil_warning("Unrecognized option %s\n", suspect);
            }
        }
    } while (p != NULL);
    strcpy(suspect, start);
    if (!find_module_name(suspect))
    {
        dagutil_warning("Unrecognized option %s\n", suspect);
    }

    if (strstr(enable_options, "all") != NULL)
    {
        /* Enable the modules */
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnablePacketCaptureModules);
        dag_config_set_boolean_attribute(card, attr, enable);

        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableProbabilitySampler);
        dag_config_set_boolean_attribute(card, attr, enable);

        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableProbabilityHashTable);
        dag_config_set_boolean_attribute(card, attr, enable);

        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableHostHashTable);
        dag_config_set_boolean_attribute(card, attr, enable);

        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableHostFlowTable);
        dag_config_set_boolean_attribute(card, attr, enable);
    }
    
    if (strstr(enable_options, "global") != NULL)
    {
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnablePacketCaptureModules);
        dag_config_set_boolean_attribute(card, attr, enable );
    }

    if (strstr(enable_options, "prob_sampler") != NULL)
    {
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableProbabilitySampler);
        dag_config_set_boolean_attribute(card, attr, enable);
    }

    if (strstr(enable_options, "prob_flow") != NULL)
    {
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableProbabilityHashTable);
        dag_config_set_boolean_attribute(card, attr, enable);
    }

    if (strstr(enable_options, "host_hash") != NULL)
    {
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableHostHashTable);
        dag_config_set_boolean_attribute(card, attr, enable);
    }

    if (strstr(enable_options, "host_flow") != NULL)
    {
        attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableHostFlowTable);
        dag_config_set_boolean_attribute(card, attr, enable);
    }
}

static void
reset_module(dag_card_ref_t card, dag_component_t pc)
{
    attr_uuid_t attr = 0;
    uint32_t val = 0;
    float percentage = 0;

    /* disable all modules */
    enable_disable_modules(card, pc, "all", false);
    attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeResetHashTable);
    dag_config_set_boolean_attribute(card, attr, 1);
    attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeResetHashTable);
    dag_config_set_boolean_attribute(card, attr, 0);

    attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeResetProbabilityHashTable);
    dag_config_set_boolean_attribute(card, attr, 1);
    attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeResetProbabilityHashTable);
    dag_config_set_boolean_attribute(card, attr, 0);

    /* clear the hash table by writing do not capture to all possible hash values.*/
    dagutil_verbose_level(0, "Clearing hash table...this might take a while...\n");
    for (val = 0; val < 0x7ffff; val++)
    {
        attr = dag_component_get_attribute_uuid(pc, kUint32AttributeUnsetCaptureHash);
        dag_config_set_uint32_attribute(card, attr, val);
        percentage = (float)val/(float)0x7ffff * 100.0;
        if ((int)percentage % 5 == 0)
            dagutil_verbose_level(1,"\r%d%%", (int)percentage);
    }
    fprintf(stderr, "\nDone\n");
}

static void
reverse_flow(const uint8_t* in, uint8_t* out)
{
    int i = 0;
    uint8_t temp[FLOW_SIZE];
    for (i = 0; i < FLOW_SIZE; i++)
    {
        temp[i] = in[12-i];
    }
    memcpy(out, temp, FLOW_SIZE);
}

static void
order_flow(const uint8_t* in, uint8_t* out)
{
    uint8_t temp;
    
    memmove(out, in, FLOW_SIZE);
    if (out[5] > out[9])
        return;
    else if (out[5] == out[9] && out[6] > out[10])
        return;
    else if (out[5] == out[9] && out[6] == out[10] && out[7] > out[11])
        return;
    else if (out[5] == out[9] && out[6] == out[10] && out[7] == out[11] && out[8] > out[12])
        return;

    /* swap the ip addresses */
    temp = out[5];
    out[5] = out[9];
    out[9] = temp;

    temp = out[6];
    out[6] = out[10];
    out[10] = temp;

    temp = out[7];
    out[7] = out[11];
    out[11] = temp;

    temp = out[8];
    out[8] = out[12];
    out[12] = temp;

    /* swap the ports */
    temp = out[0];
    out[0] = out[2];
    out[2] = temp;

    temp = out[1];
    out[1] = out[3];
    out[3] = temp;
}

static void
build_hash_table(dag_card_ref_t card, dag_component_t pc, const char* filename)
{
    FILE* f = fopen(filename, "r");
    char line[BUFFER_SIZE];
    uint8_t flow[FLOW_SIZE];
    char* p;
    char* start;
    char needle;
    uint16_t port;
    uint32_t crc32;
    attr_uuid_t attr = 0;


    attr = dag_component_get_attribute_uuid(pc, kBooleanAttributeEnableHostHashTable);
    if (dag_config_get_boolean_attribute(card, attr) == 1)
        dagutil_panic("Disable the module before loading the hash table\n");
    if (!f)
        dagutil_panic("Could not open file %s\n", filename);
    dagutil_verbose_level(1, "building hash table\n");
    memset(flow, 0, FLOW_SIZE);
    //make_ethernet_table();
    while (fgets(line, BUFFER_SIZE, f) != NULL)
    {
        /* ignore certain lines */
        dagutil_verbose_level(2, "Line: %s", line);
        if (line[0] == '#' || line[0] == '\n')
            continue;

        needle = '.';
        start = line;
        p = strchr(start, (int)needle);
        flow[5] = strtol(start, &p, 10);

        start = p;
        start++;
        p = strchr(start, (int)needle);
        flow[6] = strtol(start, &p, 10);

        start = p;
        start++;
        p = strchr(start, (int)needle);
        flow[7] = strtol(start, &p, 10);

        start = p;
        start++;
        needle = ',';
        p = strchr(start, (int)needle);
        flow[8] = strtol(start, &p, 10);

        start = p;
        start++;
        needle = '.';
        p = strchr(start, (int)needle);
        flow[9] = strtol(start, &p, 10);

        start = p;
        start++;
        p = strchr(start, (int)needle);
        flow[10] = strtol(start, &p, 10);

        start = p;
        start++;
        p = strchr(start, (int)needle);
        flow[11] = strtol(start, &p, 10);

        start = p;
        start++;
        needle = ',';
        p = strchr(start, (int)needle);
        flow[12] = strtol(start, &p, 10);

        /* ports */
        start = p;
        start++;
        p = strchr(start, (int)needle);
        port = strtol(start, &p, 10);
        flow[0] = (port >> 8);
        flow[1] = port & 0xffff;

        start = p;
        start++;
        p = strchr(start, (int)needle);
        port = strtol(start, &p, 10);
        flow[2] = (port >> 8);
        flow[3] = port & 0xffff;

        start = p;
        start++;
        needle = '\n';
        p = strchr(start, (int)needle);
        flow[4] = strtol(start, &p, 10);

        /* sort based on higher ip address */
        order_flow(flow, flow);
        /* this is just goddamn lucky - reversing the flow id allows the hash computation to mathc the hashes generated by the firmware */
        reverse_flow(flow, flow);
        dagutil_verbose_level(2, "Flow Id: 0x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x%02x\n",
                flow[0], flow[1], flow[2], flow[3], flow[4], flow[5], flow[6],
                flow[7], flow[8], flow[9], flow[10], flow[11], flow[12]);
        crc32 = make_hash(flow, 0xffffffff);
        dagutil_verbose_level(2, "Hash value = 0x%08x (%u)\n", crc32, crc32);
    
        attr = dag_component_get_attribute_uuid(pc, kUint32AttributeSetCaptureHash);
        dag_config_set_uint32_attribute(card, attr, crc32 & 0x7ffff);
    }
    fclose(f);
}


#ifndef ENDACE_UNIT_TEST
int
main(int argc, const char* const * argv)
{
	return dagrndcap_main(argc, (char**) argv);
}
#endif /* ENDACE_UNIT_TEST */




