/*
 * Copyright (c) 2002-2006 Endace Technology Ltd, Hamilton, New Zealand.
 * All rights reserved.
 *
 * This source code is proprietary to Endace Technology Limited and no part
 * of it may be redistributed, published or disclosed except as outlined in
 * the written contract supplied with this product.
 *
 * $Id: outputs.h 12512 2010-03-08 21:24:12Z jomi.gregory $
 *
 * Header file defining functions to be implemented by all output sources that
 * consume ERF records.
 */

#ifndef OUTPUTS_H
#define OUTPUTS_H


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


/*
 * Indicates no snap length limit is in effect 
 */
#define NO_SNAP_LIMIT -1

/*
 * Function called when the processor has an ERF
 * header and payload to output. Only the header
 * fields of the dag_record_t are valid. The header
 * is correct for the given payload. Returns the
 * number of records written.
 */
int write_record(dag_record_t * header, void *payload);


/*******************************************************
 * ERF file functions
 *******************************************************/
/*
 * Write given ERF header and payload in ERF format.
 * Returns 1 if record successfully written and 0
 * otherwise.
 */
int write_erf_record(dag_record_t * header, void *payload);

/*
 * Set the stream on which output is to be produced.
 */
void set_erf_output(char *out);

/*
* Converts the existing ERF type to the desired ERF type
*/
void
format_erf(dag_record_t* header, void *payload, uint8_t outerftype, void **newpayload);

/*
 * Close the currently open input (if any).
 */
void close_erf_output(void);

/*
 * Set a snap length to apply to packets. Packets
 * longer than this value are truncated to the specified
 * value. To allow arbitrary length packets, set
 * snap_length to -1. If variable_length is nonzero then packets
 * can be of variable length, otherwise all packets
 * will have the specified length.
 */
void set_erf_snap_length(int snap_length, int variable_length);
void set_erf_record_alignment(int record_alignment);

/*******************************************************
 * PCAP file functions
 *******************************************************/

/*
 * Number of dumpers for multichannel dumping (MC_ATM, MC_HDLC, etc.)
 * Currently it is equal to the maximum connection number of MC_AAL5.
 */
#define MC_DUMPERS 2048
#define TEMP_FILENAME_BUFSIZE 256

/* TODO: instead of hardcoded MC_DUMPERS add a count field to mc_dumper_t */

typedef struct {
    pcap_dumper_t * dumper;
    uint64_t written;
    unsigned rotated;
    char mc_file_name[TEMP_FILENAME_BUFSIZE];
} mc_dumper_t;

/* SunATM pseudo header */
struct sunatm_hdr {
	unsigned char	flags;		/* destination and traffic type */
	unsigned char	vpi;		/* VPI */
	unsigned short	vci;		/* VCI */
};

/* Protocol type values in the bottom for flag bits of SUNATM pseudoheader. */
#define PT_LANE		0x01	/* LANE */
#define PT_LLC		0x02	/* LLC encapsulation */

/*
 * Write given ERF header and payload in PCAP format.
 * Returns 1 if record successfully written and 0
 * otherwise.
 */
int write_pcap_record(dag_record_t * header, void *payload);

/*
 * Set the stream on which output is to be produced.
 */
void set_pcap_output(char *out);

/*
 * Get the output filename
 */
char* get_pcap_output(void);

/*
 * Prepare the output stream for writing pcap style records.
 */
pcap_dumper_t* prepare_pcap_output(dag_record_t * erf, void *payload, char *out);

/*
 * Close the output
 */
void close_pcap_output();

/*
 * Set a snap length to apply to packets. Packets
 * longer than this value are truncated to the specified
 * value.
 */
void set_pcap_snap_length(int snap_length);

/*
 * Manually set the timezone to use with pcap packets.
 * The value should be the number of seconds west of GMT.
 * If this function is not called, then the timezone is
 * guessed based on the result of tzset().
 */
void set_pcap_timezone(int tzone);

/*
 * Set the wirelength fudge value.
 * Defaults to 4.
 */
void set_pcap_wlen_fudge(int n);

/*
 * Initialize the dumper table (for multichannel dumps)
 */
void alloc_pcap_dumper_table();

/*
 * Close dumpers and free the descriptors
 */
void cleanup_pcap_dumper_table();

void close_pcap_dumper(pcap_dumper_t * dumper);

/*******************************************************
 * PRT file functions
 *******************************************************/
/*
 * Write an ASCII representation of a record. Returns 1
 * if record successfully written.
 */
int write_prt_record(dag_record_t * header, void *payload);

/*
 * Set the stream on which output is to be produced.
 */
void set_prt_output(char *out);

/*
 * Close the currently open input (if any).
 */
void close_prt_output(void);

/*******************************************************
 * Null output device functions
 *******************************************************/
/*
 * Discard ERF header and payload.
 */
int write_null_record(dag_record_t * header, void *payload);

/*
 * Stub function, included for completeness.
 */
void set_null_output(char *out);

/*
 * Stub function, included for completeness.
 */
void close_null_output(void);


#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* OUTPUTS_H */
