/*
 * Copyright (c) 2002-2006 Endace Technology Ltd, Hamilton, New Zealand.
 * All rights reserved.
 *
 * This source code is proprietary to Endace Technology Limited and no part
 * of it may be redistributed, published or disclosed except as outlined in
 * the written contract supplied with this product.
 *
 * $Id: inputs.h 12512 2010-03-08 21:24:12Z jomi.gregory $
 *
 * Header file defining functions to be implemented
 * by all input sources that produce ERF records.
 */

#ifndef INPUTS_H
#define INPUTS_H

/* libpcap headers. */
#include "pcap.h"

/* Endace headers. */
#include "dagapi.h"
#include "dagerf.h"


/* maximum number of input files allowed */
#define MAXCNT 64

/* maximum packet size in pcap output */
#define MAX_PACKET 65536

/* size of ATM cell header without HEC */
#define ATM_HDR_SIZE 4

extern uint64_t gRotateSize;
extern int gFileNumber;
extern int gFcsBits;
extern int gErfOutputIface;
extern int gOutputDataLinkType;
extern int gErfOutputType;
extern uint8_t gUseTempOutputFiles;

typedef union legacy_record
{
	int64_t      ts;
	legacy_atm_t cell;
	legacy_eth_t ether;
	legacy_pos_t pos;
	
} legacy_record_t;

typedef struct pcap_record
{
	struct pcap_pkthdr pkthdr;
	void * pkt;
	
} pcap_record_t;

extern pcap_record_t * gPcapRecord;

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */


/*
 * Get pointer to the ERF header for the next packet
 * in the input stream. Returns null if no further
 * packets are available.
 */
dag_record_t * get_next_header(void);

/*
 * Returns a pointer to the payload of the most recent
 * packet. Returns null if there is no current packet.
 */
void * get_payload(void);

/*
 * Set the name of the stream from which input is obtained.
 */
void set_input(char *name);

/*
 * Called when the input is no longer needed so that any 
 * special shutdown tasks can be performed.
 */
void close_input();

/*******************************************************
 * ERF file functions
 *******************************************************/
/*
 * Get pointer to the ERF header for the next packet
 * in the input stream. Returns null if no further
 * packets are available.
 */
dag_record_t * get_next_erf_header(void);

/*
 * Returns a pointer to the payload of the most recent
 * packet. Returns null if there is no current packet.
 */
void * get_erf_payload(void);

/*
 * Set the stream from which input is obtained.
 */
void set_erf_input(char *in);

/*
 * Close the current input stream.
 */
void close_erf_input(void);

/*
 * Return the channel(connection) number of a multichannel ERF record,
 * or -1 if the record is not multichannel.
 */
    int get_mc_channel(dag_record_t * header, void * payload);


/*******************************************************
 * Legacy file functions
 *******************************************************/
/*
 * Get pointer to the ERF header for the next packet
 * in the input stream. Returns null if no further
 * packets are available.
 */
dag_record_t * get_next_legacy_header(void);

/*
 * Returns a pointer to the payload of the most recent
 * packet. Returns null if there is no current packet.
 */
void * get_legacy_payload(void);

/*
 * Set the stream from which input is obtained.
 */
void set_legacy_input(char *in, int input_type);

/*
 * Close the current input stream.
 */
void close_legacy_input(void);

/*******************************************************
 * Direct form DAG card functions
 *******************************************************/
/*
 * Get pointer to the ERF header for the next packet
 * in the input stream from a DAG card. Returns null 
 * if no further packets are available.
 */
dag_record_t * get_next_dag_header(void);

/*
 * Returns a pointer to the payload of the most recent
 * packet. Returns null if there is no current packet.
 */
void * get_dag_payload(void);

/*
 * Set the stream from which input is obtained.  The given dagname is expected
 * to be a unix device (/dev/dag0 for example) and args is a string comprised
 * of space separated arguments that is set to the dag via dag_configure.
 */
void set_dag_input(const char *dagname,int dagstream, char *args);

/*
 * Called when the DAG input is no longer needed so that any 
 * special shutdown tasks can be performed.
 */
void close_dag_input(int verbose);

/*******************************************************
 * PCAP file functions
 *******************************************************/
/*
 * Get pointer to the ERF header for the next packet
 * in a PCAP file. Returns null if no further packets are available.
 */
dag_record_t * get_next_pcap_header(void);

/*
 * Returns a pointer to the payload of the most recent
 * packet. Returns null if there is no current packet.
 */
void * get_pcap_payload(void);

/*
 * Set the stream from which input is obtained.
 */
void set_pcap_input(char *filename);

/*
 * Called when the PCAP input is no longer needed so that any 
 * special shutdown tasks can be performed.
 */
void close_pcap_input(void);

/*
 * Utility function to create a pcap_record_t if one does not exist.
 * sets gPcapRecord global pointer.
 */
void create_pcap_record(dag_record_t * header, uint8_t *payload);

/*******************************************************
 * Null input device functions
 *******************************************************/
/*
 * Returns NULL.
 */
dag_record_t * get_next_null_header(void);

/*
 * Returns NULL.
 */
void * get_null_payload(void);

/*
 * Stub function, included for completeness.
 */
void set_null_input(char *in);

/*
 * Stub function, included for completeness.
 */
void close_null_input(void);

/* 
*   To set the capture api version. version 0 uses dag_offset
                                    version 1 uses dag_rx_stream_next_record()
*/
void set_capture_api_version(int version);
#ifdef __cplusplus
}
#endif /* __cplusplus */


#endif /* INPUTS_H */
