/*
 * Copyright (c) 2004-2005 Endace Technology Ltd, Hamilton, New Zealand.
 * All rights reserved.
 *
 * This source code is proprietary to Endace Technology Limited and no part
 * of it may be redistributed, published or disclosed except as outlined in
 * the written contract supplied with this product.
 *
 * $Id: utilities.h 6856 2007-05-20 23:55:04Z andras $
 */

#ifndef UTILITIES_H
#define UTILITIES_H

/* Project headers. */
#include "filter_rule_types.h"


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/**
 *\file
 * \brief Utility functions
 *
 * Utility functions for creating/dipslaying/disposing packet filtering related data types
 */

/**
 * Count ones in the binary
 *
 * \param value    the integer to count the bits in
 * \return         the number of 'one' bits
 */
unsigned int count_one_bits(uint32_t value);

/**
 * Cidr suffix to mask conversion
 *
 * \param suffix  width of the mask in bits
 * \return        binary with 'suffix' number of trailing ones
 */
uint32_t cidr_suffix_to_mask(unsigned int suffix);


/**
 * Create a list node with 'value' IP address and prepend to 'next'
 *
 * \param value  pointer IP address (ternary)
 * \param next   pointer to list item (linked list) to prepend to
 * \return       pointer to the new list node
 */
list_node_t* new_list_node(ip_address_t* value, list_node_t* next);

/**
 * Free resources allocated to 'node'
 *
 * \param node   pointer to node
 */
void dispose_list_node(list_node_t* node);

/**
 * Free resources allocated to 'rule', includeing the list of
 * source/destionation addresses.
 *
 * \param rule   pointer to filter rule
 */
void dispose_filter_rule(filter_rule_t* rule);

/**
 * Clone the list of addresses
 * 
 * \param list   pointer to the first node in the linked list
 * \return       pointer to the reversed list of addresses given in 'list'
 */
list_node_t* clone_address_list(list_node_t* list);

/**
 * Get list length
 *
 * \param list    pointer to the first item in list
 * \return        the number of items in 'list'
 */
unsigned int get_list_length(list_node_t* list);



/**
 * Create new port descriptor
 *
 * Fields are set to zero, except 'last', which is 65535
 *
 * \return pointer to the new port descriptor
 */
ipf_port_t* new_port(void);

/**
 * Free resources allocated to 'port'
 *
 * \param node   pointer to port descriptor
 */
void dispose_port(ipf_port_t* port);


/**
 * Get action string
 *
 * \return ASCII representation of the accept/reject action
 */
const char* get_action_string(uint8_t action);

/**
 * Get protocol string
 *
 * \return ASCII representation of the layer 3/4 protocol
 */
const char* get_protocol_string(protocol_t protocol);


/**
 * Print out the ASCII representation of 'rule'
 *
 * \param outfile   pointer to file descriptor to print to
 * \param rule      pointer to rule to print out
 */
void display_rule(FILE* outfile, filter_rule_t* rule);

/**
 * Print out the ASCII representation of the IP address.
 * The format is  'address/suffix (mask)'
 *
 * \param outfile   pointer to file descriptor to print to
 * \param address   pointer to the address
 */
void display_ip_address(FILE* outfile, ip_address_t* address);

/**
 * Print out the ASCII representation of the IP address list.
 * The format is  'address/suffix (mask)'
 *
 * \param outfile   pointer to file descriptor to print to
 * \param head      pointer to the linked-list of addresses
 */
void display_ip_address_list(FILE* outfile, list_node_t* head);

/**
 * Print out the fields of the port structure
 *
 * \param outfile   pointer to file descriptor to print to
 * \param port      pointer to the port descriptor
 */
void display_port(FILE* outfile, ipf_port_t* port);


#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* UTILITIES_H */
