/*
 * Copyright (c) 2004-2005 Endace Technology Ltd, Hamilton, New Zealand.
 * All rights reserved.
 *
 * This source code is proprietary to Endace Technology Limited and no part
 * of it may be redistributed, published or disclosed except as outlined in
 * the written contract supplied with this product.
 *
 * $Id: dag_filter.h 11725 2009-08-06 05:39:17Z wilson.zhu $
 */
/** \defgroup PFAPI Packet Filtering API
 */
/*@{*/
/** 
 * \brief High-level interface to the coprocessor
 *
 * Configuring the coprocessor and loading CAM entries. Note, that
 * these functions are dealing with preprocessed CAM entries, not
 * filter rules. For functions setting and changing parts of the CAM
 * entries or processing the rules given as ASCII strings see the
 * high-level interface.
 *
 * The following steps are necessary to load a set of filters:
 *
 * - initialise (configure) the coprocessor
 * - connect to the coprocessor
 *     - optional: query the number of interfaces (ports)
 *     - optional: query the number of rulesets allowed on the seleceted interface
 * - load CAM entries for the selected interface and ruleset
 *     - optional: set the PCI burst manager's drop behaviour
 * - activate the ruleset on the selected interface
 *     - optional: perform upper datapath reset [external function, see dagutil]
 *
 * TODO: dispose? see dagipf_dispose
 */
/** \defgroup PFCC Coprocessor control
 */
/*@{*/


#ifndef DAG_FILTER_H
#define DAG_FILTER_H

/* Endace headers. */
#include "dagutil.h"

/* Project headers. */
#include "dagipf.h"
#include "dagpf.h"

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

typedef void* FilterPrivatePtr; /**< Opaque pointer for filter private data. */

/** \defgroup PFCCF Coprocessor control functions
 */
/*@{*/

/**
 * Allocate and initialize private data buffer
 *
 * \return pointer to private data structure
 */
FilterPrivatePtr dag_filter_init();

/**
 * Dispose private data buffer
 *
 * \param private pointer to private data structure
 */
void dag_filter_dispose(FilterPrivatePtr private);

/**
 * Configure coprocessor 
 *
 * - Set header sizes depending on link type.
 * - Initialise results (output) mapping.
 * - Set the PCI Burst Manager route source select.
 * - Set up the ports and rulesets.
 * - Create template for 144 bit search <-- TODO: move it to separate function!
 * - load parameters to coprocessor and activate it (start processing)
 *
 * \param private    pointer private data
 * \param dagfd      file decriptor of the DAG card
 * \param datalink   data link layer protocol (Ethhernet, ATM, etc.)
 * \param mapping    color mapping method(location)
 * \param ports      number of ports
 * \param rulesets   number of rulesets
 * \param cptype     coprocessor type
 *
 * \return 1 if call succeeded, 0 otherwise.
 */
unsigned int dag_filter_configure_copro(FilterPrivatePtr private,
                                        int dagfd,
                                        linktype_t datalink,
                                        mapping_t mapping,
                                        uint8_t ports,
                                        uint8_t rulesets,
                                        coprocessor_t cptype);

/**
 * Connect to coprocesor
 *
 * - initialize ipf (connect to copro)
 * - set header sizes depending on link type
 * - save linktype for later
 *
 * \param private    pointer private data
 * \param dagfd     file descriptor to DAG card
 * \param datalink  data link layer type
 *
 * \return 1 if call succeeded, 0 otherwise.
 */
unsigned int dag_filter_connect_to_copro(FilterPrivatePtr private, int dagfd, linktype_t datalink, coprocessor_t cptype);

/** 
 * Load CAM entries into coprocessor
 *
 * The provided array of CAM entries has to be filled up by going
 * through the filter-list.
 *
 * \param private    pointer private data
 * \param port       which port to use
 * \param ruleset    which ruleset to use
 * \param cam_array  array of pointers to CAM entries
 * \param count      number of filters(CAM entries)
 *
 * \return 1 if call succeeded, 0 otherwise. 
 */
unsigned int dag_filter_load_cam_entries(FilterPrivatePtr private,
                                         uint8_t port,
                                         uint8_t ruleset,
                                         cam_entry_t** cam_array,
                                         uint32_t items,
                                         bool verify);

/** Get the number of ports configured. */
unsigned int dag_filter_get_port_count(FilterPrivatePtr private);

/** Get the number of rulesets configured for a particular port. */
unsigned int dag_filter_get_port_ruleset_count(FilterPrivatePtr private, uint8_t port);

/** Get the ruleset for a particular port. */
unsigned int dag_filter_get_port_ruleset(FilterPrivatePtr private, uint8_t port, uint8_t* ruleset);

/** Set the ruleset for a particular port. */
unsigned int dag_filter_set_port_ruleset(FilterPrivatePtr private, uint8_t port, uint8_t ruleset);

/** Enable dropping */
unsigned int dag_filter_set_drop_enable(FilterPrivatePtr private, int dagfd, int enabled);
/*@}*/

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* DAG_FILTER_H */
/*@}*/
/*@}*/

