#include "iwl-io.h"
#include "idi_internal.h"
#include "idi_tx.h"
#include "idi_tx_policy.h"
#include "idi_al.h"

#define IDI_CMD_QUOTA 2
#define IWL_IDI_TX_POLICY_WQ_FLAGS (WQ_HIGHPRI | WQ_UNBOUND | WQ_NON_REENTRANT)
#define IDI_CMD_TFD_QUOTA 1
#define IDI_CMD_TB_QUOTA 3

static inline
bool iwl_idi_tx_bus_agg_full(struct iwl_idi_trans_tx *trans_tx, u8 chan)
{
	return (trans_tx->txbus_added_policy[chan] >=
		trans_tx->sg_list_meta[chan].max_txbus);
}

static inline
bool iwl_idi_tx_bus_agg_empty(struct iwl_idi_trans_tx *trans_tx, u8 chan)
{
	return (trans_tx->txbus_added_policy[chan] == 0);
}

/**
 * iwl_idi_tx_policy_check_alloc - check according to available
 * and required resources that allocation request can be satisfied.
 * Since policy is only an algorithm and doesn't manages or holds any
 * resources, it should receive all the data required for the decision.
 * This function is usually called by transport.
 * @trans - the transport
 * @avail_tfds - TFDs available in transport
 * @avail_tbs - TBs available in transport
 * @req_tfds - how much TFDs are requested
 * @req_tbs - how much TBs are requested
 * Retuns true if the resources can be allocated, false otherwise.
 */
bool iwl_idi_tx_policy_check_alloc(struct iwl_trans_slv *trans_slv,
				   u8 txq_id,
				   int avail_tfds, int avail_tbs,
				   int req_tfds, int req_tbs)
{
	if (txq_id != trans_slv->cmd_queue) {
		avail_tfds -= IDI_CMD_TFD_QUOTA;
		avail_tbs -= IDI_CMD_TB_QUOTA;
	}

	if ((req_tfds <= avail_tfds) && (req_tbs <= avail_tbs))
		return true;

	return false;
}

/*
 * if there are no aggragations to channel, clears loaded bit. else
 *- calls for dma arm.
 */
static void
iwl_idi_tx_policy_dma_arm_chan(struct iwl_idi_trans_tx *trans_tx, u8 chan)
{
	int ret;
	struct iwl_trans *trans = IWL_TRANS_TX_GET_TRANS(trans_tx);

	if ((iwl_idi_tx_bus_agg_empty(trans_tx, chan))) {
		clear_bit(chan, &trans_tx->sg_list_loaded);
		return;

	} else {
		/* if reached this point, bus aggregation should be sent */
		iwl_idi_tx_close_sg_list(trans_tx, chan);
		ret = iwl_idi_tx_dma_arm(trans_tx, chan);
		if (unlikely(ret)) {
			IWL_ERR(trans,
				"[%d] %s: iwl_idi_tx_dma_arm failed, ret %d\n",
				get_current()->tgid, __func__, ret);
			return;
		}

		trans_tx->txbus_added_policy[chan] = 0;
		IWL_DEBUG_TX(trans, "%s, chan %d: TXBU size %d, ret %d",
			     __func__, chan, trans_tx->txbus_added_policy[chan],
			     ret);
	}
}

/**
 * iwl_idi_tx_policy_build_bus_agg_data - pulls tx data from data queues
 * using round robin, and adds to LP sg list.
 *
 * Note: uses transport's handlers which add data from queues to the bus
 * aggregation, here only scheduling is cared. All adding errors should be
 * handled inside transport. An error returned from transport could also mean
 * that the corresponding queue is empty.
 *
 * In this implementation it doesn't use the is_immediate return value from
 * the transport.
 *
 * @trans_slv - slave common transport
 * @return true if loading the list is permitted ('sg list loaded' bit is clear);
 */
static bool
iwl_idi_tx_policy_build_bus_agg_data(struct iwl_trans_slv *trans_slv)
{
	struct iwl_idi_trans_tx *trans_tx =
		IWL_TRANS_SLV_GET_IDI_TRANS_TX(trans_slv);

	bool is_updated;
	int i, ret;

	if ((test_and_set_bit(TX_SG_LP_CH, &trans_tx->sg_list_loaded)))
		return false;

	/* add txbus from all data queues on round robin basis */
	while (!iwl_idi_tx_bus_agg_full(trans_tx, TX_SG_LP_CH)) {
		is_updated = false;
		for (i = 0; i < IDI_TX_MAX_Q_COUNT; i++) {
			if (i == trans_slv->cmd_queue)
				continue;

			ret = iwl_idi_tx_add_burst(trans_slv, TX_SG_LP_CH, i);

			/* don't distinguish between regular and
			 * immediate tx
			 */
			if (ret >= 0) {
				is_updated = true;
				trans_tx->txbus_added_policy[TX_SG_LP_CH]++;
			}

			/* don't continue to the next queue if no resources */
			if (iwl_idi_tx_bus_agg_full(trans_tx, TX_SG_LP_CH))
				break;
		}
		if (!is_updated)
			break;
	}

	return true;
}

/**
 * iwl_idi_tx_policy_build_bus_agg_cmd - pulls hcmd from cmd queue
 * and adds to HP sg list.
 *
 * Note: uses transport's handlers which add data from queues to the bus
 * aggregation, here only scheduling is cared. All adding errors should be
 * handled inside transport. An error returned from transport could also mean
 * that the corresponding queue is empty.
 *
 * In this implementation it doesn't use the is_immediate return value from
 * the transport.
 *
 * @trans_slv - slave common transport
 * @return true if loading the list is permitted ('sg list loaded' bit is clear);
 */
static bool
iwl_idi_tx_policy_build_bus_agg_cmd(struct iwl_trans_slv *trans_slv)
{
	struct iwl_idi_trans_tx *trans_tx =
			IWL_TRANS_SLV_GET_IDI_TRANS_TX(trans_slv);
	int ret;

	if (test_and_set_bit(TX_SG_LP_CH, &trans_tx->sg_list_loaded))
		return false;

	/* try to pull as much cmds as possible */
	while (!iwl_idi_tx_bus_agg_full(trans_tx, TX_SG_LP_CH)) {
		ret = iwl_idi_tx_add_burst(trans_slv, TX_SG_LP_CH,
					   trans_slv->cmd_queue);
		if (!ret)
			trans_tx->txbus_added_policy[TX_SG_LP_CH]++;
		else
			break;
	}

	return true;
}

/**
 * iwl_idi_tx_policy_trigger - trigger an attempt to send Tx.
 * It implements a very basic scheduling/priority of queues -
 * first check if there are hcmd wating and if so - add them to sg list.
 * than for data commands perform Round Robin on all the data Tx queues.
 * @idi_policy - policy private data
 */
void iwl_idi_tx_policy_trigger(struct work_struct *data)
{
	struct iwl_trans_slv *trans_slv =
			container_of(data, struct iwl_trans_slv,
				     policy_trigger);

	struct iwl_idi_trans_tx *trans_tx =
		IWL_TRANS_SLV_GET_IDI_TRANS_TX(trans_slv);

	if (iwl_idi_tx_policy_build_bus_agg_cmd(trans_slv))
		iwl_idi_tx_policy_dma_arm_chan(trans_tx, TX_SG_LP_CH);
	if (iwl_idi_tx_policy_build_bus_agg_data(trans_slv))
		iwl_idi_tx_policy_dma_arm_chan(trans_tx, TX_SG_LP_CH);
}
