/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#include <linux/debugfs.h>

#include "iwl-drv.h"
#include "iwl-idi-ut.h"
#include "iwl-trans.h"
#include "iwl-mvm.h"
#include "iwl-mvm-old-commands.h"

#ifdef CPTCFG_IWLWIFI_TRANS_UT
#include "slave/idi_tx.h"
#include "slave/idi_internal.h"
#include "slave/shared.h"
#endif /* CPTCFG_IWLWIFI_TRANS_UT */

#ifdef CPTCFG_IWLWIFI_UT_TX_STORE
/* SRAM Defines */
#define AL_SRAM_NUM_K		48
#define AL_SRAM_SIZE		(1024 * AL_SRAM_NUM_K)

/* Copy of AL SRAM - For UT */
dma_addr_t IWL_IDI_UT_SRAM;
void *IWL_IDI_UT_SRAM_VIRTUAL;
#endif

/******************************************************************************
 *
 * Debugfs
 *
 *****************************************************************************/

/* UT defines */
#define IWL_IDI_UT_DEBUGFS_MAX_STRING		(256)
#define IWL_IDI_UT_DEBUGFS_MAX_CMD_SIZE	(3 * 1024)

/* create and remove of files */
#define IDI_UT_DEBUGFS_ADD_FILE(name, parent, mode) do {		\
	if (!debugfs_create_file(#name, mode, parent, op_mode,		\
				 &iwl_idi_ut_dbgfs_##name##_ops))	\
		goto err;						\
} while (0)

/* File Operation */
#define IDI_UT_DEBUGFS_READ_FUNC(name)					\
	static ssize_t iwl_idi_ut_dbgfs_##name##_read(struct file *file, \
					       char __user *user_buf,   \
					       size_t count, loff_t *ppos);

#define IDI_UT_DEBUGFS_WRITE_FUNC(name)					\
	static ssize_t iwl_idi_ut_dbgfs_##name##_write(struct file *file, \
					const char __user *user_buf,	\
					size_t count, loff_t *ppos);

#define IDI_UT_DEBUGFS_READ_FILE_OPS(name)				\
	static const struct file_operations iwl_idi_ut_dbgfs_##name##_ops = { \
		.read = iwl_idi_ut_dbgfs_##name##_read,			\
		.open = simple_open,					\
		.llseek = generic_file_llseek,				\
	};

#define IDI_UT_DEBUGFS_WRITE_FILE_OPS(name)				\
	IDI_UT_DEBUGFS_WRITE_FUNC(name);				\
	static const struct file_operations iwl_idi_ut_dbgfs_##name##_ops = { \
		.write = iwl_idi_ut_dbgfs_##name##_write,		\
		.open = simple_open,					\
		.llseek = generic_file_llseek,				\
	};

#define IDI_UT_DEBUGFS_READ_WRITE_FILE_OPS(name)			\
		IDI_UT_DEBUGFS_READ_FUNC(name);			\
		IDI_UT_DEBUGFS_WRITE_FUNC(name);			\
	static const struct file_operations iwl_idi_ut_dbgfs_##name##_ops = { \
		.write = iwl_idi_ut_dbgfs_##name##_write,		\
		.read = iwl_idi_ut_dbgfs_##name##_read,			\
		.open = simple_open,					\
		.llseek = generic_file_llseek,				\
	};

/*
 * Sends an echo command to the uCode and tests that it has arrived.
 *
 * Print the result of the echo sending.
 */
static ssize_t iwl_idi_ut_dbgfs_echo_command_read(struct file *file,
					      char __user *user_buf,
					      size_t count, loff_t *ppos)
{
	int pos = 0, ret;
	char result_str[IWL_IDI_UT_DEBUGFS_MAX_STRING];
	const size_t res_size = sizeof(result_str);
	struct iwl_op_mode *op_mode = file->private_data;
	struct iwl_host_cmd hcmd = {
		.id = REPLY_ECHO,
		.len = { 0 },
		.flags = CMD_SYNC,
	};

	/*Send command */
	ret = iwl_trans_send_cmd((struct iwl_trans *)op_mode->trans, &hcmd);
	pos += scnprintf(result_str + pos, res_size - pos,
			 "Command sending %s - %d\n",
			 (ret) ? "FAILED" : "SUCCESS",
			 ret);

	return simple_read_from_buffer(user_buf, count, ppos, result_str, pos);
}

/*
 * Sends a host command of the given buffer to the uCode and tests that
 * it has arrived.
 *
 * Print the result of the command sending.
 */
static ssize_t iwl_idi_ut_dbgfs_send_buffer_write(struct file *file,
					      const char __user *user_buf,
					      size_t count, loff_t *ppos)
{
	int ret;
	struct iwl_op_mode *op_mode = file->private_data;
	u8 *buf;
	int data_size = 640;
	int firstlen;
	struct sk_buff *skb;
	struct ieee80211_hdr *hdr;

	/* Test size */
	if (count > IWL_IDI_UT_DEBUGFS_MAX_CMD_SIZE || count < sizeof(*hdr))
		return 0;

	firstlen = sizeof(struct iwl_tx_cmd) +
		   sizeof(struct iwl_cmd_header) + 30;

	buf = kmalloc(data_size, GFP_KERNEL);
	skb = alloc_skb(count, GFP_KERNEL);

	/* Copy given command */
	if (copy_from_user(skb_put(skb, count), user_buf, count))
		return -EFAULT;

	hdr = (struct ieee80211_hdr *)skb->data;
	hdr->frame_control = 0;

	ret = iwl_trans_tx((struct iwl_trans *)op_mode->trans,
			skb,
			(struct iwl_device_cmd *)(buf + data_size - firstlen),
			2);
	if (ret)
		return ret;
	return count;
}

/* The debug monitor command id and struct */
#define DEBUG_MONITOR_CMD 0xf2
struct iwl_dbgm_cmd {
	__le32 mon_en_msk;
	__le32 prph_only_mon;
	__le32 tfd_info_comp_msk;
	__le32 tfd_info_comp_val;
} __packed;

/*
 * Sends a DBGM enable command to the uCode and tests that it has arrived.
 * DBGM HCMD has a 4 dword payload and has no real effect on regular driver
 * flows which enables sending a HCMD with payload tests.
 * Print the result of the dbgm sending.
 */
static ssize_t iwl_idi_ut_dbgfs_dbgm_enable_read(struct file *file,
					      char __user *user_buf,
					      size_t count, loff_t *ppos)
{
	int pos = 0, ret;
	char result_str[IWL_IDI_UT_DEBUGFS_MAX_STRING];
	const size_t res_size = sizeof(result_str);
	struct iwl_op_mode *op_mode = file->private_data;
	/* Build the commands */
	struct iwl_dbgm_cmd cmd = {
		.mon_en_msk = cpu_to_le32(0xaa),
		.prph_only_mon = cpu_to_le32(0xbb),
		.tfd_info_comp_msk = cpu_to_le32(0xcc),
		.tfd_info_comp_val = cpu_to_le32(0xdd),
	};
	struct iwl_host_cmd hcmd = {
		.id = DEBUG_MONITOR_CMD,
		.len = { sizeof(struct iwl_dbgm_cmd), },
		.flags = CMD_SYNC,
		.data = { &cmd, },
	};

	/*Send command */
	ret = iwl_trans_send_cmd((struct iwl_trans *)op_mode->trans, &hcmd);
	pos += scnprintf(result_str + pos, res_size - pos,
			 "DBGM Command sending %s - %d\n",
			 (ret) ? "FAILED" : "SUCCESS",
			 ret);

	return simple_read_from_buffer(user_buf, count, ppos, result_str, pos);
}

#ifdef CPTCFG_IWLWIFI_UT_TX_STORE
/*
 * Compares the mock AL SRAM with the actuall SRAM,
 * checking for differences that can emply stream errors.
 *
 * Prints success or fail.
 */
static ssize_t iwl_idi_ut_dbgfs_sram_test_read(struct file *file,
					       char __user *user_buf,
					       size_t count, loff_t *ppos)
{
	int pos = 0, ret;
	char result_str[IWL_IDI_UT_DEBUGFS_MAX_STRING];
	const size_t res_size = sizeof(result_str);

	/* Compare SRAMs */
	ret = memcmp(IWL_IDI_UT_SRAM_VIRTUAL,
		     AL_SRAM_VIRTUAL_ADDRESS,
		     AL_SRAM_SIZE);
	pos += scnprintf(result_str + pos, res_size - pos,
			 "Comparing SRAMs:  %s - %d\n",
			 (ret) ? "FAILED" : "SUCCESS",
			 ret);

	return simple_read_from_buffer(user_buf, count, ppos, result_str, pos);
}

IDI_UT_DEBUGFS_READ_FILE_OPS(sram_test);
#endif

IDI_UT_DEBUGFS_READ_FILE_OPS(echo_command);
IDI_UT_DEBUGFS_WRITE_FILE_OPS(send_buffer);
IDI_UT_DEBUGFS_READ_FILE_OPS(dbgm_enable);

/* The debugfs UT direcotry */
static struct dentry *dir_ut;

/*
 * Register the debugfs of the UT module.
 */
static int iwl_idi_ut_debugfs_register(struct iwl_op_mode *op_mode)
{
	struct iwl_mvm_priv *priv = IWL_OP_MODE_GET_PRIV(op_mode);
	struct dentry *drv_dir = priv->debugfs_dir;

	dir_ut = debugfs_create_dir("UT", drv_dir);
	if (!dir_ut)
		goto err;

	IDI_UT_DEBUGFS_ADD_FILE(echo_command, dir_ut, S_IWUSR | S_IRUSR);
	IDI_UT_DEBUGFS_ADD_FILE(send_buffer, dir_ut, S_IWUSR | S_IRUSR);
	IDI_UT_DEBUGFS_ADD_FILE(dbgm_enable, dir_ut, S_IRUSR);

#ifdef CPTCFG_IWLWIFI_UT_TX_STORE
	IDI_UT_DEBUGFS_ADD_FILE(sram_test, dir_ut, S_IRUSR);
#endif

	return 0;

err: /* Called from the DEBUGFS_ADD_FILE Macros*/
	return -ENOMEM;
}

/*
 * Init the IDI UT module.
 */
int iwl_idi_ut_init(struct iwl_op_mode *op_mode)
{
	int ret;

	/* Init UT debugfs */
	ret = iwl_idi_ut_debugfs_register(op_mode);
	if (ret) {
		pr_err("Cannot create debugfs dir");
		goto out;
	}
	pr_info("IDI-UT: debugfs created\n");

#ifdef CPTCFG_IWLWIFI_UT_TX_STORE
	/* Init the IDI SRAM */
	IWL_IDI_UT_SRAM_VIRTUAL = dma_alloc_coherent(op_mode->trans->dev,
						     AL_SRAM_SIZE,
						     &IWL_IDI_UT_SRAM,
						     GFP_KERNEL);

	if (!IWL_IDI_UT_SRAM_VIRTUAL) {
		pr_err("Cannot allocate memory for the IDI SRAM");
		ret = -ENOMEM;
		goto out;
	}
	memset(IWL_IDI_UT_SRAM_VIRTUAL, 0, AL_SRAM_SIZE);
	pr_info("IDI-UT: Allocated  UT SRAM : %p", IWL_IDI_UT_SRAM_VIRTUAL);
#endif

out:
	return ret;
}
IWL_EXPORT_SYMBOL(iwl_idi_ut_init);

/*
 * FREE the IDI UT module.
 */
void iwl_idi_ut_free(struct iwl_op_mode *op_mode)
{
	pr_info("IDI-UT: Freeing UT module");
#ifdef CPTCFG_IWLWIFI_UT_TX_STORE
	/* Release the IDI SRAM*/
	if (IWL_IDI_UT_SRAM_VIRTUAL) {
		dma_free_coherent(op_mode->trans->dev,
				  AL_SRAM_SIZE,
				  IWL_IDI_UT_SRAM_VIRTUAL,
				  IWL_IDI_UT_SRAM);
		pr_info("Freed UT SRAM");
		IWL_IDI_UT_SRAM_VIRTUAL = NULL;
	}
	/* Release the debugfs */
	if (dir_ut) {
		debugfs_remove_recursive(dir_ut);
		dir_ut = NULL;
		pr_info("Freed UT debugfs");
	}
#endif
}
IWL_EXPORT_SYMBOL(iwl_idi_ut_free);

/******************************************************************************
 *
 * TEST SUITS - EXAMPLE
 *
 *****************************************************************************/

static int iwl_idi_ut_test_case_example_a(void *data)
{
	char *data_str = (char *)data;
	pr_err("%s: %s", __func__, data_str);
	return -1;
}

static int iwl_idi_ut_test_case_example_b(void *data)
{
	char *data_str = (char *)data;
	pr_err("%s: %s", __func__, data_str);
	return 0;
}

int iwl_idi_ut_test_suit_example(void *data)
{
	IDI_UT_TEST_SUIT_START;

	IWL_IDI_UT_ADD_TEST_CASE(iwl_idi_ut_test_case_example_a, "TEST 1");
	IWL_IDI_UT_ADD_TEST_CASE(iwl_idi_ut_test_case_example_b, "TEST 2");

	IDI_UT_TEST_SUIT_END;
}

/******************************************************************************
 *
 * UNIT TESTS
 *
 *****************************************************************************/

#ifdef CPTCFG_IWLWIFI_TRANS_UT
/**
 * Reclaim flow emulation for unit test.
 * Does not actually perform RX flow just the reclaim flow.
 */
void iwl_idi_tx_handle_dispatch_emu(struct iwl_trans *trans,
				     struct iwl_rx_cmd_buffer *rxcb)
{
	struct iwl_rx_packet *pkt = rxb_addr(rxcb);
	struct iwl_slv_tx_cmd_entry *cmd_entry = NULL;
	struct sk_buff_head skbs;
	int txq_id = SEQ_TO_QUEUE(le16_to_cpu(pkt->hdr.sequence));
	int reclaim, ret = 0;

	struct iwl_idi_trans_tx *trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);

	IWL_DEBUG_TX(trans, "Starting debug reclaim flow");

	pr_err("%s: txq_id %d, ssn %d",
	       __func__,
	       txq_id,
	       SEQ_TO_INDEX(pkt->hdr.sequence));

	/* If not command queue reclaim data */
	if (txq_id != trans_tx->cmd) {
		__skb_queue_head_init(&skbs);
		iwl_trans_slv_tx_data_reclaim(trans, txq_id,
					SEQ_TO_INDEX(pkt->hdr.sequence) + 1,
					&skbs);
	} else {
		reclaim = !(pkt->hdr.sequence & SEQ_RX_FRAME);
		if (reclaim) {
			iwl_slv_tx_get_cmd_entry(trans, pkt, &cmd_entry);
			if (cmd_entry == NULL)
				return;
			iwl_slv_tx_cmd_complete(trans, rxcb, cmd_entry, ret);
		}
	}
}
#endif /* CPTCFG_IWLWIFI_TRANS_UT */

