/* packet-e212.c
 * Routines for output and filtering of E.212 numbers common
 * to many dissectors.
 * Copyright 2006, Anders Broman <anders.broman@ericsson.com>
 *
 * $Id: packet-e212.c 29149 2009-07-21 06:02:12Z etxrab $
 *
 * Wireshark - Network traffic analyzer
 * By Gerald Combs <gerald@wireshark.org>
 * Copyright 1998 Gerald Combs
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
# include "config.h"
#endif

#include <glib.h>
#include <epan/packet.h>

#include "packet-e212.h"


/*
 * Annex to ITU Operational Bulletin
 * No. 897 - 1.XII.2007
 * 
 * COMPLEMENT TO ITU-T RECOMMENDATION E.212 (05/2004)
 */
const value_string E212_codes[] = {
	{  202,	"Greece" },
	{  204,	"Netherlands (Kingdom of the)" },
	{  206,	"Belgium" },
	{  208,	"France" },
	{  212,	"Monaco (Principality of)" },
	{  213,	"Andorra (Principality of)" },
	{  214,	"Spain" },
	{  216,	"Hungary (Republic of)" },
	{  218,	"Bosnia and Herzegovina" },
	{  219,	"Croatia (Republic of)" },
	{  220,	"Serbia (Republic of)" },
	{  222,	"Italy" },
	{  225,	"Vatican City State" },
	{  226,	"Romania" },
	{  228,	"Switzerland (Confederation of)" },
	{  230,	"Czech Republic" },
	{  231,	"Slovak Republic" },
	{  232,	"Austria" },
	{  234,	"United Kingdom of Great Britain and Northern Ireland" },
	{  235,	"United Kingdom of Great Britain and Northern Ireland" },
	{  238,	"Denmark" },
	{  240,	"Sweden" },
	{  242,	"Norway" },
	{  244,	"Finland" },
	{  246,	"Lithuania (Republic of)" },
	{  247,	"Latvia (Republic of)" },
	{  248,	"Estonia (Republic of)" },
	{  250,	"Russian Federation" },
	{  255,	"Ukraine" },
	{  257,	"Belarus (Republic of)" },
	{  259,	"Moldova (Republic of)" },
	{  260,	"Poland (Republic of)" },
	{  262,	"Germany (Federal Republic of)" },
	{  266,	"Gibraltar" },
	{  268,	"Portugal" },
	{  270,	"Luxembourg" },
	{  272,	"Ireland" },
	{  274,	"Iceland" },
	{  276,	"Albania (Republic of)" },
	{  278,	"Malta" },
	{  280,	"Cyprus (Republic of)" },
	{  282,	"Georgia" },
	{  283,	"Armenia (Republic of)" },
	{  284,	"Bulgaria (Republic of)" },
	{  286,	"Turkey" },
	{  288,	"Faroe Islands" },
	{  290,	"Greenland (Denmark)" },
	{  292,	"San Marino (Republic of)" },
	{  293,	"Slovenia (Republic of)" },
	{  294,	"The Former Yugoslav Republic of Macedonia" },
	{  295,	"Liechtenstein (Principality of)" },
	{  297, "Montenegro (Republic of)" },
	{  302,	"Canada" },
	{  308,	"Saint Pierre and Miquelon (Collectivite territoriale de la Republique francaise)" },
	{  310,	"United States of America" },
	{  311,	"United States of America" },
	{  312,	"United States of America" },
	{  313,	"United States of America" },
	{  314,	"United States of America" },
	{  315,	"United States of America" },
	{  316,	"United States of America" },
	{  330,	"Puerto Rico" },
	{  332,	"United States Virgin Islands" },
	{  334,	"Mexico" },
	{  338,	"Jamaica" },
	{  340,	"Martinique (French Department of)" },
	{  340,	"Guadeloupe (French Department of)" },
	{  342,	"Barbados" },
	{  344,	"Antigua and Barbuda" },
	{  346,	"Cayman Islands" },
	{  348,	"British Virgin Islands" },
	{  350,	"Bermuda" },
	{  352,	"Grenada" },
	{  354,	"Montserrat" },
	{  356,	"Saint Kitts and Nevis" },
	{  358,	"Saint Lucia" },
	{  360,	"Saint Vincent and the Grenadines" },
	{  362,	"Netherlands Antilles" },
	{  363,	"Aruba" },
	{  364,	"Bahamas (Commonwealth of the)" },
	{  365,	"Anguilla" },
	{  366,	"Dominica (Commonwealth of)" },
	{  368,	"Cuba" },
	{  370,	"Dominican Republic" },
	{  372,	"Haiti (Republic of)" },
	{  374,	"Trinidad and Tobago" },
	{  376,	"Turks and Caicos Islands" },
	{  400,	"Azerbaijani Republic" },
	{  401,	"Kazakhstan (Republic of)" },
	{  402,	"Bhutan (Kingdom of)" },
	{  404,	"India (Republic of)" },
	{  410,	"Pakistan (Islamic Republic of)" },
	{  412,	"Afghanistan" },
	{  413,	"Sri Lanka (Democratic Socialist Republic of)" },
	{  414,	"Myanmar (Union of)" },
	{  415,	"Lebanon" },
	{  416,	"Jordan (Hashemite Kingdom of)" },
	{  417,	"Syrian Arab Republic" },
	{  418,	"Iraq (Republic of)" },
	{  419,	"Kuwait (State of)" },
	{  420,	"Saudi Arabia (Kingdom of)" },
	{  421,	"Yemen (Republic of)" },
	{  422,	"Oman (Sultanate of)" },
	{  424,	"United Arab Emirates" },
	{  425,	"Israel (State of)" },
	{  426,	"Bahrain (Kingdom of)" },
	{  427,	"Qatar (State of)" },
	{  428,	"Mongolia" },
	{  429,	"Nepal" },
	{  430,	"United Arab Emirates (Abu Dhabi)" },
	{  431,	"United Arab Emirates (Dubai)" },
	{  432,	"Iran (Islamic Republic of)" },
	{  434,	"Uzbekistan (Republic of)" },
	{  436,	"Tajikistan (Republic of)" },
	{  437,	"Kyrgyz Republic" },
	{  438,	"Turkmenistan" },
	{  440,	"Japan" },
	{  441,	"Japan" },
	{  450,	"Korea (Republic of)" },
	{  452,	"Viet Nam (Socialist Republic of)" },
	{  454,	"Hongkong, China" },
	{  455,	"Macao, China" },
	{  456,	"Cambodia (Kingdom of)" },
	{  457,	"Lao People's Democratic Republic" },
	{  460,	"China (People's Republic of)" },
	{  461,	"China (People's Republic of)" },
	{  466,	"Taiwan, China" },
	{  467,	"Democratic People's Republic of Korea" },
	{  470,	"Bangladesh (People's Republic of)" },
	{  472,	"Maldives (Republic of)" },
	{  502,	"Malaysia" },
	{  505,	"Australia" },
	{  510,	"Indonesia (Republic of)" },
	{  514,	"Democratic Republic of Timor-Leste" },
	{  515,	"Philippines (Republic of the)" },
	{  520,	"Thailand" },
	{  525,	"Singapore (Republic of)" },
	{  528,	"Brunei Darussalam" },
	{  530,	"New Zealand" },
	{  534,	"Previously assigned to Northern Mariana Islands (Commonwealth of the)" },
	{  535,	"Previously assigned to Guam" },
	{  536,	"Nauru (Republic of)" },
	{  537,	"Papua New Guinea" },
	{  539,	"Tonga (Kingdom of)" },
	{  540,	"Solomon Islands" },
	{  541,	"Vanuatu (Republic of)" },
	{  542,	"Fiji (Republic of)" },
	{  543,	"Wallis and Futuna (Territoire francais d'outre-mer)" },
	{  544,	"American Samoa" },
	{  545,	"Kiribati (Republic of)" },
	{  546,	"New Caledonia (Territoire francais d'outre-mer)" },
	{  547,	"French Polynesia (Territoire francais d'outre-mer)" },
	{  548,	"Cook Islands" },
	{  549,	"Samoa (Independent State of)" },
	{  550,	"Micronesia (Federated States of)" },
	{  551,	"Marshall Islands (Republic of the)" },
	{  552,	"Palau (Republic of)" },
	{  602,	"Egypt (Arab Republic of)" },
	{  603,	"Algeria (People's Democratic Republic of)" },
	{  604,	"Morocco (Kingdom of)" },
	{  605,	"Tunisia" },
	{  606,	"Libya (Socialist People's Libyan Arab Jamahiriya)" },
	{  607,	"Gambia (Republic of the)" },
	{  608,	"Senegal (Republic of)" },
	{  609,	"Mauritania (Islamic Republic of)" },
	{  610,	"Mali (Republic of)" },
	{  611,	"Guinea (Republic of)" },
	{  612,	"Cote d'Ivoire (Republic of)" },
	{  613,	"Burkina Faso" },
	{  614,	"Niger (Republic of the)" },
	{  615,	"Togolese Republic" },
	{  616,	"Benin (Republic of)" },
	{  617,	"Mauritius (Republic of)" },
	{  618,	"Liberia (Republic of)" },
	{  619,	"Sierra Leone" },
	{  620,	"Ghana" },
	{  621,	"Nigeria (Federal Republic of)" },
	{  622,	"Chad (Republic of)" },
	{  623,	"Central African Republic" },
	{  624,	"Cameroon (Republic of)" },
	{  625,	"Cape Verde (Republic of)" },
	{  626,	"Sao Tome and Principe (Democratic Republic of)" },
	{  627,	"Equatorial Guinea (Republic of)" },
	{  628,	"Gabonese Republic" },
	{  629,	"Congo (Republic of the)" },
	{  630,	"Democratic Republic of the Congo" },
	{  631,	"Angola (Republic of)" },
	{  632,	"Guinea-Bissau (Republic of)" },
	{  633,	"Seychelles (Republic of)" },
	{  634,	"Sudan (Republic of the)" },
	{  635,	"Rwanda (Republic of)" },
	{  636,	"Ethiopia (Federal Democratic Republic of)" },
	{  637,	"Somali Democratic Republic" },
	{  638,	"Djibouti (Republic of)" },
	{  639,	"Kenya (Republic of)" },
	{  640,	"Tanzania (United Republic of)" },
	{  641,	"Uganda (Republic of)" },
	{  642,	"Burundi (Republic of)" },
	{  643,	"Mozambique (Republic of)" },
	{  645,	"Zambia (Republic of)" },
	{  646,	"Madagascar (Republic of)" },
	{  647,	"French Departments and Territories in the Indian Ocean" },
	{  648,	"Zimbabwe (Republic of)" },
	{  649,	"Namibia (Republic of)" },
	{  650,	"Malawi" },
	{  651,	"Lesotho (Kingdom of)" },
	{  652,	"Botswana (Republic of)" },
	{  653,	"Swaziland (Kingdom of)" },
	{  654,	"Comoros (Union of the)" },
	{  655,	"South Africa (Republic of)" },
	{  657,	"Eritrea" },
	{  702,	"Belize" },
	{  704,	"Guatemala (Republic of)" },
	{  706,	"El Salvador (Republic of)" },
	{  708,	"Honduras (Republic of)" },
	{  710,	"Nicaragua" },
	{  712,	"Costa Rica" },
	{  714,	"Panama (Republic of)" },
	{  716,	"Peru" },
	{  722,	"Argentine Republic" },
	{  724,	"Brazil (Federative Republic of)" },
	{  730,	"Chile" },
	{  732,	"Colombia (Republic of)" },
	{  734,	"Venezuela (Bolivarian Republic of)" },
	{  736,	"Bolivia (Republic of)" },
	{  738,	"Guyana" },
	{  740,	"Ecuador" },
	{  742,	"French Guiana (French Department of)" },
	{  744,	"Paraguay (Republic of)" },
	{  746,	"Suriname (Republic of)" },
	{  748,	"Uruguay (Eastern Republic of)" },
	{  750, "Falkland Islands (Malvinas)" },
	{  901,	"International Mobile, shared code" },
	{ 0, NULL }
};

/*
 * Annex to ITU Operational Bulletin
 * No. 932 - 15.V.2009
 * 
 * According to Recommendation ITU-T E.212 (05/2008)
 */
const value_string mcc_mnc_codes[] = {
	{  202001, "Cosmote" },
	{  202005, "Vodafone - Panafon" },
	{  202009, "Info Quest S.A." },
	{  202010, "WIND Hellas" },
	{  204002, "Tele2 (Netherlands) B.V." },
	{  204003, "Blyk N.V." },
	{  204004, "Vodafone Libertel N.V." },
	{  204005, "Elephant Talk Comm. Premium Rate Serv. Neth. B.V." },
	{  204006, "Barablu Mobile Benelux Ltd" },
	{  204007, "Teleena holding B.V." },
	{  204008, "KPN Mobile The Netherlands B.V." },
	{  204010, "KPN B.V." },
	{  204012, "Telfort B.V." },
	{  204014, "INMO B.V." },
	{  204016, "T-Mobile Netherlands B.V." },
	{  204018, "Telfort B.V." },
	{  204020, "Orange Nederland N.V." },
	{  204021, "ProRail B.V." },
	{  204060, "KPN B.V." },
	{  204069, "KPN Mobile The Netherlands B.V." },
	{  206001, "Proximus" },
	{  206010, "Mobistar" },
	{  206020, "Base" },
	{  208001, "Orange France" },
	{  208002, "Orange France" },
	{  208005, "Globalstar Europe" },
	{  208006, "Globalstar Europe" },
	{  208007, "Globalstar Europe" },
	{  208010, "S.F.R." },
	{  208011, "S.F.R. (UMTS)" },
	{  208013, "SFR (Zones Blanches)" },
	{  208020, "Bouygues Telecom" },
	{  208021, "Bouygues Telecom" },
	{  208088, "Bouygues Telecom (Zones Blanches)" },
	{  213003, "Mobiland" },
	{  214001, "Vodafone" },
	{  214003, "Amena" },
	{  214004, "Xfera" },
	{  214007, "Movistar" },
	{  216001, "Pannon GSM" },
	{  216030, "Westel Mobile" },
	{  216070, "Vodafone" },
	{  218003, "Eronet Mobile Communications Ltd." },
	{  218005, "MOBI'S (Mobilina Srpske)" },
	{  218090, "GSMBIH" },
	{  219001, "T-Mobile Hrvatska d.o.o./T-Mobile Croatia LLC" },
	{  219002, "Tele2/Tele2 d.o.o." },
	{  219010, "VIPnet/VIPnet d.o.o." },
	{  220001, "Telenor d.o.o." },
	{  220003, "Telekom Srbija a.d." },
	{  220005, "Vip mobile d.o.o." },
	{  222001, "Telecom Italia Mobile (TIM)" },
	{  222002, "Elsacom" },
	{  222010, "Omnitel Pronto Italia (OPI)" },
	{  222077, "IPSE 2000" },
	{  222088, "Wind" },
	{  222098, "Blu" },
	{  222099, "H3G" },
	{  226001, "Vodafone Romania SA" },
	{  226002, "Romtelecom (CDMA)" },
	{  226003, "Cosmorom" },
	{  226010, "Orange Romania" },
	{  228001, "Swisscom Mobile AG" },
	{  228002, "TDC Switzerland AG" },
	{  228003, "Orange Communications SA" },
	{  228005, "Comfone AG" },
	{  228006, "SBB AG" },
	{  228007, "IN&Phone SA" },
	{  228008, "Tele2 Telecommunications AG" },
	{  228012, "TDC Switzerland AG" },
	{  228051, "Bebbicell AG" },
	{  230001, "T-Mobile Czech Republic a.s." },
	{  230002, "Telefonica O2 Czech Republic a.s." },
	{  230003, "Vodafone Czech Republic a.s." },
	{  230004, "Mobilkom a.s." },
	{  230098, "Sprava Zeleznicni Dopravni Cesty" },
	{  230099, "Vodafone Czech Republic a.s. R&D Centre" },
	{  231001, "Orange, GSM" },
	{  231002, "Eurotel, GSM & NMT" },
	{  231004, "Eurotel, UMTS" },
	{  231005, "Orange, UMTS" },
	{  232001, "Mobilkom Austria Aktiengesellschaft" },
	{  232002, "Mobilkom Austria Aktiengesellschaft" },
	{  232003, "T-Mobile Austria GmbH" },
	{  232004, "T-Mobile Austria GmbH" },
	{  232005, "Orange Austria Telecommunication GmbH" },
	{  232006, "Orange Austria Telecommunication GmbH" },
	{  232007, "T-Mobile Austria GmbH" },
	{  232009, "Mobilkom Austria Aktiengesellschaft" },
	{  232010, "Hutchison 3G Austria GmbH" },
	{  232011, "Mobilkom Austria Aktiengesellschaft" },
	{  232012, "Orange Austria Telecommunication GmbH" },
	{  232014, "Hutchison 3G Austria GmbH" },
	{  232015, "Barablu Mobile Austria Ltd" },
	{  232091, "OBB - Infrastruktur Bau AG" },
	{  234000, "British Telecom" },
	{  234001, "Mapesbury Communications Ltd." },
	{  234002, "O2 UK Ltd." },
	{  234003, "Jersey Telenet Ltd" },
	{  234004, "FMS Solutions Ltd" },
	{  234005, "Colt Mobile Telecommunications Ltd" },
	{  234006, "Internet One Ltd" },
	{  234007, "Cable and Wireless plc" },
	{  234009, "Wire9 Telecom plc" },
	{  234010, "O2 UK Ltd." },
	{  234011, "O2 UK Ltd." },
	{  234012, "Ntework Rail Infrastructure Ltd" },
	{  234013, "Ntework Rail Infrastructure Ltd" },
	{  234014, "Hay Systems Ltd" },
	{  234015, "Vodafone Ltd." },
	{  234016, "Opal Telecom Ltd" },
	{  234017, "Flextel Ltd" },
	{  234018, "Wire9 Telecom plc" },
	{  234019, "Teleware plc" },
	{  234020, "Hutchison 3G UK Ltd." },
	{  234030, "T-Mobile UK" },
	{  234031, "T-Mobile UK" },
	{  234032, "T-Mobile UK" },
	{  234033, "Orange" },
	{  234034, "Orange" },
	{  234050, "Jersey Telecom" },
	{  234055, "Cable and Wireless Guensey Ltd" },
	{  234058, "Manx Telecom" },
	{  234075, "Inquam Telecom (Holdings) Ltd." },
	{  234076, "British Telecom" },
	{  234078, "Airwave mmO2 Ltd" },
	{  238001, "TDC Mobil" },
	{  238002, "Sonofon" },
	{  238003, "MIGway A/S" },
	{  238006, "Hi3G" },
	{  238007, "Barablu Mobile Ltd." },
	{  238010, "TDC Mobil" },
	{  238020, "Telia" },
	{  238030, "Telia Mobile" },
	{  238077, "Tele2" },
	{  240001, "Telia Sonera AB" },
	{  240002, "H3G Access AB" },
	{  240003, "AINMT Sverige AB" },
	{  240004, "3G Infrastructure Services AB" },
	{  240005, "Svenska UMTS-Nat AB" },
	{  240006, "Telenor Sverige AB" },
	{  240007, "Tele2 Sverige AB" },
	{  240008, "Telenor Sverige AB" },
	{  240009, "Telenor Mobile Sverige" },
	{  240010, "Spring Mobil AB" },
	{  240011, "Linholmen Science Park AB" },
	{  240012, "Barablu Mobile Scandinavia Ltd" },
	{  240013, "Ventelo Sverige AB" },
	{  240014, "TDC Mobil A/S" },
	{  240015, "Wireless Maingate Nordic AB" },
	{  240016, "42IT AB" },
	{  240017, "Gotalandsnatet AB" },
	{  240018, "Generic Mobile Systems Sweden AB" },
	{  240020, "Wireless Maingate Message Services AB" },
	{  240021, "Banverket" },
	{  242001, "Telenor Mobil AS" },
	{  242002, "Netcom GSM AS" },
	{  242003, "Teletopia Mobile Communications AS" },
	{  242004, "Tele2 Norge AS" },
	{  244004, "Finnet Networks Ltd." },
	{  244005, "Elisa Matkapuhelinpalvelut Ltd." },
	{  244009, "Finnet Group" },
	{  244012, "Finnet Networks Ltd." },
	{  244014, "Alands Mobiltelefon AB" },
	{  244016, "Oy Finland Tele2 AB" },
	{  244021, "Saunalahti Group Ltd." },
	{  244091, "Sonera Carrier Networks Oy" },
	{  246001, "Omnitel" },
	{  246002, "Bite GSM" },
	{  246003, "Tele2" },
	{  247001, "Latvijas Mobilais Telefons SIA" },
	{  247002, "Tele2" },
	{  247003, "Telekom Baltija" },
	{  247004, "Beta Telecom" },
	{  247005, "Bite Mobile" },
	{  247006, "Rigatta" },
	{  247007, "Master Telecom" },
	{  247008, "IZZI" },
	{  248001, "EMT GSM" },
	{  248002, "RLE" },
	{  248003, "Tele2" },
	{  248004, "OY Top Connect" },
	{  248005, "AS Bravocom Mobiil" },
	{  248006, "OY ViaTel" },
	{  248007, "Televorgu AS" },
	{  248071, "Siseministeerium (Ministry of Interior)" },
	{  250001, "Mobile Telesystems" },
	{  250002, "Megafon" },
	{  250003, "Nizhegorodskaya Cellular Communications" },
	{  250004, "Sibchallenge" },
	{  250005, "Mobile Comms System" },
	{  250007, "BM Telecom" },
	{  250010, "Don Telecom" },
	{  250011, "Orensot" },
	{  250012, "Baykal Westcom" },
	{  250013, "Kuban GSM" },
	{  250016, "New Telephone Company" },
	{  250017, "Ermak RMS" },
	{  250019, "Volgograd Mobile" },
	{  250020, "ECC" },
	{  250028, "Extel" },
	{  250039, "Uralsvyazinform" },
	{  250044, "Stuvtelesot" },
	{  250092, "Printelefone" },
	{  250093, "Telecom XXI" },
	{  250099, "Bec Line GSM" },
	{  255001, "Ukrainian Mobile Communication, UMC" },
	{  255002, "Ukranian Radio Systems, URS" },
	{  255003, "Kyivstar GSM" },
	{  255004, "International Telecommunications Ltd." },
	{  255005, "Golden Telecom" },
	{  255006, "Astelit" },
	{  255007, "Ukrtelecom" },
	{  255021, "CJSC - Telesystems of Ukraine" },
	{  257001, "MDC Velcom" },
	{  257002, "MTS" },
	{  259001, "Orange Moldova GSM" },
	{  259002, "Moldcell GSM" },
	{  259004, "Eventis Mobile GSM" },
	{  260001, "Plus GSM (Polkomtel S.A.)" },
	{  260002, "ERA GSM (Polska Telefonia Cyfrowa Sp. Z.o.o.)" },
	{  260003, "Idea (Polska Telefonia Komorkowa Centertel Sp. Z.o.o)" },
	{  260004, "Tele2 Polska (Tele2 Polska Sp. Z.o.o.)" },
	{  260005, "IDEA (UMTS)/PTK Centertel sp. Z.o.o." },
	{  260006, "Netia Mobile" },
	{  260007, "Premium internet" },
	{  260008, "E-Telko" },
	{  260009, "Telekomunikacja Kolejowa (GSM-R)" },
	{  260010, "Telefony Opalenickie" },
	{  262001, "T-Mobile Deutschland GmbH" },
	{  262002, "Vodafone D2 GmbH" },
	{  262003, "E-Plus Mobilfunk GmbH & Co. KG" },
	{  262004, "Vodafone D2 GmbH" },
	{  262005, "E-Plus Mobilfunk GmbH & Co. KG" },
	{  262006, "T-Mobile Deutschland GmbH" },
	{  262007, "O2 (Germany) GmbH & Co. OHG" },
	{  262008, "O2 (Germany) GmbH & Co. OHG" },
	{  262009, "Vodafone D2 GmbH" },
	{  262010, "Arcor AG & Co." },
	{  262011, "O2 (Germany) GmbH & Co. OHG" },
	{  262012, "Dolphin Telecom (Deutschland) GmbH" },
	{  262013, "Mobilcom Multimedia GmbH" },
	{  262014, "Group 3G UMTS GmbH (Quam)" },
	{  262015, "Airdata AG" },
	{  262076, "Siemens AG, ICMNPGUSTA" },
	{  262077, "E-Plus Mobilfunk GmbH & Co. KG" },
	{  266001, "Gibtelecom GSM" },
	{  266006, "CTS" },
	{  266009, "Cloud9 Mobile Communications" },
	{  268001, "Vodafone Telecel - Comunicacoes Pessoais, S.A." },
	{  268003, "Optimus - Telecomunicacoes, S.A." },
	{  268005, "Oniway - Inforcomunicacoes, S.A." },
	{  268006, "TMN - Telecomunicacoes Moveis Nacionais, S.A." },
	{  270001, "P&T Luxembourg" },
	{  270077, "Tango" },
	{  270099, "Voxmobile S.A." },
	{  272001, "Vodafone Ireland Plc" },
	{  272002, "Digifone mm02 Ltd." },
	{  272003, "Meteor Mobile Communications Ltd." },
	{  272007, "Eircom" },
	{  272009, "Clever Communications Ltd." },
	{  274001, "Iceland Telecom Ltd." },
	{  274002, "Og fjarskipti hf (Vodafone Iceland)" },
	{  274003, "Og fjarskipti hf (Vodafone Iceland)" },
	{  274004, "IMC Islande ehf" },
	{  274007, "IceCell ehf" },
	{  276001, "AMC" },
	{  276002, "Vodafone Albania" },
	{  276003, "Eagle Mobile" },
	{  278001, "Vodafone Malta" },
	{  278021, "go mobile" },
	{  278077, "3G Telecommunications Ltd" },
	{  280001, "CYTA" },
	{  280010, "Scancom (Cyprus) Ltd." },
	{  282001, "Geocell Ltd." },
	{  282002, "Magti GSM Ltd." },
	{  282003, "Iberiatel Ltd." },
	{  282004, "Mobitel Ltd." },
	{  284001, "M-Tel GSM BG" },
	{  284005, "Globul" },
	{  286001, "Turkcell" },
	{  286002, "Telsim GSM" },
	{  286003, "Aria" },
	{  286004, "Aycell" },
	{  288001, "Faroese Telecom - GSM" },
	{  288002, "Kall GSM" },
	{  290001, "Tele Greenland" },
	{  292001, "SMT - San Marino Telecom" },
	{  293040, "SI Mobil" },
	{  293041, "Mobitel" },
	{  293064, "T-2 d.o.o." },
	{  293070, "Tusmobil d.o.o." },
	{  294001, "T-Mobile" },
	{  294002, "Cosmofon" },
	{  294003, "Nov Operator" },
	{  295001, "Telecom FL AG" },
	{  295002, "Viag Europlatform AG" },
	{  295005, "Mobilkom (Liechstein) AG" },
	{  295077, "Tele2 AG" },
	{  297003, "MTEL d.o.o. Podgorica" },
	{  302036, "Clearnet" },
	{  302037, "Microcell" },
	{  302062, "Ice Wireless" },
	{  302063, "Aliant Mobility" },
	{  302064, "Bell Mobility" },
	{  302656, "Tbay Mobility" },
	{  302066, "MTS Mobility" },
	{  302067, "CityTel Mobility" },
	{  302068, "Sask Tel Mobility" },
	{  302071, "Globalstar" },
	{  302072, "Rogers Wireless" },
	{  302086, "Telus Mobility" },
	{  308001, "St. Pierre-et-Miquelon Telecom" },
	{  310010, "MCI" },
	{  310012, "Verizon Wireless" },
	{  310013, "ALLTEL Communications Inc" },
	{  310016, "Cricket Communications" },
	{  310017, "North Sight Communications Inc" },
	{  310020, "Union Telephone Company" },
	{  310030, "Centennial Communications" },
	{  310035, "ETEX Communications" },
	{  310040, "MTA Communications dba MTA Wireless" },
	{  310050, "ACS Wireless Inc" },
	{  310060, "Consolidated Telcom" },
	{  310070, "Cingular Wireless" },
	{  310080, "Corr Wireless Communications LLC" },
	{  310090, "Cingular Wireless" },
	{  310100, "New Mexico RSA 4 East Ltd. Partnership" },
	{  310110, "Pacific Telecom Inc" },
	{  310130, "Carolina West Wireless" },
	{  310140, "GTA Wireless LLC" },
	{  310150, "Cingular Wireless" },
	{  310160, "T-Mobile USA" },
	{  310170, "Cingular Wireless" },
	{  310180, "West Central Wireless" },
	{  310190, "Alaska Wireless Communications LLC" },
	{  310200, "T-Mobile USA" },
	{  310210, "T-Mobile USA" },
	{  310220, "T-Mobile USA" },
	{  310230, "T-Mobile USA" },
	{  310240, "T-Mobile USA" },
	{  310250, "T-Mobile USA" },
	{  310260, "T-Mobile USA" },
	{  310270, "T-Mobile USA" },
	{  310280, "Contennial Puerto Rio License Corp." },
	{  310290, "Nep Cellcorp Inc." },
	{  310300, "Blanca Telephone Company" },
	{  310310, "T-Mobile USA" },
	{  310320, "Smith Bagley Inc, dba Cellular One" },
	{  310330, "Oklahoma Independent RSA5 Partnership" },
	{  310340, "High Plains Midwest LLC, dba Wetlink Communications" },
	{  310350, "Mohave Cellular L.P." },
	{  310360, "Cellular Network Partnership dba Pioneer Cellular" },
	{  310370, "Guamcell Cellular and Paging" },
	{  310380, "New Cingular Wireless PCS, LLC" },
	{  310390, "TX-11 Acquistion LLC" },
	{  310400, "Wave Runner LLC" },
	{  310410, "Cingular Wireless" },
	{  310420, "Cincinnati Bell Wireless LLC" },
	{  310430, "Alaska Digitel LLC" },
	{  310440, "United Wireless Inc" },
	{  310450, "North East Cellular Inc." },
	{  310460, "TMP Corporation" },
	{  310470, "INTELOS Communications" },
	{  310480, "Choice Phone LLC" },
	{  310490, "T-Mobile USA" },
	{  310500, "Public Service Cellular, Inc." },
	{  310510, "Airtel Wireless LLC" },
	{  310520, "VeriSign" },
	{  310530, "Iowa Wireless Services LLC" },
	{  310540, "Oklahoma Western Telephone Company" },
	{  310550, "Wireless Solutions International" },
	{  310560, "Cingular Wireless" },
	{  310570, "MTPCS LLC" },
	{  310580, "Inland Cellular Telephone Company" },
	{  310590, "Western Wireless Corporation" },
	{  310600, "New Cell Inc. dba Cellcom" },
	{  310610, "Elkhart Telephone Co. Inc. dba Epic Touch Co." },
	{  310620, "Coleman County Telecommunications Inc. (Trans Texas PCS)" },
	{  310630, "Texas-10 LLC" },
	{  310640, "Airadigm Communications" },
	{  310650, "Jasper Wireless Inc." },
	{  310660, "T-Mobile USA" },
	{  310670, "AT&T Mobility Vanguard Services" },
	{  310680, "Cingular Wireless" },
	{  310690, "Keystane Wireless LLC" },
	{  310700, "Cross Valiant Cellular Partnership" },
	{  310710, "Arctic Slope Telephone Association Cooperative" },
	{  310720, "Wireless Solutions International Inc." },
	{  310730, "Sea Mobile" },
	{  310740, "Convey Communications Inc" },
	{  310750, "East Kentucky Network LLC dba Appalachian Wireless" },
	{  310760, "Panhandle Telecommunications Systems Inc." },
	{  310770, "Iowa Wireless Services LLC dba I Wireless" },
	{  310780, "Connect Net Inc" },
	{  310790, "PinPoint Communications Inc." },
	{  310800, "T-Mobile USA" },
	{  310820, "South Canaan Cellular Communications Co. LP" },
	{  310830, "Caprock Cellular Ltd. Partnership" },
	{  310850, "Aeris Communications, Inc." },
	{  310860, "TX RSA 15B2, LP dba Five Star Wireless" },
	{  310870, "Kaplan Telephone Company Inc." },
	{  310880, "Advantage Cellular Systems, Inc." },
	{  310890, "Rural Cellular Corporation" },
	{  310910, "ALLTEL Communications Inc" },
	{  310930, "Copper Valley Wireless" },
	{  310940, "Iris Wireless LLC" },
	{  310950, "Texas RSA 1 dba XIT Cellular" },
	{  310960, "UBET Wireless" },
	{  310970, "Globalstar USA" },
	{  310990, "Worldcall Interconnect" },
	{  311000, "Mid-Tex Cellular Ltd." },
	{  311010, "Chariton Valley Communications Corp., Inc." },
	{  311020, "Missouri RSA No. 5 Partnership" },
	{  311030, "Indigo Wireless, Inc." },
	{  311040, "Commnet Wireless, LLC" },
	{  311050, "Thumb Cellular Limited Partnership" },
	{  311060, "Space Data Corporation" },
	{  311080, "Pine Telephone Company dba Pine Cellular" },
	{  311090, "Siouxland PCS" },
	{  311100, "Nex-Tech Wireless LLC" },
	{  311110, "Alltel Communications Inc" },
	{  311120, "Choice Phone LLC" },
	{  311140, "MBO Wireless Inc./Cross Telephone Company" },
	{  311150, "Wilkes Cellular Inc." },
	{  311170, "PetroCom LLC" },
	{  311180, "Cingular Wireless, Licensee Pacific Telesis Mobile Services, LLC" },
	{  311190, "Cellular Properties Inc." },
	{  311200, "ARINC" },
	{  311230, "Cellular South Inc." },
	{  311240, "Cordova Wireless Communications Inc" },
	{  311250, "Wave Runner LLC" },
	{  311260, "SLO Cellular Inc. dba CellularOne of San Luis Obispo" },
	{  311270, "Alltel Communications Inc." },
	{  311271, "Alltel Communications Inc." },
	{  311272, "Alltel Communications Inc." },
	{  311273, "Alltel Communications Inc." },
	{  311274, "Alltel Communications Inc." },
	{  311275, "Alltel Communications Inc." },
	{  311276, "Alltel Communications Inc." },
	{  311277, "Alltel Communications Inc." },
	{  311278, "Alltel Communications Inc." },
	{  311279, "Alltel Communications Inc." },
	{  311280, "Verizon Wireless" },
	{  311281, "Verizon Wireless" },
	{  311282, "Verizon Wireless" },
	{  311283, "Verizon Wireless" },
	{  311284, "Verizon Wireless" },
	{  311285, "Verizon Wireless" },
	{  311286, "Verizon Wireless" },
	{  311287, "Verizon Wireless" },
	{  311288, "Verizon Wireless" },
	{  311289, "Verizon Wireless" },
	{  311290, "Pinpoint Wireless Inc." },
	{  311300, "Rutal Cellular Corporation" },
	{  311310, "Leaco Rural Telephone Company Inc" },
	{  311320, "Commnet Wireless LLC" },
	{  311330, "Bug Tussel Wireless LLC" },
	{  311340, "Illinois Valley Cellular" },
	{  311350, "Sagebrush Cellular Inc" },
	{  311360, "Stelera Wireless LLC" },
	{  311370, "GCI Communications Corp." },
	{  311380, "GreenFly LLC" },
	{  311390, "ALLTEL Communications Inc" },
	{  311410, "Iowa RSA No.2 Ltd Partnership" },
	{  311420, "Northwest Missouri Cellular Limited Partnership" },
	{  311430, "RSA 1 Limited Partnership dba Cellular 29 Plus" },
	{  311440, "Bluegrass Cellular LLC" },
	{  311450, "Panhandle Telecommunication Systems Inc." },
	{  311460, "Fisher Wireless Services Inc" },
	{  311470, "Vitelcom Cellular Inc dba Innovative Wireless" },
	{  311480, "Verizon Wireless" },
	{  311481, "Verizon Wireless" },
	{  311482, "Verizon Wireless" },
	{  311483, "Verizon Wireless" },
	{  311484, "Verizon Wireless" },
	{  311485, "Verizon Wireless" },
	{  311486, "Verizon Wireless" },
	{  311487, "Verizon Wireless" },
	{  311488, "Verizon Wireless" },
	{  311489, "Verizon Wireless" },
	{  311490, "Strata8 Networks Inc/Wirefree Partners LLC" },
	{  311500, "CTC Telecom Inc" },
	{  311510, "Benton-Linn Wireless" },
	{  311520, "Crossroads Wireless Inc" },
	{  311530, "Wireless Communications Venture" },
	{  311540, "Keystone Wireless Inc" },
	{  311550, "Commnet Midwest LLC" },
	{  311560, "OTZ Communications Inc" },
	{  311570, "Bend Cable Communications LLC" },
	{  311580, "United States Cellular" },
	{  311590, "California RSA No3 Ltd Partnership" },
	{  311600, "Cox TMI Wireless LLC" },
	{  311610, "North Dakota Network Co." },
	{  316010, "Nextel Communications Inc" },
	{  316011, "Southern Communications Services Inc." },
	{  334020, "Telcel" },
	{  338020, "Cable & Wireless Jamaica Ltd." },
	{  338050, "Mossel (Jamaica) Ltd." },
	{  340001, "Orange Caraibe Mobiles" },
	{  340002, "Outremer Telecom" },
	{  340003, "Saint Martin et Saint Barthelemy Telcell Sarl" },
	{  340020, "Bouygues Telecom Caraibe" },
	{  342600, "Cable & Wireless (Barbados) Ltd." },
	{  342820, "Sunbeach Communications" },
	{  344030, "APUA PCS" },
	{  344920, "Cable & Wireless (Antigua)" },
	{  344930, "AT&T Wireless (Antigua)" },
	{  346140, "Cable & Wireless (Cayman)" },
	{  348170, "Cable & Wireless (BVI) Ltd" },
	{  348370, "BVI Cable TV Ltd" },
	{  348570, "Caribbean Cellular Telephone Ltd." },
	{  348770, "Digicel (BVI) Ltd" },
	{  362051, "TELCELL GSM" },
	{  362069, "CT GSM" },
	{  362091, "SETEL GSM" },
	{  363001, "Setar GSM" },
	{  365010, "Weblinks Limited" },
	{  368001, "ETECSA" },
	{  370001, "Orange Dominicana, S.A." },
	{  370002, "Verizon Dominicana S.A." },
	{  370003, "Tricom S.A." },
	{  370004, "CentennialDominicana" },
	{  372001, "Comcel" },
	{  372002, "Digicel" },
	{  372003, "Rectel" },
	{  374012, "TSTT Mobile" },
	{  374130, "Digicel Trinidad and Tobago Ltd." },
	{  374140, "LaqTel Ltd." },
	{  376352, "IslandCom Communications Ltd." },
	{  400001, "Azercell Limited Liability Joint Venture" },
	{  400002, "Bakcell Limited Liabil ity Company" },
	{  400003, "Catel JV" },
	{  400004, "Azerphone LLC" },
	{  401001, "Kar-Tel llc" },
	{  401002, "TSC Kazak Telecom" },
	{  402011, "Bhutan Telecom Ltd" },
	{  402017, "B-Mobile of Bhutan Telecom" },
	{  404000, "Dishnet Wireless Ltd, Madhya Pradesh" },
	{  404001, "Aircell Digilink India Ltd., Haryana" },
	{  404002, "Bharti Airtel Ltd., Punjab" },
	{  404003, "Bharti Airtel Ltd., H.P." },
	{  404004, "Idea Cellular Ltd., Delhi" },
	{  404005, "Fascel Ltd., Gujarat" },
	{  404006, "Bharti Airtel Ltd., Karnataka" },
	{  404007, "Idea Cellular Ltd., Andhra Pradesh" },
	{  404009, "Reliance Telecom Ltd., Assam" },
	{  404010, "Bharti Airtel Ltd., Delhi" },
	{  404011, "Hutchison Essar Mobile Services Ltd, Delhi" },
	{  404012, "Idea Mobile Communications Ltd., Haryana" },
	{  404013, "Hutchinson Essar South Ltd., Andhra Pradesh" },
	{  404014, "Spice Communications PVT Ltd., Punjab" },
	{  404015, "Aircell Digilink India Ltd., UP (East)" },
	{  404016, "Bharti Airtel Ltd, North East" },
	{  404017, "Dishnet Wireless Ltd, West Bengal" },
	{  404018, "Reliance Telecom Ltd., H.P." },
	{  404019, "Idea Mobile Communications Ltd., Kerala" },
	{  404020, "Hutchinson Essar Ltd, Mumbai" },
	{  404021, "BPL Mobile Communications Ltd., Mumbai" },
	{  404022, "Idea Cellular Ltd., Maharashtra" },
	{  404023, "Idea Cellular Ltd, Maharashtra" },
	{  404024, "Idea Cellular Ltd., Gujarat" },
	{  404025, "Dishnet Wireless Ltd, Bihar" },
	{  404027, "Hutchison Essar Cellular Ltd., Maharashtra" },
	{  404029, "Dishnet Wireless Ltd, Assam" },
	{  404030, "Hutchison Telecom East Ltd, Kolkata" },
	{  404031, "Bharti Airtel Ltd., Kolkata" },
	{  404033, "Dishnet Wireless Ltd, North East" },
	{  404034, "BSNL, Haryana" },
	{  404035, "Dishnet Wireless Ltd, Himachal Pradesh" },
	{  404036, "Reliance Telecom Ltd., Bihar" },
	{  404037, "Dishnet Wireless Ltd, J&K" },
	{  404038, "BSNL, Assam" },
	{  404040, "Bharti Airtel Ltd., Chennai" },
	{  404041, "Aircell Cellular Ltd, Chennai" },
	{  404042, "Aircel Ltd., Tamil Nadu" },
	{  404043, "Hutchison Essar Cellular Ltd., Tamil Nadu" },
	{  404044, "Spice Communications PVT Ltd., Karnataka" },
	{  404046, "Hutchison Essar Cellular Ltd., Kerala" },
	{  404048, "Dishnet Wireless Ltd, UP (West)" },
	{  404049, "Bharti Airtel Ltd., Andra Pradesh" },
	{  404050, "Reliance Telecom Ltd., North East" },
	{  404051, "BSNL, H.P." },
	{  404052, "Reliance TelecomLtd., Orissa" },
	{  404053, "BSNL, Punjab" },
	{  404054, "BSNL, UP (West)" },
	{  404055, "BSNL, UP (East)" },
	{  404056, "Idea Mobile Communications Ltd., UP (West)" },
	{  404057, "BSNL, Gujarat" },
	{  404058, "BSNL, Madhya Pradesh" },
	{  404059, "BSNL, Rajasthan" },
	{  404060, "Aircell Digilink India Ltd., Rajasthan" },
	{  404061, "Dishnet Wireless Ltd, Punjab" },
	{  404062, "BSNL, J&K" },
	{  404063, "Dishnet Wireless Ltd, Haryana" },
	{  404064, "BSNL, Chennai" },
	{  404065, "Dishnet Wireless Ltd, UP (East)" },
	{  404066, "BSNL, Maharashtra" },
	{  404067, "Reliance Telecom Ltd., Madhya Pradesh" },
	{  404068, "MTNL, Delhi" },
	{  404069, "MTNL, Mumbai" },
	{  404070, "Bharti Hexacom Ltd, Rajasthan" },
	{  404071, "BSNL, Karnataka" },
	{  404072, "BSNL, Kerala" },
	{  404073, "BSNL, Andhra Pradesh" },
	{  404074, "BSNL, West Bengal" },
	{  404075, "BSNL, Bihar" },
	{  404076, "BSNL, Orissa" },
	{  404077, "BSNL, North East" },
	{  404078, "BTA Cellcom Ltd., Madhya Pradesh" },
	{  404079, "BSNL, Andaman & Nicobar" },
	{  404080, "BSNL, Tamil Nadu" },
	{  404081, "BSNL, Kolkata" },
	{  404082, "Idea Telecommunications Ltd, H.P." },
	{  404083, "Reliable Internet Services Ltd., Kolkata" },
	{  404084, "Hutchinson Essar South Ltd., Chennai" },
	{  404085, "Reliance Telecom Ltd., W.B. & A.N." },
	{  404086, "Hutchinson Essar South Ltd., Karnataka" },
	{  404087, "Idea Telecommunications Ltd, Rajasthan" },
	{  404088, "Hutchison Essar South Ltd, Punjab" },
	{  404089, "Idea Telecommunications Ltd, UP (East)" },
	{  404090, "Bharti Airtel Ltd., Maharashtra" },
	{  404091, "Dishnet Wireless Ltd, Kolkata" },
	{  404092, "Bharti Airtel Ltd., Mumbai" },
	{  404093, "Bharti Airtel Ltd., Madhya Pradesh" },
	{  404094, "Bharti Airtel Ltd., Tamil Nadu" },
	{  404095, "Bharti Airtel Ltd., Kerala" },
	{  404096, "Bharti Airtel Ltd., Haryana" },
	{  404097, "Bharti Airtel Ltd., UP (West)" },
	{  404098, "Bharti Airtel Ltd., Gujarat" },
	{  404099, "Dishnet Wireless Ltd, Kerala" },
	{  405000, "Shyam Telelink Ltd, Rajasthan" },
	{  405001, "Reliance Infocomm Ltd, Andhra Pradesh" },
	{  405003, "Reliance Infocomm Ltd, Bihar" },
	{  405004, "Reliance Infocomm Ltd, Chennai" },
	{  405005, "Reliance Infocomm Ltd, Delhi" },
	{  405006, "Reliance Infocomm Ltd, Gujarat" },
	{  405007, "Reliance Infocomm Ltd, Haryana" },
	{  405008, "Reliance Infocomm Ltd, Himachal Pradesh" },
	{  405009, "Reliance Infocomm Ltd, J&K" },
	{  405010, "Reliance Infocomm Ltd, Karnataka" },
	{  405011, "Reliance Infocomm Ltd, Kerala" },
	{  405012, "Reliance Infocomm Ltd, Kolkata" },
	{  405013, "Reliance Infocomm Ltd, Maharashtra" },
	{  405014, "Reliance Infocomm Ltd, Madhya Pradesh" },
	{  405015, "Reliance Infocomm Ltd, Mumbai" },
	{  405017, "Reliance Infocomm Ltd, Orissa" },
	{  405018, "Reliance Infocomm Ltd, Punjab" },
	{  405020, "Reliance Infocomm Ltd, Tamilnadu" },
	{  405021, "Reliance Infocomm Ltd, UP (East)" },
	{  405022, "Reliance Infocomm Ltd, UP (West)" },
	{  405023, "Reliance Infocomm Ltd, West bengal" },
	{  405025, "Tata Teleservices Ltd, Andhra Pradesh" },
	{  405027, "Tata Teleservices Ltd, Bihar" },
	{  405028, "Tata Teleservices Ltd, Chennai" },
	{  405029, "Tata Teleservices Ltd, Delhi" },
	{  405030, "Tata Teleservices Ltd, Gujarat" },
	{  405031, "Tata Teleservices Ltd, Haryana" },
	{  405032, "Tata Teleservices Ltd, Himachal Pradesh" },
	{  405034, "Tata Teleservices Ltd, Karnataka" },
	{  405035, "Tata Teleservices Ltd, Kerala" },
	{  405036, "Tata Teleservices Ltd, Kolkata" },
	{  405037, "Tata Teleservices Ltd, Maharashtra" },
	{  405038, "Tata Teleservices Ltd, Madhya Pradesh" },
	{  405039, "Tata Teleservices Ltd, Mumbai" },
	{  405041, "Tata Teleservices Ltd, Orissa" },
	{  405042, "Tata Teleservices Ltd, Punjab" },
	{  405043, "Tata Teleservices Ltd, Rajasthan" },
	{  405044, "Tata Teleservices Ltd, Tamilnadu" },
	{  405045, "Tata Teleservices Ltd, UP (East)" },
	{  405046, "Tata Teleservices Ltd, UP (West)" },
	{  405047, "Tata Teleservices Ltd, West Bengal" },
	{  405052, "Bharti Airtel Ltd, Bihar" },
	{  405053, "Bharti Airtel Ltd, Orissa" },
	{  405054, "Bharti Airtel Ltd, UP (East)" },
	{  405055, "Bharti Airtel Ltd, J&K" },
	{  405056, "Bharti Airtel Ltd, Assam" },
	{  405066, "Hutchison Essar South Ltd, UP (West)" },
	{  405067, "Hutchison Essar South Ltd, Orissa" },
	{  405068, "Vodaphone/Hutchison, Madhya Pradesh" },
	{  405070, "Aditya Birla Telecom Ltd, Bihar" },
	{  405071, "Essar Spacetel Ltd, Himachal Pradesh" },
	{  405072, "Essar Spacetel Ltd, North East" },
	{  405073, "Essar Spacetel Ltd, Assam" },
	{  405074, "Essar Spacetel Ltd, J&K" },
	{  405075, "Essar Spacetel Ltd, Bihar" },
	{  405076, "Essar Spacetel Ltd, Orissa" },
	{  405077, "Essar Spacetel Ltd, Maharashtra" },
	{  405080, "Aircell Ltd, Karnataka" },
	{  405081, "Aircell Ltd, Delhi" },
	{  405082, "Aircell Ltd, Andhra Pradesh" },
	{  405083, "Aircell Ltd, Gujarat" },
	{  405084, "Aircell Ltd, Maharashtra" },
	{  405085, "Aircell Ltd, Mumbai" },
	{  405086, "Aircell Ltd, Rajasthan" },
	{  410001, "Mobilink" },
	{  410003, "PAK Telecom Mobile Ltd. (UFONE)" },
	{  410004, "CMPak" },
	{  410006, "Telenor Pakistan" },
	{  410007, "Warid Telecom" },
	{  412001, "AWCC" },
	{  412020, "Roshan" },
	{  412030, "New1" },
	{  412040, "Areeba Afghanistan" },
	{  412088, "Afghan Telecom" },
	{  413002, "MTN Network Ltd." },
	{  413003, "Celtel Lanka Ltd." },
	{  414001, "Myanmar Post and Telecommunication" },
	{  415005, "Ogero Telecom" },
	{  415032, "Cellis" },
	{  415033, "Cellis" },
	{  415034, "Cellis" },
	{  415035, "Cellis" },
	{  415036, "Libancell" },
	{  415037, "Libancell" },
	{  415038, "Libancell" },
	{  415039, "Libancell" },
	{  416001, "Fastlink" },
	{  416002, "Xpress" },
	{  416003, "Umniah" },
	{  416077, "Mobilecom" },
	{  417001, "Syriatel" },
	{  417002, "Spacetel Syria" },
	{  417009, "Syrian Telecom" },
	{  418005, "Asia Cell" },
	{  418020, "Zain Iraq (previously Atheer)" },
	{  418030, "Zain Iraq (previously Iraqna)" },
	{  418040, "Korek Telecom" },
	{  418047, "Iraq Central Cooperative Association for Communication and Transportation" },
	{  418048, "ITC Fanoos" },
	{  418049, "Iraqtel" },
	{  418062, "Itisaluna" },
	{  418070, "Kalimat" },
	{  418080, "Iraqi Telecommunications & Post Company (ITPC)" },
	{  418081, "ITPC (Al-Mazaya)" },
	{  418083, "ITPC (Sader Al-Iraq)" },
	{  418084, "ITPC (Eaamar Albasrah)" },
	{  418085, "ITPC (Anwar Yagotat Alkhalee)" },
	{  418086, "ITPC (Furatfone)" },
	{  418087, "ITPC (Al-Seraj)" },
	{  418088, "ITPC (High Link)" },
	{  418089, "ITPC (Al-Shams)" },
	{  418091, "ITPC (Belad Babel)" },
	{  418092, "ITPC (Al Nakheel)" },
	{  418093, "ITPC (Iraqcell)" },
	{  418094, "ITPC (Shaly)" },
	{  419002, "Mobile Telecommunications Company" },
	{  419003, "Wataniya Telecom" },
	{  419004, "Viva" },
	{  420001, "Saudi Telecom" },
	{  420003, "Etihad Etisalat Company (Mobily)" },
	{  421001, "Yemen Mobile Phone Company" },
	{  421002, "Spacetel Yemen" },
	{  422002, "Oman Mobile Telecommunications Company (Oman Mobile)" },
	{  422003, "Oman Qatari Telecommunications Company (Nawras)" },
	{  422004, "Oman Telecommunications Company (Omantel)" },
	{  424002, "Etisalat" },
	{  425001, "Partner Communications Co. Ltd." },
	{  425002, "Cellcom Israel Ltd." },
	{  425003, "Pelephone Communications Ltd." },
	{  425004, "Globalsim Ltd" },
	{  425006, "Wataniya" },
	{  426001, "BHR Mobile Plus" },
	{  427001, "QATARNET" },
	{  428099, "Mobicom" },
	{  429001, "Nepal Telecommunications" },
	{  432011, "Telecommunication Company of Iran (TCI)" },
	{  432014, "Telecommunication Kish Co. (KIFZO)" },
	{  432019, "Telecommunication Company of Iran (TCI) - Isfahan Celcom" },
	{  434001, "Buztel" },
	{  434002, "Uzmacom" },
	{  434004, "Daewoo Unitel" },
	{  434005, "Coscom" },
	{  434007, "Uzdunrobita" },
	{  436001, "JC Somoncom" },
	{  436002, "CJSC Indigo Tajikistan" },
	{  436003, "TT mobile" },
	{  436004, "Josa Babilon-T" },
	{  436005, "CTJTHSC Tajik-tel" },
	{  437001, "Bitel GSM" },
	{  438001, "Barash Communication Technologies (BCTI)" },
	{  438002, "TM-Cell" },
	{  440001, "NTT DoCoMo, Inc." },
	{  440002, "NTT DoCoMo Kansai, Inc." },
	{  440003, "NTT DoCoMo Hokuriku, Inc." },
	{  440004, "Vodafone" },
	{  440006, "Vodafone" },
	{  440007, "KDDI Corporation" },
	{  440008, "KDDI Corporation" },
	{  440009, "NTT DoCoMo Kansai Inc." },
	{  440010, "NTT DoCoMo Kansai Inc." },
	{  440011, "NTT DoCoMo Tokai Inc." },
	{  440012, "NTT DoCoMo Inc." },
	{  440013, "NTT DoCoMo Inc." },
	{  440014, "NTT DoCoMo Tohoku Inc." },
	{  440015, "NTT DoCoMo Inc." },
	{  440016, "NTT DoCoMo Inc." },
	{  440017, "NTT DoCoMo Inc." },
	{  440018, "NTT DoCoMo Tokai Inc." },
	{  440019, "NTT DoCoMo Hokkaido" },
	{  440020, "NTT DoCoMo Hokuriku Inc." },
	{  440021, "NTT DoCoMo Inc." },
	{  440022, "NTT DoCoMo Kansai Inc." },
	{  440023, "NTT DoCoMo Tokai Inc." },
	{  440024, "NTT DoCoMo Chugoku Inc." },
	{  440025, "NTT DoCoMo Hokkaido Inc." },
	{  440026, "NTT DoCoMo Kyushu Inc." },
	{  440027, "NTT DoCoMoTohoku Inc." },
	{  440028, "NTT DoCoMo Shikoku Inc." },
	{  440029, "NTT DoCoMo Inc." },
	{  440030, "NTT DoCoMo Inc." },
	{  440031, "NTT DoCoMo Kansai Inc." },
	{  440032, "NTT DoCoMo Inc." },
	{  440033, "NTT DoCoMo Tokai Inc." },
	{  440034, "NTT DoCoMo Kyushu Inc." },
	{  440035, "NTT DoCoMo Kansai Inc." },
	{  440036, "NTT DoCoMo Inc." },
	{  440037, "NTT DoCoMo Inc." },
	{  440038, "NTT DoCoMo Inc." },
	{  440039, "NTT DoCoMo Inc." },
	{  440040, "Vodafone" },
	{  440041, "Vodafone" },
	{  440042, "Vodafone" },
	{  440043, "Vodafone" },
	{  440044, "Vodafone" },
	{  440045, "Vodafone" },
	{  440046, "Vodafone" },
	{  440047, "Vodafone" },
	{  440048, "Vodafone" },
	{  440049, "NTT DoCoMo Inc." },
	{  440050, "KDDI Corporation" },
	{  440051, "KDDI Corporation" },
	{  440052, "KDDI Corporation" },
	{  440053, "KDDI Corporation" },
	{  440054, "KDDI Corporation" },
	{  440055, "KDDI Corporation" },
	{  440056, "KDDI Corporation" },
	{  440058, "NTT DoCoMo Kansai Inc." },
	{  440060, "NTT DoCoMo Kansai Inc." },
	{  440061, "NTT DoCoMo Chugoku Inc." },
	{  440062, "NTT DoCoMo Kyushu Inc." },
	{  440063, "NTT DoCoMo Inc." },
	{  440064, "NTT DoCoMo Inc." },
	{  440065, "NTT DoCoMo Shikoku Inc." },
	{  440066, "NTT DoCoMo Inc." },
	{  440067, "NTT DoCoMo Tohoku Inc." },
	{  440068, "NTT DoCoMo Kyushu Inc." },
	{  440069, "NTT DoCoMo Inc." },
	{  440070, "KDDI Corporation" },
	{  440071, "KDDI Corporation" },
	{  440072, "KDDI Corporation" },
	{  440073, "KDDI Corporation" },
	{  440074, "KDDI Corporation" },
	{  440075, "KDDI Corporation" },
	{  440076, "KDDI Corporation" },
	{  440077, "KDDI Corporation" },
	{  440078, "Okinawa Cellular Telephone" },
	{  440079, "KDDI Corporation" },
	{  440080, "TU-KA Cellular Tokyo Inc." },
	{  440081, "TU-KA Cellular Tokyo Inc." },
	{  440082, "TU-KA Phone Kansai Inc." },
	{  440083, "TU-KA Cellular Tokai Inc." },
	{  440084, "TU-KA Phone Kansai Inc." },
	{  440085, "TU-KA Cellular Tokai Inc." },
	{  440086, "TU-KA Cellular Tokyo Inc." },
	{  440087, "NTT DoCoMo Chugoku Inc." },
	{  440088, "KDDI Corporation" },
	{  440089, "KDDI Corporation" },
	{  440090, "Vodafone" },
	{  440092, "Vodafone" },
	{  440093, "Vodafone" },
	{  440094, "Vodafone" },
	{  440095, "Vodafone" },
	{  440096, "Vodafone" },
	{  440097, "Vodafone" },
	{  440098, "Vodafone" },
	{  440099, "NTT DoCoMo Inc." },
	{  441040, "NTT DoCoMo Inc." },
	{  441041, "NTT DoCoMo Inc." },
	{  441042, "NTT DoCoMo Inc." },
	{  441043, "NTT DoCoMo Kansai Inc." },
	{  441044, "NTT DoCoMo Chugoku Inc." },
	{  441045, "NTT DoCoMo Shikoku Inc." },
	{  441050, "TU-KA Cellular Tokyo Inc." },
	{  441051, "TU-KA Phone Kansai Inc." },
	{  441061, "Vodafone" },
	{  441062, "Vodafone" },
	{  441063, "Vodafone" },
	{  441064, "Vodafone" },
	{  441065, "Vodafone" },
	{  441070, "KDDI Corporation" },
	{  441090, "NTT DoCoMo Inc." },
	{  441091, "NTT DoCoMo Inc." },
	{  441092, "NTT DoCoMo Inc." },
	{  441093, "NTT DoCoMo Hokkaido Inc." },
	{  441094, "NTT DoCoMo Tohoku Inc." },
	{  441098, "NTT DoCoMo Kyushu Inc." },
	{  441099, "NTT DoCoMo Kyushu Inc." },
	{  450002, "KT Freetel" },
	{  450003, "SK Telecom" },
	{  452001, "Mobifone" },
	{  452002, "Vinaphone" },
	{  452004, "Viettel" },
	{  452007, "Beeline VN/GTEL Mobile JSC" },
	{  454000, "GSM900/HKCSL" },
	{  454001, "MVNO/CITIC" },
	{  454002, "3G Radio System/HKCSL3G" },
	{  454003, "3G Radio System/Hutchison 3G" },
	{  454004, "GSM900/GSM1800/Hutchison" },
	{  454005, "CDMA/Hutchison" },
	{  454006, "GSM900/SmarTone" },
	{  454007, "MVNO/China Unicom International Ltd." },
	{  454008, "MVNO/Trident" },
	{  454009, "MVNO/China Motion Telecom (HK) Ltd." },
	{  454010, "GSM1800New World PCS Ltd." },
	{  454011, "MVNO/CHKTL" },
	{  454012, "GSM1800/Peoples Telephone Company Ltd." },
	{  454015, "3G Radio System/SMT3G" },
	{  454016, "GSM1800/Mandarin Communications Ltd." },
	{  454018, "GSM7800/Hong Kong CSL Ltd." },
	{  454019, "3G Radio System/Sunday3G" },
	{  455000, "Smartone Mobile Communications (Macao) Ltd." },
	{  455001, "CTM GSM" },
	{  455003, "Hutchison Telecom" },
	{  456001, "Mobitel (Cam GSM)" },
	{  456002, "Hello" },
	{  456003, "S Telecom (CDMA)" },
	{  456004, "Cadcomms" },
	{  456005, "Starcell" },
	{  456006, "Smart" },
	{  456008, "Viettel" },
	{  456018, "Mfone" },
	{  457001, "Lao Telecommunications" },
	{  457002, "ETL Mobile" },
	{  457008, "Millicom" },
	{  460000, "China Mobile" },
	{  460001, "China Unicom" },
	{  460003, "China Unicom CDMA" },
	{  460004, "China Satellite Global Star Network" },
	{  470001, "GramenPhone" },
	{  470002, "Aktel" },
	{  470003, "Mobile 2000" },
	{  472001, "DhiMobile" },
	{  502010, "DIGI Telecommunications" },
	{  502012, "Malaysian Mobile Services Sdn Bhd" },
	{  502013, "Celcom (Malaysia) Berhad" },
	{  502014, "Telekom Malaysia Berhad" },
	{  502016, "DIGI Telecommunications" },
	{  502017, "Malaysian Mobile Services Sdn Bhd" },
	{  502018, "U Mobile Sdn. Bhd." },
	{  502019, "CelCom (Malaysia) Berhad" },
	{  502020, "Electcoms Wireless Sdn Bhd" },
	{  505001, "Telstra Corporation Ltd." },
	{  505002, "Optus Mobile Pty. Ltd." },
	{  505003, "Vodafone Network Pty. Ltd." },
	{  505004, "Department of Defence" },
	{  505005, "The Ozitel Network Pty. Ltd." },
	{  505006, "Hutchison 3G Australia Pty. Ltd." },
	{  505007, "Vodafone Network Pty. Ltd." },
	{  505008, "One.Tel GSM 1800 Pty. Ltd." },
	{  505009, "Airnet Commercial Australia Ltd." },
	{  505010, "Norfolk Telecom" },
	{  505011, "Telstra Corporation Ltd." },
	{  505012, "Hutchison Telecommunications (Australia) Pty. Ltd." },
	{  505013, "Railcorp" },
	{  505014, "AAPT Ltd." },
	{  505015, "3GIS Pty Ltd. (Telstra & Hutchison 3G)" },
	{  505024, "Advanced Communications Technologies Pty. Ltd." },
	{  505071, "Telstra Corporation Ltd." },
	{  505072, "Telstra Corporation Ltd." },
	{  505088, "Localstar Holding Pty. Ltd." },
	{  505090, "Optus Ltd." },
	{  505099, "One.Tel GSM 1800 Pty. Ltd." },
	{  510000, "PSN" },
	{  510001, "Satelindo" },
	{  510008, "Natrindo (Lippo Telecom)" },
	{  510010, "Telkomsel" },
	{  510011, "Excelcomindo" },
	{  510021, "Indosat - M3" },
	{  510028, "Komselindo" },
	{  515001, "Islacom" },
	{  515002, "Globe Telecom" },
	{  515003, "Smart Communications" },
	{  515005, "Digitel" },
	{  520000, "CAT CDMA" },
	{  520001, "AIS GSM" },
	{  520015, "ACT Mobile" },
	{  525001, "SingTel ST GSM900" },
	{  525002, "SingTel ST GSM1800" },
	{  525003, "MobileOne" },
	{  525005, "Starhub" },
	{  525012, "Digital Trunked Radio Network" },
	{  528011, "DST Com" },
	{  530000, "Reserved for AMPS MIN based IMSI's" },
	{  530001, "Vodafone New Zealand GSM Network" },
	{  530002, "Teleom New Zealand CDMA Network" },
	{  530003, "Woosh Wireless - CDMA Network" },
	{  530004, "TelstraClear - GSM Network" },
	{  530005, "Telecom New Zealand - UMTS Ntework" },
	{  530024, "NZ Communications - UMTS Network" },
	{  537001, "Bmobile" },
	{  537002, "Greencom" },
	{  537003, "Digicel Ltd" },
	{  539001, "Tonga Communications Corporation" },
	{  539043, "Digicel" },
	{  539088, "Digicel (Tonga) Ltd" },
	{  541001, "SMILE" },
	{  541005, "Digicel Vanuatu" },
	{  542001, "Vodafone (Fiji) Ltd" },
	{  542002, "Digicel (Fiji) Ltd" },
	{  542003, "Telecom Fiji Ltd (CDMA)" },
	{  546001, "OPT Mobilis" },
	{  547010, "Mara Telecom" },
	{  547020, "Tikiphone" },
	{  548001, "Telecom Cook" },
	{  549001, "Telecom Samoa Cellular Ltd." },
	{  549027, "GoMobile SamoaTel Ltd" },
	{  550001, "FSM Telecom" },
	{  552001, "Palau National Communications Corp. (a.k.a. PNCC)" },
	{  602001, "Mobinil" },
	{  602002, "Vodafone" },
	{  602003, "Etisalat" },
	{  603001, "Algerie Telecom" },
	{  603002, "Orascom Telecom Algerie" },
	{  604000, "Meditelecom (GSM)" },
	{  604001, "Ittissalat Al Maghrid" },
	{  605002, "Tunisie Telecom" },
	{  605003, "Orascom Telecom" },
	{  607001, "Gamcel" },
	{  607002, "Africell" },
	{  607003, "Comium Services Ltd" },
	{  608001, "Sonatel" },
	{  608002, "Sentel GSM" },
	{  609001, "Mattel S.A." },
	{  609002, "Chinguitel S.A." },
	{  609010, "Mauritel Mobiles" },
	{  610001, "Malitel" },
	{  611001, "Orange Guinee" },
	{  611002, "Sotelgui" },
	{  611005, "Cellcom Guinee SA" },
	{  612002, "Atlantique Cellulaire" },
	{  612003, "Orange Cote d'Ivoire" },
	{  612004, "Comium Cote d'Ivoire" },
	{  612005, "Loteny Telecom" },
	{  612006, "Oricel Cote d'Ivoire" },
	{  612007, "Aircomm Cote d'Ivoire" },
	{  613002, "Celtel" },
	{  613003, "Telecel" },
	{  614001, "Sahel.Com" },
	{  614002, "Celtel" },
	{  614003, "Telecel" },
	{  615001, "Togo Telecom" },
	{  616001, "Libercom" },
	{  616002, "Telecel" },
	{  616003, "Spacetel Benin" },
	{  617001, "Cellplus" },
	{  617002, "Mahanagar Telephone (Mauritius) Ltd." },
	{  617010, "Emtel" },
	{  618004, "Comium Liberia" },
	{  619001, "Celtel" },
	{  619002, "Millicom" },
	{  619003, "Africell" },
	{  619004, "Comium (Sierra Leone) Ltd." },
	{  619005, "Lintel (Sierra Leone) Ltd." },
	{  619025, "Mobitel" },
	{  619040, "Datatel (SL) Ltd GSM" },
	{  619050, "Datatel (SL) Ltd CDMA" },
	{  620001, "Spacefon" },
	{  620002, "Ghana Telecom Mobile" },
	{  620003, "Mobitel" },
	{  620004, "Kasapa Telecom Ltd." },
	{  620011, "Netafriques Dot Com Ltd" },
	{  621020, "Econet Wireless Nigeria Ltd." },
	{  621030, "MTN Nigeria Communications" },
	{  621040, "MTEL" },
	{  621050, "Globacom" },
	{  621060, "EMTS" },
	{  622001, "Celtel" },
	{  622002, "Tchad Mobile" },
	{  623001, "Centrafrique Telecom Plus (CTP)" },
	{  623002, "Telecel Centrafrique (TC)" },
	{  623003, "Celca (Socatel)" },
	{  624001, "Mobile Telephone Networks Cameroon" },
	{  624002, "Orange Cameroun" },
	{  625001, "Cabo Verde Telecom" },
	{  625002, "T+Telecomunicacoes" },
	{  626001, "Companhia Santomese de Telecomunicacoes" },
	{  627001, "Guinea Ecuatorial de Telecomunicaciones Sociedad Anonima" },
	{  628001, "Libertis S.A." },
	{  628002, "Telecel Gabon S.A." },
	{  628003, "Celtel Gabon S.A." },
	{  628004, "USAN Gabon" },
	{  629001, "Celtel" },
	{  629010, "Libertis Telecom" },
	{  630001, "Vodacom Congo RDC sprl" },
	{  630005, "Supercell Sprl" },
	{  630086, "Congo-Chine Telecom s.a.r.l." },
	{  631002, "Unitel" },
	{  632001, "Guinetel S.A." },
	{  632002, "Spacetel Guine-Bissau S.A." },
	{  633001, "Cable & Wireless (Seychelles) Ltd." },
	{  633002, "Mediatech International Ltd." },
	{  633010, "Telecom (Seychelles) Ltd." },
	{  634001, "SD Mobitel" },
	{  634002, "Areeba-Sudan" },
	{  634005, "Network of the World Ltd (NOW)" },
	{  634009, "Privet Network (NEC)" },
	{  635010, "MTN Rwandacell" },
	{  636001, "ETH MTN" },
	{  637030, "Golis Telecommunications Company" },
	{  638001, "Evatis" },
	{  639002, "Safaricom Ltd." },
	{  639003, "Kencell Communications Ltd." },
	{  640002, "MIC (T) Ltd." },
	{  640003, "Zantel" },
	{  640004, "Vodacom (T) Ltd." },
	{  640005, "Celtel (T) Ltd." },
	{  641001, "Celtel Uganda" },
	{  641010, "MTN Uganda Ltd." },
	{  641011, "Uganda Telecom Ltd." },
	{  641014, "House of Integrated Technology and Systems Uganda Ltd" },
	{  641022, "Warid Telecom Uganda Ltd." },
	{  641066, "i-Tel Ltd" },
	{  642001, "Spacetel Burundi" },
	{  642002, "Safaris" },
	{  642003, "Telecel Burundi Company" },
	{  643001, "T.D.M. GSM" },
	{  643004, "VM Sarl" },
	{  645001, "Celtel Zambia Ltd." },
	{  645002, "Telecel Zambia Ltd." },
	{  645003, "Zamtel" },
	{  646001, "Celtel Madagascar (Zain), GSM" },
	{  646002, "Orange Madagascar, GSM" },
	{  646003, "Madamobil, CDMA 2000" },
	{  646004, "Telecom Malagasy Mobile, GSM" },
	{  647000, "Orange La Reunion" },
	{  647002, "Outremer Telecom" },
	{  647010, "Societe Reunionnaise du Radiotelephone" },
	{  648001, "Net One" },
	{  648003, "Telecel" },
	{  648004, "Econet" },
	{  649001, "Mobile Telecommunications Ltd." },
	{  649003, "Powercom Pty Ltd" },
	{  650001, "Telekom Network Ltd." },
	{  650010, "Celtel ltd." },
	{  651001, "Vodacom Lesotho (pty) Ltd." },
	{  651002, "Econet Ezin-cel" },
	{  652001, "Mascom Wireless (Pty) Ltd." },
	{  652002, "Orange Botswana (Pty) Ltd." },
	{  653001, "SPTC" },
	{  653010, "Swazi MTN" },
	{  654001, "HURI - SNPT" },
	{  655001, "Vodacom (Pty) Ltd." },
	{  655002, "Telkom SA Ltd" },
	{  655006, "Sentech (Pty) Ltd." },
	{  655007, "Cell C (Pty) Ltd." },
	{  655010, "Mobile Telephone Networks (MTN) Pty Ltd" },
	{  655011, "SAPS Gauteng" },
	{  655012, "Mobile Telephone Networks (MTN) Pty Ltd" },
	{  655013, "Neotel Pty Ltd" },
	{  655021, "Cape Town Metropolitan Council" },
	{  655030, "Bokamoso Consortium Pty Ltd" },
	{  655031, "Karabo Telecoms (Pty) Ltd." },
	{  655032, "Ilizwi Telecommunications Pty Ltd" },
	{  655033, "Thinta Thinta Telecommunications Pty Ltd" },
	{  655034, "Bokone Telecoms Pty Ltd" },
	{  655035, "Kingdom Communications Pty Ltd" },
	{  655036, "Amatole Telecommunication Pty Ltd" },
	{  702067, "Belize Telecommunications Ltd., GSM 1900" },
	{  702068, "International Telecommunications Ltd. (INTELCO)" },
	{  704001, "Servicios de Comunicaciones Personales Inalambricas, S.A." },
	{  704002, "Comunicaciones Celulares S.A." },
	{  704003, "Telefonica Centroamerica Guatemala S.A." },
	{  706001, "CTE Telecom Personal, S.A. de C.V." },
	{  706002, "Digicel, S.A. de C.V." },
	{  706003, "Telemovil El Salvador, S.A." },
	{  708001, "Megatel" },
	{  708002, "Celtel" },
	{  708040, "Digicel Honduras" },
	{  710021, "Empresa Nicaraguense de Telecomunicaciones, S.A. (ENITEL)" },
	{  710073, "Servicios de Comunicaciones, S.A. (SERCOM)" },
	{  712001, "Instituto Costarricense de Electricidad - ICE" },
	{  714001, "Cable & Wireless Panama S.A." },
	{  714002, "BSC de Panama S.A." },
	{  714003, "Claro Panama, S.A." },
	{  714004, "Digicel (Panama), S.A." },
	{  716010, "TIM Peru" },
	{  722010, "Compania de Radiocomunicaciones Moviles S.A." },
	{  722020, "Nextel Argentina srl" },
	{  722070, "Telefonica Comunicaciones Personales S.A." },
	{  722310, "CTI PCS S.A." },
	{  722320, "Compania de Telefonos del Interior Norte S.A." },
	{  722330, "Compania de Telefonos del Interior S.A." },
	{  722341, "Telecom Personal S.A." },
	{  724000, "Telet" },
	{  724001, "CRT Cellular" },
	{  724002, "Global Telecom" },
	{  724003, "CTMR Cel" },
	{  724004, "BCP" },
	{  724005, "Telesc Cel" },
	{  724006, "Tess" },
	{  724007, "Sercontel Cel" },
	{  724008, "Maxitel MG" },
	{  724009, "Telepar Cel" },
	{  724010, "ATL Algar" },
	{  724011, "Telems Cel" },
	{  724012, "Americel" },
	{  724013, "Telesp Cel" },
	{  724014, "Maxitel BA" },
	{  724015, "CTBC Cel" },
	{  724016, "BSE" },
	{  724017, "Ceterp Cel" },
	{  724018, "Norte Brasil Tel" },
	{  724019, "Telemig Cel" },
	{  724021, "Telerj Cel" },
	{  724023, "Telest Cel" },
	{  724025, "Telebrasilia Cel" },
	{  724027, "Telegoias Cel" },
	{  724029, "Telemat Cel" },
	{  724031, "Teleacre Cel" },
	{  724033, "Teleron Cel" },
	{  724035, "Telebahia Cel" },
	{  724037, "Telergipe Cel" },
	{  724039, "Telasa Cel" },
	{  724041, "Telpe Cel" },
	{  724043, "Telepisa Cel" },
	{  724045, "Telpa Cel" },
	{  724047, "Telern Cel" },
	{  724048, "Teleceara Cel" },
	{  724051, "Telma Cel" },
	{  724053, "Telepara Cel" },
	{  724055, "Teleamazon Cel" },
	{  724057, "Teleamapa Cel" },
	{  724059, "Telaima Cel" },
	{  730001, "Entel Telefonica Movil" },
	{  730002, "Telefonica Movil" },
	{  730003, "Smartcom" },
	{  730004, "Centennial Cayman Corp. Chile S.A." },
	{  730005, "Multikom S.A." },
	{  730010, "Entel" },
	{  732001, "Colombia Telecomunicaciones S.A. - Telecom" },
	{  732002, "Edatel S.A." },
	{  732020, "Emtelsa" },
	{  732099, "Emcali" },
	{  732101, "Comcel S.A. Occel S.A./Celcaribe" },
	{  732102, "Bellsouth Colombia S.A." },
	{  732103, "Colombia Movil S.A." },
	{  732111, "Colombia Movil S.A." },
	{  732123, "Telefonica Moviles Colombia S.A." },
	{  732130, "Avantel" },
	{  734001, "Infonet" },
	{  734002, "Corporacion Digitel" },
	{  734003, "Digicel" },
	{  734004, "Telcel, C.A." },
	{  734006, "Telecomunicaciones Movilnet, C.A." },
	{  736001, "Nuevatel S.A." },
	{  736002, "ENTEL S.A." },
	{  736003, "Telecel S.A." },
	{  738001, "Cel*Star (Guyana) Inc." },
	{  740000, "Otecel S.A. - Bellsouth" },
	{  740001, "Porta GSM" },
	{  740002, "Telecsa S.A." },
	{  744001, "Hola Paraguay S.A." },
	{  744002, "Hutchison Telecom S.A." },
	{  744003, "Compania Privada de Comunicaciones S.A." },
	{  746002, "Telesur" },
	{  746003, "Digicel" },
	{  746004, "Intelsur" },
	{  746005, "Telesur (CDMA)" },
	{  748000, "Ancel - TDMA" },
	{  748001, "Ancel - GSM" },
	{  748003, "Ancel" },
	{  748007, "Movistar" },
	{  748010, "CTI Movil" },
	{  901001, "ICO Global Communications" },
	{  901002, "Sense Communications International AS" },
	{  901003, "Iridium Satellite, LLC (GMSS)" },
	{  901005, "Thuraya RMSS Network" },
	{  901006, "Thuraya Satellite Telecommunications Company" },
	{  901009, "Tele1 Europe" },
	{  901010, "Asia Cellular Satellite (AceS)" },
	{  901011, "Inmarsat Ltd." },
	{  901012, "Maritime Communications Partner AS (MCP network)" },
	{  901013, "Global Networks, Inc." },
	{  901014, "Telenor GSM - services in aircraft" },
	{  901015, "OnAir" },
	{  901016, "Jasper Systems, Inc." },
	{  901017, "Jersey Telecom" },
	{  901018, "Cingular Wireless" },
	{  901019, "Vodaphone Malta Vodafone Group)" },
	{  901020, "Intermatica" },
	{  901021, "Seanet Maritime Communications" },
	{  901022, "MediaLincc Ltd" },
	{  901023, "Ukrainian Radiosystems" },
	{  901024, "Voxbone SA" },
	{  901025, "In & Phone" },
	{  901026, "Telecom Italia" },
	{  901027, "Onair" },
	{  901028, "Vodafone Group" },
	{  901029, "Telenor Connexion AB" },
	{  901088, "Telecommunications for Disaster Relief (TDR) (OCHA)" },
	{  0, NULL }
};

static int proto_e212						= -1;
static int hf_E212_mcc						= -1;
static int hf_E212_mnc						= -1;
static int hf_E212_msin						= -1;

/*
 *	8   7   6   5   4   3   2   1
 *	MCC digit 2     MCC digit 1    octet x
 *	MNC digit 3     MCC digit 3    octet x+1
 *	MNC digit 2     MNC digit 1    octet x+2
 */


int
dissect_e212_mcc_mnc(tvbuff_t *tvb, proto_tree *tree, int offset){

	int			start_offset;	
	guint8		octet;
	guint16		mcc, mnc;
	guint8		mcc1, mcc2, mcc3, mnc1, mnc2, mnc3;

	start_offset = offset;
	/* Mobile country code MCC */
	octet = tvb_get_guint8(tvb,offset);
	mcc1 = octet & 0x0f;
	mcc2 = octet >> 4;
	offset++;
	octet = tvb_get_guint8(tvb,offset);
	mcc3 = octet & 0x0f;
	/* MNC, Mobile network code (octet 3 bits 5 to 8, octet 4)  */
	mnc3 = octet >> 4;
	offset++;
	octet = tvb_get_guint8(tvb,offset);
	mnc1 = octet & 0x0f;
	mnc2 = octet >> 4;

	mcc = 100 * mcc1 + 10 * mcc2 + mcc3;
	mnc = 10 * mnc1 + mnc2;
	if (mnc3 != 0xf) {
		mnc = 10 * mnc + mnc3;
	}
	proto_tree_add_uint(tree, hf_E212_mcc , tvb, start_offset, 2, mcc );
	proto_tree_add_uint_format(tree, hf_E212_mnc , tvb, start_offset + 1, 2, mnc,
				   "Mobile Network Code (MNC): %s (%u)",
				   val_to_str(mcc * 1000 + mnc, mcc_mnc_codes, "Unknown"),
				   mnc);
	offset++;
	return offset;
}

/*
 * Register the protocol with Wireshark.
 *
 * This format is required because a script is used to build the C function
 * that calls all the protocol registration.
 */


void
proto_register_e212(void)
{

/* Setup list of header fields  See Section 1.6.1 for details */
	static hf_register_info hf[] = {
	{ &hf_E212_mcc,
		{ "Mobile Country Code (MCC)","e212.mcc",
		FT_UINT16, BASE_DEC, VALS(E212_codes), 0x0,          
		"Mobile Country Code MCC", HFILL }
	},
	{ &hf_E212_mnc,
		{ "Mobile network code (MNC)","e212.mnc",
		FT_UINT16, BASE_DEC, NULL, 0x0,          
		"Mobile network code", HFILL }
	},
	{ &hf_E212_msin,
      { "Mobile Subscriber Identification Number (MSIN)", "e212.msin",
        FT_STRING, BASE_NONE, NULL, 0,
        "Mobile Subscriber Identification Number(MSIN)", HFILL }},
	};

	/*
	 * Register the protocol name and description
	 */
	proto_e212 = proto_register_protocol(
			"ITU-T E.212 number",
			"E.212",
			"e212");

	/*
	 * Required function calls to register
	 * the header fields and subtrees used.
	 */
	proto_register_field_array(proto_e212, hf, array_length(hf));

}
