/*******************************************************************************
* vls.cpp: vls main file
*-------------------------------------------------------------------------------
* (c)1999-2001 VideoLAN
* $Id: vls.cpp,v 1.5 2001/11/29 17:55:19 bozo Exp $
*
* Authors: Benoit Steiner <benny@via.ecp.fr>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*
*-------------------------------------------------------------------------------
* This file implements the application object and launches the program
*******************************************************************************/


//------------------------------------------------------------------------------
// Preamble
//------------------------------------------------------------------------------
#include "../core/defs.h"

#ifdef GETPROTOBYNAME_IN_NETDB_H
#include <netdb.h>
#elif defined WIN32
#include <winsock2.h>
#endif

#include "config.h"

#include "../core/core.h"

#include "../mpeg/mpeg.h"
#include "../mpeg/ts.h"

#include "program.h"
#include "buffer.h"
#include "output.h"
#include "channel.h"
#include "request.h"
#include "broadcast.h"
#include "input.h"
#include "repository.h"
#include "directory.h"
#include "admin.h"
#include "manager.h"

#include "vls.h"

#include "repository.cpp"



//------------------------------------------------------------------------------
// Constructor
//------------------------------------------------------------------------------
C_Vls::C_Vls(const C_String& strName) : C_Application(strName)
{
  m_pManager = NULL;
  m_pAdministrator = NULL;
}


//------------------------------------------------------------------------------
// Destructor
//------------------------------------------------------------------------------
C_Vls::~C_Vls()
{
}


//------------------------------------------------------------------------------
// Initialization
//------------------------------------------------------------------------------
int C_Vls::OnAppInit()
{
  // Startup message
  printf("%s\n", PGRM_INFO);

  // Check environnement
  int iRc = CheckEnvir();

  if(!iRc)
  {
    LoadBuiltins();
    m_pModuleManager->BrowseDirectory(".");
    m_pModuleManager->BrowseDirectory(PLUGIN_PATH);
  }

  // Init the manager
  if(!iRc)
  {
    m_pManager = new C_Manager(m_hLog, this);
    iRc = m_pManager->Init(); 
  }

  // Init the administrator
  if(!iRc)
  {
    m_pAdministrator = new C_Admin(m_hLog, this);
    iRc = m_pAdministrator->Init();
  }

  return iRc;
}


//------------------------------------------------------------------------------
// Execution
//------------------------------------------------------------------------------
int C_Vls::OnAppRun()
{
  // Run the manager
  int iRc = m_pManager->Run();

  // Run the administrator
  if(!iRc)
    iRc = m_pAdministrator->Run();

  return iRc;
}


//------------------------------------------------------------------------------
// Stop the execution
//------------------------------------------------------------------------------
int C_Vls::OnAppExit()
{
  int iRc = NO_ERR;

  if(m_pAdministrator)
  {
    m_pAdministrator->DisableRequests();
    iRc = m_pAdministrator->Stop();
  }

  return iRc;
}


//------------------------------------------------------------------------------
// Stop the execution
//------------------------------------------------------------------------------
int C_Vls::SafeStop()
{
  int iRc = NO_ERR;

  if(m_pManager)
    iRc = m_pManager->Stop();

  return iRc;
}


//------------------------------------------------------------------------------
// Destruction
//------------------------------------------------------------------------------
int C_Vls::OnAppDestroy()
{
  int iRc = NO_ERR;

  // Destroy the administrator if any
  if(m_pAdministrator)
  {
    Log(m_hLog, LOG_NOTE, "Destroying administrator module");
    iRc = m_pAdministrator->Destroy();
    delete m_pAdministrator;
    m_pAdministrator = NULL;
  }

  // Destroy the manager if any
  if(m_pManager)
  {
    Log(m_hLog, LOG_NOTE, "Destroying manager module");
    iRc |= m_pManager->Destroy();
    delete m_pManager;
    m_pManager = NULL;
  }

  return iRc;
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
C_Answer C_Vls::ForwardRequest(const C_Request& cRequest)
{
  // For now, request only come from the admin and go to the manager: simply
  // forward the request to the manager
  ASSERT(m_pManager);
  return m_pManager->HandleRequest(cRequest);
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
void C_Vls::ForwardEvent(const C_Event& cEvent)
{
  // For now, events only come from the manager and go to the admin: simply
  // forward the request to the admin
  ASSERT(m_pAdministrator);
  m_pAdministrator->HandleEvent(cEvent);
}


//------------------------------------------------------------------------------
//
//------------------------------------------------------------------------------
int C_Vls::CheckEnvir()
{
  int iRc = 0;
  
#ifdef GETPROTOBYNAME_IN_NETDB_H
  if(!getprotobyname("tcp"))
  {
    printf("Unknown protocol: TCP\n");
    iRc = GEN_ERR;
  }

  else if(!getprotobyname("udp"))
  {
    printf("Unknown protocol: UDP\n");
    iRc = GEN_ERR;
  }

#elif defined WIN32
  // TO DO

#endif

  return iRc;
}



//------------------------------------------------------------------------------
// Usage
//------------------------------------------------------------------------------
void C_Vls::Usage(const C_String& strProgName)
{
#define HELP ""                         \
"Usage : %s\n"                          \
"[-c <cfg>]: Config file\n"

  printf(HELP, strProgName.GetString());
}


  
//------------------------------------------------------------------------------
// Get the configuration from the file vls.cfg
//------------------------------------------------------------------------------
/*int C_Vls::GetConfig(int iArgc, char** paArgv, C_String* pstrCfgFile,
                             C_String* pstrLogFile)
{
  int iRc = NO_ERR;
  
  // Parse command line

#ifdef HAVE_GETOPT_H
  // First parse optional args
  int iOptArg;
  opterr = 0;           // Force getopt not to generate an error message for '?'
  while((iOptArg = getopt(iArgc, paArgv, "c:h")) != EOF)
  {
    switch(iOptArg)
    {
      case 'c':
        (*pstrCfgFile) = optarg;
        break;
      case 'h':
      {
        // Display help message
        Usage(*paArgv);
        iRc = GEN_ERR;
        break;
      }
      default:
      {
        printf("Invalid '%c' option or missing parameter", optopt);
        // Display help message
        Usage(*paArgv);
        iRc = GEN_ERR;
      }
    }
  }
    
  // Now parse the mandatory ones
  if(iArgc - optind != 0)
  {
    printf("Invalid %d extra argument(s)", iArgc - optind);
    Usage(*paArgv);
    iRc = GEN_ERR;
  }
  
  // Get additional options from configuration file
  if(!iRc)
  {
    // TO DO ???
  }
  
#elif defined WIN32
  // TO DO

#endif

  return iRc;
}
*/




#include "vls_builtins.cpp"



//------------------------------------------------------------------------------
// Main function
//------------------------------------------------------------------------------
int main(int iArgc, char* paArgv[])
{
  int iRc = NO_ERR;

#ifdef DEBUG
  try
  {
#endif
    C_Vls cApplication("Vls");

    // Init the application
    iRc = cApplication.Init(iArgc, paArgv);
  
    // Start the program unless something went wrong during the initialisation
    if (!iRc)
      iRc = cApplication.Run();

    // Safe stop executed by the main thread
    iRc |= cApplication.SafeStop();

    // Free all ressources used by the program
    iRc |= cApplication.Destroy();
#ifdef DEBUG
  }
  catch(E_Exception e)
  {
    printf("Unhandled Exception : %s\n", e.Dump().GetString());
  }
#endif

  return iRc;
}

