/*****************************************************************************
 * interface.cpp : wxWindows plugin for vlc
 *****************************************************************************
 * Copyright (C) 2000-2001 VideoLAN
 * $Id: interface.cpp,v 1.54 2003/07/29 21:14:10 gbazin Exp $
 *
 * Authors: Gildas Bazin <gbazin@netcourrier.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

/*****************************************************************************
 * Preamble
 *****************************************************************************/
#include <stdlib.h>                                      /* malloc(), free() */
#include <errno.h>                                                 /* ENOMEM */
#include <string.h>                                            /* strerror() */
#include <stdio.h>

#include <vlc/vlc.h>
#include <vlc/aout.h>

#ifdef WIN32                                                 /* mingw32 hack */
#undef Yield
#undef CreateDialog
#endif

/* Let vlc take care of the i18n stuff */
#define WXINTL_NO_GETTEXT_MACRO

#include <wx/wxprec.h>
#include <wx/wx.h>

#include <vlc/intf.h>
#include "stream_control.h"

#include "wxwindows.h"

/* include the toolbar graphics */
#include "bitmaps/file.xpm"
#include "bitmaps/disc.xpm"
#include "bitmaps/net.xpm"
#if 0
#include "bitmaps/sat.xpm"
#endif
#include "bitmaps/play.xpm"
#include "bitmaps/pause.xpm"
#include "bitmaps/stop.xpm"
#include "bitmaps/previous.xpm"
#include "bitmaps/next.xpm"
#include "bitmaps/playlist.xpm"
#include "bitmaps/fast.xpm"
#include "bitmaps/slow.xpm"

#define TOOLBAR_BMP_WIDTH 36
#define TOOLBAR_BMP_HEIGHT 36

/* include the icon graphic */
#include "../../../share/vlc32x32.xpm"

/*****************************************************************************
 * Local class declarations.
 *****************************************************************************/
class wxMenuExt: public wxMenu
{
public:
    /* Constructor */
    wxMenuExt( wxMenu* parentMenu, int id, const wxString& text,
                   const wxString& helpString, wxItemKind kind,
                   char *_psz_var, int _i_object_id, vlc_value_t _val,
                   int _i_val_type );

    virtual ~wxMenuExt() {};

    char *psz_var;
    int  i_val_type;
    int  i_object_id;
    vlc_value_t val;

private:

};

class wxVolCtrl: public wxGauge
{
public:
    /* Constructor */
    wxVolCtrl( intf_thread_t *_p_intf, wxWindow* parent, wxWindowID id );
    virtual ~wxVolCtrl() {};

    void Change( int i_volume );

    void OnChange( wxMouseEvent& event );

private:
    intf_thread_t *p_intf;

    DECLARE_EVENT_TABLE();
};

BEGIN_EVENT_TABLE(wxVolCtrl, wxWindow)
    /* Mouse events */
    EVT_LEFT_DOWN(wxVolCtrl::OnChange)
    EVT_MOTION(wxVolCtrl::OnChange)
END_EVENT_TABLE()

/*****************************************************************************
 * Event Table.
 *****************************************************************************/

/* IDs for the controls and the menu commands */
enum
{
    /* menu items */
    Exit_Event = wxID_HIGHEST,
    OpenFileSimple_Event,
    OpenFile_Event,
    OpenDisc_Event,
    OpenNet_Event,
    OpenSat_Event,
    EjectDisc_Event,

    Playlist_Event,
    Logs_Event,
    FileInfo_Event,

    Prefs_Event,

    SliderScroll_Event,
    StopStream_Event,
    PlayStream_Event,
    PrevStream_Event,
    NextStream_Event,
    SlowStream_Event,
    FastStream_Event,

    /* it is important for the id corresponding to the "About" command to have
     * this standard value as otherwise it won't be handled properly under Mac
     * (where it is special and put into the "Apple" menu) */
    About_Event = wxID_ABOUT
};

BEGIN_EVENT_TABLE(Interface, wxFrame)
    /* Menu events */
    EVT_MENU(Exit_Event, Interface::OnExit)
    EVT_MENU(About_Event, Interface::OnAbout)

    EVT_MENU(Playlist_Event, Interface::OnShowDialog)
    EVT_MENU(Logs_Event, Interface::OnShowDialog)
    EVT_MENU(FileInfo_Event, Interface::OnShowDialog)
    EVT_MENU(Prefs_Event, Interface::OnShowDialog)

    EVT_MENU_OPEN(Interface::OnMenuOpen)

#if defined( __WXMSW__ ) || defined( __WXMAC__ )
    EVT_CONTEXT_MENU(Interface::OnContextMenu2)
#endif
    EVT_RIGHT_UP(Interface::OnContextMenu)

    /* Toolbar events */
    EVT_MENU(OpenFileSimple_Event, Interface::OnShowDialog)
    EVT_MENU(OpenFile_Event, Interface::OnShowDialog)
    EVT_MENU(OpenDisc_Event, Interface::OnShowDialog)
    EVT_MENU(OpenNet_Event, Interface::OnShowDialog)
    EVT_MENU(OpenSat_Event, Interface::OnShowDialog)
    EVT_MENU(StopStream_Event, Interface::OnStopStream)
    EVT_MENU(PlayStream_Event, Interface::OnPlayStream)
    EVT_MENU(PrevStream_Event, Interface::OnPrevStream)
    EVT_MENU(NextStream_Event, Interface::OnNextStream)
    EVT_MENU(SlowStream_Event, Interface::OnSlowStream)
    EVT_MENU(FastStream_Event, Interface::OnFastStream)

    /* Slider events */
    EVT_COMMAND_SCROLL(SliderScroll_Event, Interface::OnSliderUpdate)

END_EVENT_TABLE()

/*****************************************************************************
 * Constructor.
 *****************************************************************************/
Interface::Interface( intf_thread_t *_p_intf ):
    wxFrame( NULL, -1, wxT("VLC media player"),
             wxDefaultPosition, wxDefaultSize, wxDEFAULT_FRAME_STYLE )
{
    /* Initializations */
    p_intf = _p_intf;
    i_old_playing_status = PAUSE_S;

    /* Give our interface a nice little icon */
    SetIcon( wxIcon( vlc_xpm ) );

    /* Create a sizer for the main frame */
    frame_sizer = new wxBoxSizer( wxHORIZONTAL );
    SetSizer( frame_sizer );

    /* Creation of the menu bar */
    CreateOurMenuBar();

    /* Creation of the tool bar */
    CreateOurToolBar();

    /* Creation of the slider sub-window */
    CreateOurSlider();
    frame_sizer->Add( slider_frame, 1, wxGROW, 0 );
    frame_sizer->Hide( slider_frame );

    /* Creation of the status bar
     * Helptext for menu items and toolbar tools will automatically get
     * displayed here. */
    int i_status_width[3] = {-6, -2, -9};
    statusbar = CreateStatusBar( 3 );                            /* 2 fields */
    statusbar->SetStatusWidths( 3, i_status_width );
    statusbar->SetStatusText( wxString::Format(wxT("x%.2f"), 1.0), 1 );

    /* Make sure we've got the right background colour */
    SetBackgroundColour( slider_frame->GetBackgroundColour() );

    /* Layout everything */
    SetAutoLayout( TRUE );
    frame_sizer->Layout();
    frame_sizer->Fit(this);

#if !defined(__WXX11__)
    /* Associate drop targets with the main interface */
    SetDropTarget( new DragAndDrop( p_intf ) );
#endif
}

Interface::~Interface()
{
    if( p_intf->p_sys->p_wxwindow )
    {
        delete p_intf->p_sys->p_wxwindow;
    }
    /* Clean up */
}

/*****************************************************************************
 * Private methods.
 *****************************************************************************/
void Interface::CreateOurMenuBar()
{
#define HELP_FILE  N_("Open a file")
#define HELP_DISC  N_("Open a DVD or (S)VCD")
#define HELP_NET   N_("Open a network stream")
#define HELP_SAT   N_("Open a satellite stream")
#define HELP_EJECT N_("Eject the DVD/CD")
#define HELP_EXIT  N_("Exit this program")

#define HELP_PLAYLIST   N_("Open the playlist")
#define HELP_LOGS       N_("Show the program logs")
#define HELP_FILEINFO       N_("Show information about the file being played")

#define HELP_PREFS N_("Go to the preferences menu")

#define HELP_ABOUT N_("About this program")

    /* Create the "File" menu */
    wxMenu *file_menu = new wxMenu;
    file_menu->Append( OpenFileSimple_Event, wxU(_("Simple &Open ...")),
                       wxU(_(HELP_FILE)) );
    file_menu->Append( OpenFile_Event, wxU(_("Open &File...")),
                       wxU(_(HELP_FILE)) );
    file_menu->Append( OpenDisc_Event, wxU(_("Open &Disc...")),
                       wxU(_(HELP_DISC)) );
    file_menu->Append( OpenNet_Event, wxU(_("Open &Network Stream...")),
                       wxU(_(HELP_NET)) );
#if 0
    file_menu->Append( OpenSat_Event, wxU(_("Open &Satellite Stream...")),
                       wxU(_(HELP_NET)) );
#endif
#if 0
    file_menu->AppendSeparator();
    file_menu->Append( EjectDisc_Event, wxU(_("&Eject Disc")),
                       wxU(_(HELP_EJECT)) );
#endif
    file_menu->AppendSeparator();
    file_menu->Append( Exit_Event, wxU(_("E&xit")), wxU(_(HELP_EXIT)) );

    /* Create the "View" menu */
    wxMenu *view_menu = new wxMenu;
    view_menu->Append( Playlist_Event, wxU(_("&Playlist...")),
                       wxU(_(HELP_PLAYLIST)) );
    view_menu->Append( Logs_Event, wxU(_("&Messages...")), wxU(_(HELP_LOGS)) );
    view_menu->Append( FileInfo_Event, wxU(_("&File info...")),
                       wxU(_(HELP_FILEINFO)) );

    /* Create the "Settings" menu */
    wxMenu *settings_menu = new wxMenu;
    settings_menu->Append( Prefs_Event, wxU(_("&Preferences...")),
                           wxU(_(HELP_PREFS)) );

    /* Create the "Audio" menu */
    p_audio_menu = new wxMenu;
    b_audio_menu = 1;

    /* Create the "Video" menu */
    p_video_menu = new wxMenu;
    b_video_menu = 1;

    /* Create the "Navigation" menu */
    p_navig_menu = new wxMenu;
    b_navig_menu = 1;

    /* Create the "Help" menu */
    wxMenu *help_menu = new wxMenu;
    help_menu->Append( About_Event, wxU(_("&About...")), wxU(_(HELP_ABOUT)) );

    /* Append the freshly created menus to the menu bar... */
    wxMenuBar *menubar = new wxMenuBar( wxMB_DOCKABLE );
    menubar->Append( file_menu, wxU(_("&File")) );
    menubar->Append( view_menu, wxU(_("&View")) );
    menubar->Append( settings_menu, wxU(_("&Settings")) );
    menubar->Append( p_audio_menu, wxU(_("&Audio")) );
    menubar->Append( p_video_menu, wxU(_("&Video")) );
    menubar->Append( p_navig_menu, wxU(_("&Navigation")) );
    menubar->Append( help_menu, wxU(_("&Help")) );

    /* Attach the menu bar to the frame */
    SetMenuBar( menubar );

    /* Intercept all menu events in our custom event handler */
    PushEventHandler( new MenuEvtHandler( p_intf, this ) );

#if !defined(__WXX11__)
    /* Associate drop targets with the menubar */
    menubar->SetDropTarget( new DragAndDrop( p_intf ) );
#endif
}

void Interface::CreateOurToolBar()
{
#define HELP_STOP N_("Stop current playlist item")
#define HELP_PLAY N_("Play current playlist item")
#define HELP_PAUSE N_("Pause current playlist item")
#define HELP_PLO N_("Open playlist")
#define HELP_PLP N_("Previous playlist item")
#define HELP_PLN N_("Next playlist item")
#define HELP_SLOW N_("Play slower")
#define HELP_FAST N_("Play faster")

    wxLogNull LogDummy; /* Hack to suppress annoying log message on the win32
                         * version because we don't include wx.rc */

    wxToolBar *toolbar = CreateToolBar(
        wxTB_HORIZONTAL | wxTB_FLAT | wxTB_DOCKABLE );

    toolbar->SetToolBitmapSize( wxSize(TOOLBAR_BMP_WIDTH,TOOLBAR_BMP_HEIGHT) );

    toolbar->AddTool( OpenFile_Event, wxU(_("File")), wxBitmap( file_xpm ),
                      wxU(_(HELP_FILE)) );
    toolbar->AddTool( OpenDisc_Event, wxU(_("Disc")), wxBitmap( disc_xpm ),
                      wxU(_(HELP_DISC)) );
    toolbar->AddTool( OpenNet_Event, wxU(_("Net")), wxBitmap( net_xpm ),
                      wxU(_(HELP_NET)) );
#if 0
    toolbar->AddTool( OpenSat_Event, wxU(_("Sat")), wxBitmap( sat_xpm ),
                      wxU(_(HELP_SAT)) );
#endif
    toolbar->AddSeparator();
    toolbar->AddTool( StopStream_Event, wxU(_("Stop")), wxBitmap( stop_xpm ),
                      wxU(_(HELP_STOP)) );
    toolbar->AddTool( PlayStream_Event, wxU(_("Play")), wxBitmap( play_xpm ),
                      wxU(_(HELP_PLAY)) );
    toolbar->AddSeparator();
    toolbar->AddTool( Playlist_Event, wxU(_("Playlist")),
                      wxBitmap( playlist_xpm ), wxU(_(HELP_PLO)) );
    toolbar->AddTool( PrevStream_Event, wxU(_("Prev")),
                      wxBitmap( previous_xpm ), wxU(_(HELP_PLP)) );
    toolbar->AddTool( NextStream_Event, wxU(_("Next")), wxBitmap( next_xpm ),
                      wxU(_(HELP_PLN)) );
    toolbar->AddTool( SlowStream_Event, wxU(_("Slower")), wxBitmap( slow_xpm ),
                      wxU(_(HELP_SLOW)) );
    toolbar->AddTool( FastStream_Event, wxU(_("Faster")), wxBitmap( fast_xpm ),
                      wxU(_(HELP_FAST)) );

    toolbar->Realize();

    /* Place the toolbar in a sizer, so we can calculate the width of the
     * toolbar and set this as the minimum for the main frame size. */
    wxBoxSizer *toolbar_sizer = new wxBoxSizer( wxHORIZONTAL );
    toolbar_sizer->Add( toolbar, 0, 0, 0 );
    toolbar_sizer->Layout();

#ifndef WIN32
    frame_sizer->SetMinSize( toolbar_sizer->GetMinSize().GetWidth(), -1 );
#else
    frame_sizer->SetMinSize( toolbar->GetToolSize().GetWidth() *
                             toolbar->GetToolsCount(), -1 );
#endif

#if !defined(__WXX11__)
    /* Associate drop targets with the toolbar */
    toolbar->SetDropTarget( new DragAndDrop( p_intf ) );
#endif
}

void Interface::CreateOurSlider()
{
    /* Create a new frame and sizer containing the slider */
    slider_frame = new wxPanel( this, -1, wxDefaultPosition, wxDefaultSize );
    slider_frame->SetAutoLayout( TRUE );
    wxBoxSizer *frame_sizer =
        new wxBoxSizer( wxHORIZONTAL );

    /* Create static box to surround the slider */
    slider_box = new wxStaticBox( slider_frame, -1, wxT("") );

    /* Create sizer for slider frame */
    wxStaticBoxSizer *slider_sizer =
        new wxStaticBoxSizer( slider_box, wxHORIZONTAL );
    slider_sizer->SetMinSize( -1, 50 );

    /* Create slider */
    slider = new wxSlider( slider_frame, SliderScroll_Event, 0, 0,
                           SLIDER_MAX_POS, wxDefaultPosition, wxDefaultSize );
    slider_sizer->Add( slider, 1, wxEXPAND | wxALL, 5 );


    volctrl = new wxVolCtrl( p_intf, slider_frame, -1 );

    /* Add everything to the frame */
    frame_sizer->Add( slider_sizer, 1, wxEXPAND | wxBOTTOM, 5 );
    frame_sizer->Add( volctrl, 0, wxEXPAND | wxALL, 5 );
    slider_frame->SetSizer( frame_sizer );
    frame_sizer->Layout();
    frame_sizer->SetSizeHints(slider_frame);

    /* Hide the slider by default */
    slider_frame->Hide();
}

/*****************************************************************************
 * Event Handlers.
 *****************************************************************************/
/* Work-around helper for buggy wxGTK */
void RecursiveDestroy( wxMenu *menu )
{
    wxMenuItemList::Node *node = menu->GetMenuItems().GetFirst();
    for( ; node; )
    {
        wxMenuItem *item = node->GetData();
        node = node->GetNext();

        /* Delete the submenus */
        wxMenu *submenu = item->GetSubMenu();
        if( submenu )
        {
            RecursiveDestroy( submenu );
        }
        menu->Delete( item );
    }
}

void Interface::OnMenuOpen(wxMenuEvent& event)
{
#if !defined( __WXMSW__ )
    if( event.GetEventObject() == p_audio_menu )
    {
        if( b_audio_menu )
        {
            p_audio_menu = AudioMenu( p_intf, this );

            /* Work-around for buggy wxGTK */
            wxMenu *menu = GetMenuBar()->GetMenu( 3 );
            RecursiveDestroy( menu );
            /* End work-around */

            menu =
                GetMenuBar()->Replace( 3, p_audio_menu, wxU(_("&Audio")) );
            if( menu ) delete menu;

            b_audio_menu = 0;
        }
        else b_audio_menu = 1;
    }
    else if( event.GetEventObject() == p_video_menu )
    {
        if( b_video_menu )
        {
            p_video_menu = VideoMenu( p_intf, this );

            /* Work-around for buggy wxGTK */
            wxMenu *menu = GetMenuBar()->GetMenu( 4 );
            RecursiveDestroy( menu );
            /* End work-around */

            menu =
                GetMenuBar()->Replace( 4, p_video_menu, wxU(_("&Video")) );
            if( menu ) delete menu;

            b_video_menu = 0;
        }
        else b_video_menu = 1;
    }
    else if( event.GetEventObject() == p_navig_menu )
    {
        if( b_navig_menu )
        {
            p_navig_menu = NavigMenu( p_intf, this );

            /* Work-around for buggy wxGTK */
            wxMenu *menu = GetMenuBar()->GetMenu( 5 );
            RecursiveDestroy( menu );
            /* End work-around */

            menu =
                GetMenuBar()->Replace( 5, p_navig_menu, wxU(_("&Navigation")));
            if( menu ) delete menu;

            b_navig_menu = 0;
        }
        else b_navig_menu = 1;
    }

#else
    p_audio_menu = AudioMenu( p_intf, this );
    wxMenu *menu = GetMenuBar()->Replace( 3, p_audio_menu, wxU(_("&Audio")) );
    if( menu ) delete menu;

    p_video_menu = VideoMenu( p_intf, this );
    menu = GetMenuBar()->Replace( 4, p_video_menu, wxU(_("&Video")) );
    if( menu ) delete menu;

    p_navig_menu = NavigMenu( p_intf, this );
    menu = GetMenuBar()->Replace( 5, p_navig_menu, wxU(_("&Navigation")) );
    if( menu ) delete menu;

#endif
}

#if defined( __WXMSW__ ) || defined( __WXMAC__ )
void Interface::OnContextMenu2(wxContextMenuEvent& event)
{
    /* Only show the context menu for the main interface */
    if( GetId() != event.GetId() )
    {
        event.Skip();
        return;
    }

    if( p_intf->p_sys->pf_show_dialog )
        p_intf->p_sys->pf_show_dialog( p_intf, INTF_DIALOG_POPUPMENU, 0, 0 );
}
#endif
void Interface::OnContextMenu(wxMouseEvent& event)
{
    if( p_intf->p_sys->pf_show_dialog )
        p_intf->p_sys->pf_show_dialog( p_intf, INTF_DIALOG_POPUPMENU, 0, 0 );
}

void Interface::OnExit( wxCommandEvent& WXUNUSED(event) )
{
    /* TRUE is to force the frame to close. */
    Close(TRUE);
}

void Interface::OnAbout( wxCommandEvent& WXUNUSED(event) )
{
    wxString msg;
    msg.Printf( wxString(wxT("VLC media player " VERSION)) +
        wxU(_(" (wxWindows interface)\n\n")) +
        wxU(_("(C) 1996-2003 - the VideoLAN Team\n\n")) +
        wxU(_("The VideoLAN team <videolan@videolan.org>\n"
              "http://www.videolan.org/\n\n")) +
        wxU(_("This is the VideoLAN Client, a DVD, MPEG and DivX player."
              "\nIt can play MPEG and MPEG2 files from a file or from a "
              "network source.")) );

    wxMessageBox( msg, wxString::Format(wxU(_("About %s")),
                  wxT("VLC media player")), wxOK | wxICON_INFORMATION, this );
}

void Interface::OnShowDialog( wxCommandEvent& event )
{
    if( p_intf->p_sys->pf_show_dialog )
    {
        int i_id;

        switch( event.GetId() )
        {
        case OpenFileSimple_Event:
            i_id = INTF_DIALOG_FILE_SIMPLE;
            break;
        case OpenFile_Event:
            i_id = INTF_DIALOG_FILE;
            break;
        case OpenDisc_Event:
            i_id = INTF_DIALOG_DISC;
            break;
        case OpenNet_Event:
            i_id = INTF_DIALOG_NET;
            break;
        case OpenSat_Event:
            i_id = INTF_DIALOG_SAT;
            break;
        case Playlist_Event:
            i_id = INTF_DIALOG_PLAYLIST;
            break;
        case Logs_Event:
            i_id = INTF_DIALOG_MESSAGES;
            break;
        case FileInfo_Event:
            i_id = INTF_DIALOG_FILEINFO;
            break;
        case Prefs_Event:
            i_id = INTF_DIALOG_PREFS;
            break;
        default:
            i_id = INTF_DIALOG_FILE;
            break;

        }

        p_intf->p_sys->pf_show_dialog( p_intf, i_id, 1, 0 );
    }
}

void Interface::OnPlayStream( wxCommandEvent& WXUNUSED(event) )
{
    wxCommandEvent dummy;
    playlist_t *p_playlist =
        (playlist_t *)vlc_object_find( p_intf, VLC_OBJECT_PLAYLIST,
                                       FIND_ANYWHERE );
    if( p_playlist == NULL ) return;

    if( p_playlist->i_size )
    {
        input_thread_t *p_input = (input_thread_t *)vlc_object_find( p_intf,
                                                       VLC_OBJECT_INPUT,
                                                       FIND_ANYWHERE );
        if( p_input == NULL )
        {
            /* No stream was playing, start one */
            playlist_Play( p_playlist );
            TogglePlayButton( PLAYING_S );
            vlc_object_release( p_playlist );
            return;
        }

        if( p_input->stream.control.i_status != PAUSE_S )
        {
            /* A stream is being played, pause it */
            input_SetStatus( p_input, INPUT_STATUS_PAUSE );
            TogglePlayButton( PAUSE_S );
            vlc_object_release( p_playlist );
            vlc_object_release( p_input );
            return;
        }

        /* Stream is paused, resume it */
        input_SetStatus( p_input, INPUT_STATUS_PLAY );
        TogglePlayButton( PLAYING_S );
        vlc_object_release( p_input );
        vlc_object_release( p_playlist );
    }
    else
    {
        /* If the playlist is empty, open a file requester instead */
        vlc_object_release( p_playlist );
        OnShowDialog( dummy );
    }
}

void Interface::OnStopStream( wxCommandEvent& WXUNUSED(event) )
{
    playlist_t * p_playlist =
        (playlist_t *)vlc_object_find( p_intf, VLC_OBJECT_PLAYLIST,
                                       FIND_ANYWHERE );
    if( p_playlist == NULL )
    {
        return;
    }

    playlist_Stop( p_playlist );
    TogglePlayButton( PAUSE_S );
    vlc_object_release( p_playlist );
}

void Interface::OnSliderUpdate( wxScrollEvent& event )
{
    vlc_mutex_lock( &p_intf->change_lock );

#ifdef WIN32
    if( event.GetEventType() == wxEVT_SCROLL_THUMBRELEASE
        || event.GetEventType() == wxEVT_SCROLL_ENDSCROLL )
    {
#endif
        if( p_intf->p_sys->i_slider_pos != event.GetPosition()
            && p_intf->p_sys->p_input )
        {
            p_intf->p_sys->i_slider_pos = event.GetPosition();
            input_Seek( p_intf->p_sys->p_input, p_intf->p_sys->i_slider_pos *
                        100 / SLIDER_MAX_POS,
                        INPUT_SEEK_PERCENT | INPUT_SEEK_SET );
        }

#ifdef WIN32
        p_intf->p_sys->b_slider_free = VLC_TRUE;
    }
    else
    {
        p_intf->p_sys->b_slider_free = VLC_FALSE;

        if( p_intf->p_sys->p_input )
        {
            /* Update stream date */
#define p_area p_intf->p_sys->p_input->stream.p_selected_area
            char psz_time[ OFFSETTOTIME_MAX_SIZE ];

            slider_box->SetLabel(
                wxU(input_OffsetToTime( p_intf->p_sys->p_input,
                                        psz_time,
                                        p_area->i_size * event.GetPosition()
                                        / SLIDER_MAX_POS )) );
#undef p_area
        }
    }
#endif

    vlc_mutex_unlock( &p_intf->change_lock );
}

void Interface::OnPrevStream( wxCommandEvent& WXUNUSED(event) )
{
    playlist_t * p_playlist =
        (playlist_t *)vlc_object_find( p_intf, VLC_OBJECT_PLAYLIST,
                                       FIND_ANYWHERE );
    if( p_playlist == NULL )
    {
        return;
    }

    vlc_mutex_lock( &p_playlist->object_lock );
    if( p_playlist->p_input != NULL )
    {
        vlc_mutex_lock( &p_playlist->p_input->stream.stream_lock );
        if( p_playlist->p_input->stream.p_selected_area->i_id > 1 )
        {
            vlc_value_t val; val.b_bool = VLC_TRUE;
            vlc_mutex_unlock( &p_playlist->p_input->stream.stream_lock );
            var_Set( p_playlist->p_input, "prev-title", val );
            vlc_mutex_unlock( &p_playlist->object_lock );
            vlc_object_release( p_playlist );  
            return;
        }
        vlc_mutex_unlock( &p_playlist->p_input->stream.stream_lock );
    }
    vlc_mutex_unlock( &p_playlist->object_lock );

    playlist_Prev( p_playlist );
    vlc_object_release( p_playlist );
}

void Interface::OnNextStream( wxCommandEvent& WXUNUSED(event) )
{
    playlist_t * p_playlist =
        (playlist_t *)vlc_object_find( p_intf, VLC_OBJECT_PLAYLIST,
                                       FIND_ANYWHERE );
    if( p_playlist == NULL )
    {
        return;
    }

    vlc_mutex_lock( &p_playlist->object_lock );
    if( p_playlist->p_input != NULL )
    {
        vlc_mutex_lock( &p_playlist->p_input->stream.stream_lock );
        if( p_playlist->p_input->stream.i_area_nb > 1 &&
            p_playlist->p_input->stream.p_selected_area->i_id <
              p_playlist->p_input->stream.i_area_nb - 1 )
        {
            vlc_value_t val; val.b_bool = VLC_TRUE;
            vlc_mutex_unlock( &p_playlist->p_input->stream.stream_lock );
            var_Set( p_playlist->p_input, "next-title", val );
            vlc_mutex_unlock( &p_playlist->object_lock );
            vlc_object_release( p_playlist );  
            return;
        }
        vlc_mutex_unlock( &p_playlist->p_input->stream.stream_lock );
    }
    vlc_mutex_unlock( &p_playlist->object_lock );

    playlist_Next( p_playlist );
    vlc_object_release( p_playlist );
}

void Interface::OnSlowStream( wxCommandEvent& WXUNUSED(event) )
{
    input_thread_t *p_input =
        (input_thread_t *)vlc_object_find( p_intf, VLC_OBJECT_INPUT,
                                           FIND_ANYWHERE );
    if( p_input )
    {
        input_SetStatus( p_input, INPUT_STATUS_SLOWER );
        vlc_object_release( p_input );
    }
}

void Interface::OnFastStream( wxCommandEvent& WXUNUSED(event) )
{
    input_thread_t *p_input =
        (input_thread_t *)vlc_object_find( p_intf, VLC_OBJECT_INPUT,
                                           FIND_ANYWHERE );
    if( p_input )
    {
        input_SetStatus( p_input, INPUT_STATUS_FASTER );
        vlc_object_release( p_input );
    }
}

void Interface::TogglePlayButton( int i_playing_status )
{
    if( i_playing_status == i_old_playing_status )
        return;

    GetToolBar()->DeleteTool( PlayStream_Event );

    if( i_playing_status == PLAYING_S )
    {
        GetToolBar()->InsertTool( 5, PlayStream_Event, wxU(_("Pause")),
                                  wxBitmap( pause_xpm ), wxNullBitmap,
                                  wxITEM_NORMAL, wxU(_(HELP_PAUSE)) );
    }
    else
    {
        GetToolBar()->InsertTool( 5, PlayStream_Event, wxU(_("Play")),
                                  wxBitmap( play_xpm ), wxNullBitmap,
                                  wxITEM_NORMAL, wxU(_(HELP_PLAY)) );
    }

    GetToolBar()->Realize();

    i_old_playing_status = i_playing_status;
}

#if !defined(__WXX11__)
/*****************************************************************************
 * Definition of DragAndDrop class.
 *****************************************************************************/
DragAndDrop::DragAndDrop( intf_thread_t *_p_intf )
{
    p_intf = _p_intf;
}

bool DragAndDrop::OnDropFiles( wxCoord, wxCoord,
                               const wxArrayString& filenames )
{
    /* Add dropped files to the playlist */

    playlist_t *p_playlist =
        (playlist_t *)vlc_object_find( p_intf, VLC_OBJECT_PLAYLIST,
                                       FIND_ANYWHERE );
    if( p_playlist == NULL )
    {
        return FALSE;
    }

    for( size_t i = 0; i < filenames.GetCount(); i++ )
        playlist_Add( p_playlist, (const char *)filenames[i].mb_str(), 0, 0,
                      PLAYLIST_APPEND | (i ? 0 : PLAYLIST_GO), PLAYLIST_END );

    vlc_object_release( p_playlist );

    return TRUE;
}
#endif

/*****************************************************************************
 * Definition of wxVolCtrl class.
 *****************************************************************************/
wxVolCtrl::wxVolCtrl( intf_thread_t *_p_intf, wxWindow* parent, wxWindowID id )
  : wxGauge( parent, id, 200, wxDefaultPosition, wxDefaultSize,
             wxGA_VERTICAL | wxGA_SMOOTH )
{
    p_intf = _p_intf;
}

void wxVolCtrl::OnChange( wxMouseEvent& event )
{
    if( !event.LeftDown() && !event.LeftIsDown() ) return;

    int i_volume = (GetClientSize().GetHeight() - event.GetY()) * 200 /
                    GetClientSize().GetHeight();
    Change( i_volume );
}

void wxVolCtrl::Change( int i_volume )
{
    aout_VolumeSet( p_intf, i_volume * AOUT_VOLUME_MAX / 200 );
    SetValue( i_volume );
    SetToolTip( wxString::Format((wxString)wxU(_("Volume")) + wxT(" %d"),
                i_volume ) );
}
