/*
 *  wasp.c
 *
 *  Write Audio Spectrum Projection
 *
 *  This program reads a binary input file, extracts its audio
 *  spectrum projection, and writes the extraction to the output
 *  file as tab-delimited text.  Optionally, its audio spectrum
 *  basis can be extracted to a separate text file.
 *
 *  The binary input file shall contain an audio spectrum
 *  envelope in the intermediate EASE format.
 *
 *  The extraction algorithm conforms to the MPEG-7 standard.
 *
 *  This program assumes a high QoI of floating point
 *  mathematics, the availability of the LAPACK library, and
 *  that int and long are of the same size.
 *
 *  Copyright (C) 2002, 2003 Tak-Shing Chan
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * $Log: wasp.c,v $
 * Revision 1.1.1.1  2003/12/18 21:25:32  chan12
 * Initial import.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <float.h>
#include <math.h>
#include <time.h>
#include <unistd.h>
#include "config.h"
#include "ica.h"

/* Prototypes for DNRM2, DGESDD, fastICA and Jade */
extern double	F77_FUNC(dnrm2, DNRM2)(int *, double *, int *);
extern void	F77_FUNC(dgesdd, DGESDD)(char *, int *, int *,
			double *, int *, double *, double *,
			int *, double *, int *, double *, int *,
			int *, int *);
extern void	icainc_JM(double *, double *, int *, int *,
			int *, double *, int *, int *, int *,
			int *, double *, int *, int *, double *,
			double *, double *, double *, double *);
extern void	Jade(double *, double *, int, int);

/* The famous constant */
#ifndef M_PI
#define M_PI		3.14159265358979323846
#endif

/* A Perl-like error handler */
static void
die(const char *msg)
{
    perror(msg);
    exit(EXIT_FAILURE);
}

/*
 * Generate ceil(n / 2) pairs of Gaussian random variables using
 * the Box-Muller method and put them into X.
 */
static void
box_muller(double *X, int n)
{
    double	U1, U2;

    for (; n > 0; n -= 2) {
	U1 = rand() / (RAND_MAX + 1.0) + DBL_MIN;
	U2 = rand() / (RAND_MAX + 1.0);
	*X++ = sqrt(-2 * log(U1)) * cos(2 * M_PI * U2);
	*X++ = sqrt(-2 * log(U1)) * sin(2 * M_PI * U2);
    }
}

/*
 * Extract the audio spectrum projection of the input file
 * pointed to by ifp into the output file pointed to by ofp,
 * using the AudioSpectrumProjection D format.  If bfp is not
 * NULL, the first k basis functions will be written to the
 * output file pointed to by bfp.  The value of icatype has the
 * following meanings:
 *
 *      icatype = 0	No ICA
 *              = 1	Symmetric fastICA (tanh)
 *              = 2	Jade
 *              = 3	Extended infomax
 */
static void
ExtractAudioSpectrumProjection(int k, int icatype, FILE *ifp,
				FILE *ofp, FILE *bfp)
{
    int		m, n, min, max, incx = 1, lwork, *iwork, info,
		i, j, rowflag = 1, colflag = 0,
		g = 1,		/* g(u) = tanh(a1 * u) */
		maxNumIterations = 1000,
		approach = 0,	/* Symmetric decorrelation */
		*signs;
    double	*XT, *r, da, *X, *U, *SIGMA, *VT, *work,
		*initGuess, a1 = 1, epsilon = 0.0001,
		*K, *W, *A, *S, alpha = 1, beta = 0, *YT;

    /*
     * Get data dimensions.  Assumes a modern file system where
     * SEEK_END on binary files is meaningful.
     */
    if (fseek(ifp, 0, SEEK_END))
	die("fseek");
    if ((m = (ftell(ifp) - sizeof n) / sizeof *XT) < 0)
	die("ftell");
    rewind(ifp);
    if (fread(&n, sizeof n, 1, ifp) != 1)
	die("fread");
    if (n < k) {
	fprintf(stderr, "Not enough dimensions\n");
	exit(EXIT_FAILURE);
    }
    m /= n;

    /* Prepare for DGESDD */
    min = m < n ? m : n;
    max = m > n ? m : n;
    if (4 * min * (min + 1) > max)
	max = 4 * min * (min + 1);
    lwork = 3 * min * min + max;

    /* Allocate working memory */
    if (!(XT = calloc(n * m, sizeof *XT))			||
	!(r = calloc(m, sizeof *r))				||
	!(X = calloc(m * n, sizeof *X))				||
	!(U = calloc(m * min, sizeof *U))			||
	!(SIGMA = calloc(min, sizeof *SIGMA))			||
	!(VT = calloc(min * n, sizeof *VT))			||
	!(work = calloc(lwork, sizeof *work))			||
	!(iwork = calloc(8 * min, sizeof *iwork))		||
	!(initGuess = calloc(k * k + 1, sizeof *initGuess))	||
	!(K = calloc(k * k, sizeof *K))				||
	!(W = calloc(k * k, sizeof *W))				||
	!(A = calloc(k * k, sizeof *A))				||
	!(S = calloc(k * n, sizeof *S))				||
	!(signs = calloc(k, sizeof *signs))			||
	!(YT = calloc(k * m, sizeof *YT)))
	die("calloc");

    /*
     * Read in row major data.  LAPACK and BLAS will see this as
     * transposed data.
     */
    if (fread(XT, sizeof *XT, n * m, ifp) != n * m)
	die("fread");

    /* dB scaling */
    for (i = 0; i < n * m; i++)
	XT[i] = 10 * log10(XT[i] + DBL_MIN);

    /* L2 norming (and saving norm values in r) */
    for (i = 0; i < m; i++) {
	r[i] = F77_FUNC(dnrm2, DNRM2)(&n, &XT[n * i], &incx);
	da = 1 / r[i];
	F77_FUNC(dscal, DSCAL)(&n, &da, &XT[n * i], &incx);
    }

    /* Get X from X' */
    for (i = 0; i < m; i++)
	for (j = 0; j < n; j++)
	    X[m * j + i] = XT[n * i + j];

    /* Basis extraction using economy SVD */
    F77_FUNC(dgesdd, DGESDD)("S", &m, &n, X, &m, SIGMA, U, &m,
		VT, &min, work, &lwork, iwork, &info);
    if (info) {
	fprintf(stderr, "DGESDD failed\n");
	exit(EXIT_FAILURE);
    }

    /* Dimensionality reduction on V' */
    if (k < n)
	for (j = 0; j < n; j++)
	    for (i = 0; i < k; i++)
		VT[k * j + i] = VT[n * j + i];

    /* Check for the optional ICA step */
    switch (icatype) {
    case 0:			/* Skip the ICA step */
	/* Dump basis if requested */
	if (bfp) {
	    for (i = 0; i < n; i++) {
		fprintf(bfp, "%.*g", DBL_DIG, VT[k * i]);
		for (j = 1; j < k; j++)
		    fprintf(bfp, "\t%.*g", DBL_DIG, VT[k * i + j]);
		putc('\n', bfp);
	    }
	}

	/* Audio spectrum projection Y' = V'X' */
	F77_FUNC(dgemm, DGEMM)("N", "N", &k, &m, &n, &alpha, VT,
		&k, XT, &n, &beta, YT, &k);
	break;

    case 1:			/* Symmetric fastICA (tanh) */
	/* FastICA on V' */
	srand(time(NULL));
	box_muller(initGuess, k * k);
	verbose = 0;
	icainc_JM(VT, initGuess, &k, &n, &k, &a1, &rowflag,
		&colflag, &g, &maxNumIterations, &epsilon,
		&approach, &verbose, X, K, W, A, S);

	/* Statistically independent basis V' = S' */
	for (i = 0; i < k; i++)
	    for (j = 0; j < n; j++)
		VT[k * j + i] = S[n * i + j];

	/* Dump basis if requested */
	if (bfp) {
	    for (i = 0; i < n; i++) {
		fprintf(bfp, "%.*g", DBL_DIG, VT[k * i]);
		for (j = 1; j < k; j++)
		    fprintf(bfp, "\t%.*g", DBL_DIG, VT[k * i + j]);
		putc('\n', bfp);
	    }
	}

	/* Audio spectrum projection Y' = V'X' */
	F77_FUNC(dgemm, DGEMM)("N", "N", &k, &m, &n, &alpha, VT,
		&k, XT, &n, &beta, YT, &k);
	break;

    case 2:					/* Jade */
	/* Transposed copy X = (V')' */
	for (i = 0; i < k; i++)
	    for (j = 0; j < n; j++)
		X[k * j + i] = VT[n * i + j];

	/* Jade on X */
	Jade(A, X, k, n);

	/* Transposed independent basis S = AV */
	F77_FUNC(dgemm, DGEMM)("N", "T", &k, &n, &k, &alpha, A,
		&k, VT, &n, &beta, S, &k);

	/* Dump basis if requested */
	if (bfp) {
	    for (i = 0; i < n; i++) {
		fprintf(bfp, "%.*g", DBL_DIG, S[k * i]);
		for (j = 1; j < k; j++)
		    fprintf(bfp, "\t%.*g", DBL_DIG, S[k * i + j]);
		putc('\n', bfp);
	    }
	}

	/* Audio spectrum projection Y' = SX' */
	F77_FUNC(dgemm, DGEMM)("N", "N", &k, &m, &n, &alpha, S,
		&k, XT, &n, &beta, YT, &k);
	break;

    case 3:				/* Extended infomax */
	/* Transposed copy X = (V')' */
	for (i = 0; i < k; i++)
	    for (j = 0; j < n; j++)
		X[k * j + i] = VT[n * i + j];

	/* Extended infomax on X */
	ext = 1;
	extblocks = DEFAULT_EXTBLOCKS;
	pdfsize = MAX_PDFSIZE;
	nsub = DEFAULT_NSUB;
	verbose = 0;
	block = DEFAULT_BLOCK(n);
	maxsteps = DEFAULT_MAXSTEPS;
	lrate = DEFAULT_LRATE(k);
	annealstep = DEFAULT_EXTANNEAL;
	annealdeg = DEFAULT_ANNEALDEG;
	nochange = DEFAULT_STOP;
	momentum = DEFAULT_MOMENTUM;
	zero(k * k, W);
	rmmean(X, k, n);
	do_sphere(X, k, n, K);
	syproj(X, K, k, n, VT);
	runica(VT, W, k, n, 1, NULL, signs);

	/* Transposed independent basis S */
	posact(VT, W, k, n, S);
	varsort(S, W, K, NULL, NULL, signs, k, n, k);

	/* Dump basis if requested */
	if (bfp) {
	    for (i = 0; i < n; i++) {
		fprintf(bfp, "%.*g", DBL_DIG, S[k * i]);
		for (j = 1; j < k; j++)
		    fprintf(bfp, "\t%.*g", DBL_DIG, S[k * i + j]);
		putc('\n', bfp);
	    }
	}

	/* Audio spectrum projection Y' = SX' */
	F77_FUNC(dgemm, DGEMM)("N", "N", &k, &m, &n, &alpha, S,
		&k, XT, &n, &beta, YT, &k);
    }

    /*
     * Output r and Y as tab-delimited text.  This corresponds
     * to the vector values of AudioSpectrumProjection D.
     */
    for (i = 0; i < m; i++) {
	fprintf(ofp, "%.*g", DBL_DIG, r[i]);
	for (j = 0; j < k; j++)
	    fprintf(ofp, "\t%.*g", DBL_DIG, YT[k * i + j]);
	putc('\n', ofp);
    }

    /* Clean up */
    free(YT);
    free(signs);
    free(S);
    free(A);
    free(W);
    free(K);
    free(initGuess);
    free(iwork);
    free(work);
    free(VT);
    free(SIGMA);
    free(U);
    free(X);
    free(r);
    free(XT);
}

/* The main program */
int
main(int argc, char *argv[])
{
    extern char	*optarg;
    extern int	optind;
    int		c, flag = 0, k = 5, icatype = 0;
    FILE	*ifp, *ofp, *bfp = NULL;

    /* Get command options */
    while ((c = getopt(argc, argv, "n:t:b:")) != EOF) {
	switch (c) {
	case 'n':
	    if (sscanf(optarg, "%d", &k) != 1 || k < 1)
		flag++;
	    break;
	case 't':
	    if (sscanf(optarg, "%d", &icatype) != 1 ||
		icatype < 0 || icatype > 3)
		flag++;
	    break;
	case 'b':
	    if (!(bfp = fopen(optarg, "w")))
		die("fopen");
	    break;
	case '?':
	    flag++;
	}
    }

    /* Illegal options? */
    if (flag ||	argc != optind + 2) {
	fprintf(stderr, "usage: %s [-n k] [-t icatype] "
		"[-b basisfile] infile outfile\n\n"
		"Valid values for icatype:\n"
		"     0    No ICA\n"
		"     1    Symmetric fastICA (tanh)\n"
		"     2    Jade\n"
		"     3    Extended infomax\n", argv[0]);
	return EXIT_FAILURE;
    }

    /* Open input file */
    if (!(ifp = fopen(argv[optind], "rb"))	||
	!(ofp = fopen(argv[optind + 1], "w")))
	die("fopen");

    /* Extract the audio spectrum projection */
    ExtractAudioSpectrumProjection(k, icatype, ifp, ofp, bfp);

    /* Closing up */
    if (bfp)
	fclose(bfp);
    fclose(ofp);
    fclose(ifp);
    return EXIT_SUCCESS;
}
