/******************************************************************************/
/* This file is originally written by:                                        */
/*                                                                            */
/*   Sigurd Enghoff                                                           */
/*   The Salk Institute, CNL                                                  */
/*   enghoff@salk.edu                                                         */
/*                                                                            */
/* And has been modified by Tak-Shing Chan:                                   */
/*                                                                            */
/*   - Assumed real is double and integer is int                              */
/*   - Renamed extended to ext for compatibility with BSD and Solaris         */
/*   - Merged in dsum.c                                                       */
/*   - Incorporated F77_FUNC wrappers from autoconf                           */
/*   - Removed R250 stuffs                                                    */
/*                                                                            */
/******************************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <math.h>
#include "config.h"
#include "ica.h"

/* Global and externally accessible variables */
int ext, extblocks, pdfsize, nsub;
int verbose, block, maxsteps;

double lrate, annealstep, annealdeg, nochange, momentum;

/************************ Takes the sum of the values *************************/
/* Returns the sum of the values.                                             */
/*                                                                            */
/* n: int (input)                                                             */
/* dx: double array [n] (input)                                               */
/* incx: int (input)                                                          */

double F77_FUNC(dsum,DSUM)(int *n, double *dx, int *incx)
{


    /* System generated locals */
    int i__1, i__2;
    double ret_val, d__1, d__2, d__3, d__4, d__5, d__6;

    /* Local variables */
    static int i, m;
    static double dtemp;
    static int nincx, mp1;


/*     takes the sum of the values.   
       jack dongarra, linpack, 3/11/78.   
       modified 3/93 to return if incx .le. 0.   
       modified 12/3/93, array(1) declarations changed to array(*)   


    
   Parameter adjustments   
       Function Body */
#define DX(I) dx[(I)-1]


    ret_val = 0.;
    dtemp = 0.;
    if (*n <= 0 || *incx <= 0) {
	return ret_val;
    }
    if (*incx == 1) {
	goto L20;
    }

/*        code for increment not equal to 1 */

    nincx = *n * *incx;
    i__1 = nincx;
    i__2 = *incx;
    for (i = 1; *incx < 0 ? i >= nincx : i <= nincx; i += *incx) {
	dtemp += (d__1 = DX(i), d__1);
/* L10: */
    }
    ret_val = dtemp;
    return ret_val;

/*        code for increment equal to 1   


          clean-up loop */

L20:
    m = *n % 6;
    if (m == 0) {
	goto L40;
    }
    i__2 = m;
    for (i = 1; i <= m; ++i) {
	dtemp += (d__1 = DX(i), d__1);
/* L30: */
    }
    if (*n < 6) {
	goto L60;
    }
L40:
    mp1 = m + 1;
    i__2 = *n;
    for (i = mp1; i <= *n; i += 6) {
	dtemp = dtemp + (d__1 = DX(i), d__1) + (d__2 = DX(i + 1), 
		d__2) + (d__3 = DX(i + 2), d__3) + (d__4 = DX(i + 3), 
		d__4) + (d__5 = DX(i + 4), d__5) + (d__6 = DX(i + 5)
		, d__6);
/* L50: */
    }
L60:
    ret_val = dtemp;
    return ret_val;
} /* F77_FUNC(dsum,DSUM) */


/***************************** Zero-fill a array ******************************/
/* Zero-fill array A of n elements.                                           */
/*                                                                            */
/* n: int (input)                                                             */
/* A: double array [n] (output)                                               */

void zero(int n, double *A) {
	double dzero = 0.0;
	int izero = 0, ione = 1;

	F77_FUNC(dcopy,DCOPY)(&n,&dzero,&izero,A,&ione);
}

/************************ Construct an identity matrix ************************/
/* Construct an identity matrix A of size m x m.                              */
/*                                                                            */
/* m: int (input)                                                             */
/* A: double array [m,m] (output)                                             */

void eye(int m, double *A) {
	double dzero = 0.0, done = 1.0;
	int izero = 0, ione = 1;
	int mxm = m*m, m1 = m+1;

	F77_FUNC(dcopy,DCOPY)(&mxm,&dzero,&izero,A,&ione);
	F77_FUNC(dcopy,DCOPY)(&m,&done,&izero,A,&m1);
}


/*********************** Initialize a permutation vector **********************/
/* Initialize a vector perm of n elements for constructing a random.          */
/*                                                                            */
/* permutation of the number in the range [0..n-1]                            */
/* perm: int array [n] (output)                                               */
/* n:    int (input)                                                          */

void initperm(int *perm, int n) {
	int i;
	for (i=0 ; i<n ; i++) perm[i]=i;
}


/******************* Obtain and project a random permutation ******************/
/* n elements of vector perm are picked at random and used as indecies into   */
/* data. The designated elements of data are projected by trsf employing the  */
/* bias, if not NULL. k denotes the number of elements of perm left to be     */
/* extracted; perm is updated while calling routine must update k. The        */
/* resulting projections are returned in proj.                                */
/*                                                                            */
/* data: double array [m,?] (input)                                           */
/* trsf: double array [m,m] (input)                                           */
/* bias: double array [m] or NULL (input)                                     */
/* perm: int array [k>] (input/output)                                        */
/* m:    int (input)                                                          */
/* n:    int (input)                                                          */
/* k:    int (input)                                                          */
/* proj: double array [m,n] (output)                                          */

void randperm(double *data, double *trsf, double *bias, int *perm, int m, int n, int k, double *proj) {
	int i, im, swap, inc = 1;
	double alpha = 1.0, beta;
	char trans='N';
	
	if (bias) {
		for (i=0,im=0 ; i<n ; i++,im+=m) F77_FUNC(dcopy,DCOPY)(&m,bias,&inc,&proj[im],&inc);
		beta = 1.0;
	}
	else
		beta = 0.0;

	for (i=0,im=0 ; i<n ; i++,im+=m) {
		swap = rand()%k;
		F77_FUNC(dgemv,DGEMV)(&trans,&m,&m,&alpha,trsf,&m,&data[perm[swap]*m],&inc,&beta,&proj[im],&inc);
		perm[swap] = perm[--k];
	}
}


/********* Obtain and project data points from Laplacian distribution *********/
/* n samples are picked from a Laplacian distribution centered at the center  */
/* point of a length frames window. Data points are extracted from frames in  */
/* data according to the windowed distribution while chosen uniformly over    */
/* epochs. The designated elements of data are projected by trsf employing    */
/* the bias, if not NULL. The resulting projections are returned in proj.     */
/*                                                                            */
/* data:   double array [m,?] (input)                                         */
/* trsf:   double array [m,m] (input)                                         */
/* bias:   double array [m] or NULL (input)                                   */
/* m:      int (input)                                                        */
/* n:      int (input)                                                        */
/* frames: int (input)                                                        */
/* epochs: int (input)                                                        */
/* proj:   double array [m,n] (output)                                        */

void probperm(double *data, double *trsf, double *bias, int m, int n, int frames, int epochs, double *proj) {
	int i, im, idx, norm, prob, inc = 1;
	double alpha = 1.0, beta;
	char trans='N';

	if (bias) {
		for (i=0,im=0 ; i<n ; i++,im+=m) F77_FUNC(dcopy,DCOPY)(&m,bias,&inc,&proj[im],&inc);
		beta = 1.0;
	}
	else
		beta = 0.0;

	norm = (1<<((frames+1)/2+1))-2*((frames%2)-1);

	for (i=0,im=0 ; i<n ; i++,im+=m) {
		prob = rand()%norm;

		if (prob & 1) {
			idx = 0;
			prob = (prob>>1) + 1;
			while (prob > 1) {
				prob >>= 1;
				idx++;
			}
		}
		else {
			idx = frames-1;
			prob = (prob>>1) + 1;
			while (prob > 1) {
				prob >>= 1;
				idx--;
			}
		}

		idx += frames*(rand()%epochs);
		F77_FUNC(dgemv,DGEMV)(&trans,&m,&m,&alpha,trsf,&m,&data[idx*m],&inc,&beta,&proj[im],&inc);
	}
}


/**************************** Compute PDF estimate ****************************/
/* If perm is not NULL, n elements of vector perm are picked at random and    */
/* used as indecies into data. If perm is NULL, the first n elements of data  */
/* are referenced. The designated n elements of data used for the Kurosis     */
/* estimate. k denotes the number of elements of perm left to be extracted;   */
/* perm is updated while calling routine must update k. The resulting         */
/* PDF estimate is returned in kk.                                            */
/*                                                                            */
/* data: double array [m,? or n] (input)                                      */
/* trsf: double array [m,m] (input)                                           */
/* perm: int array [k>] (input/output) or NULL                                */
/* m:    int (input)                                                          */
/* n:    int (input)                                                          */
/* k:    int (input)                                                          */
/* kk:   double array [m,n] (output)                                          */

void pdf(double *data, double *trsf, int *perm, int m, int n, int k, double *kk) {
	int i, j, im, swap, inc = 1;
	double alpha = 1.0, beta = 0.0, tmp;
	char trans='N';

#ifdef KURTOSIS_ESTIMATE
    zero(2*m,&kk[m]);

    for (i=0,im=0 ; i<n ; i++,im+=m) {
        if (perm) {
            swap = rand()%k;
            F77_FUNC(dgemv,DGEMV)(&trans,&m,&m,&alpha,trsf,&m,&data[perm[swap]*m],&inc,&beta,kk,&inc);
            perm[swap] = perm[--k];
        }
        else
            F77_FUNC(dgemv,DGEMV)(&trans,&m,&m,&alpha,trsf,&m,&data[im],&inc,&beta,kk,&inc);
        
        for (j=0 ; j<m ; j++) {
            tmp = kk[j]*kk[j];
            kk[j+m] += tmp;
            kk[j+2*m] += tmp*tmp;
        }
    }

    for (i=0 ; i<m ; i++)
        kk[i] = kk[i+2*m]*(double)n / (kk[i+m]*kk[i+m]) - 3.0;

#else

	zero(3*m,&kk[m]);

	for (i=0,im=0 ; i<n ; i++,im+=m) {
		if (perm) {
			swap = rand()%k;
			F77_FUNC(dgemv,DGEMV)(&trans,&m,&m,&alpha,trsf,&m,&data[perm[swap]*m],&inc,&beta,kk,&inc);
			perm[swap] = perm[--k];
		}
		else
			F77_FUNC(dgemv,DGEMV)(&trans,&m,&m,&alpha,trsf,&m,&data[im],&inc,&beta,kk,&inc);
		
		for (j=0 ; j<m ; j++) {
			tmp = 2.0 / (exp(kk[j])+exp(-kk[j]));
			kk[j+m] += tmp*tmp;
			kk[j+2*m] += kk[j]*kk[j];
			kk[j+3*m] += tanh(kk[j])*kk[j];
		}
	}

	for (i=0 ; i<m ; i++)
		kk[i] = (kk[i+m]*kk[i+2*m]/((double)n)-kk[i+3*m])/((double)n);
#endif
}


/******************* Project data using a general projection ******************/
/* Project data using trsf. Return projections in proj.                       */
/*                                                                            */
/* data: double array [m,n] (input)                                           */
/* trsf: double array [m,m] (input)                                           */
/* m:    int (input)                                                          */
/* n:    int (input)                                                          */
/* proj: double array [m,n] (output)                                          */

void geproj(double *data, double *trsf, int m, int n, double *proj) {
	double alpha = 1.0, beta = 0.0;
	char trans='N';

	F77_FUNC(dgemm,DGEMM)(&trans,&trans,&m,&n,&m,&alpha,trsf,&m,data,&m,&beta,proj,&m);
}


/***************** Project data using a symmetrical projection ****************/
/* Project data using the upper triangular part of trsf, assuming trsf is     */
/* symmetrical. Return projections in proj.                                   */
/*                                                                            */
/* data: double array [m,n] (input)                                           */
/* trsf: double array [m,m] (input)                                           */
/* m:    int (input)                                                          */
/* n:    int (input)                                                          */
/* proj: double array [m,n] (output)                                          */

void syproj(double *data, double *trsf, int m, int n, double *proj) {
	double alpha = 1.0, beta = 0.0;
	char uplo='U', side = 'L';

	F77_FUNC(dsymm,DSYMM)(&side,&uplo,&m,&n,&alpha,trsf,&m,data,&m,&beta,proj,&m);
}


/******************* Project data using a reduced projection ******************/
/* Project data using the first k rows of trsf only. Return projections       */
/* in proj.                                                                   */
/*                                                                            */
/* data: double array [k,n] (input)                                           */
/* trsf: double array [m,k] (input)                                           */
/* m:    int (input)                                                          */
/* n:    int (input)                                                          */
/* k:    int (input)                                                          */
/* proj: double array [m,n] (output)                                          */

void pcaproj(double *data, double *eigv, int m, int n, int k, double *proj) {
	double alpha = 1.0, beta = 0.0;
	char transn='N', transt='T';

	F77_FUNC(dgemm,DGEMM)(&transt,&transn,&m,&n,&k,&alpha,eigv,&k,data,&k,&beta,proj,&m);
}


/******************************* Used by varsort ******************************/

int compar(const void *x, const void *y) {
	if (((idxelm*)x)->val < ((idxelm*)y)->val) return 1;
	if (((idxelm*)x)->val > ((idxelm*)y)->val) return -1;
	return 0;
}

/****************** Sort data according to projected variance *****************/
/* Compute back-projected variances for each component based on inverse of    */
/* weights and sphere or pseudoinverse of weights, sphere, and eigv. Reorder  */
/* data and weights accordingly. Also if not NULL, reorder bias and signs.    */
/*                                                                            */
/* data:    double array [m,n] (input/output)                                 */
/* weights: double array [m,m] (input/output)                                 */
/* sphere:  double array [m,m] (input)                                        */
/* eigv:    double array [m,k] (input) or NULL                                */
/* bias:    double array [m] (input/output) or NULL                           */
/* signs:   int array [m] (input/output) or NULL                              */
/* m:       int (input)                                                       */
/* n:       int (input)                                                       */
/* k:       int (input)                                                       */

void varsort(double *data, double *weights, double *sphere, double *eigv, double *bias, int *signs, int m, int n, int k) {
	char name[] = "DGETRI\0", opts[] = "\0";
	double alpha = 1.0, beta = 0.0;
	int i, j, l, jm, ik, info = 0, ispec = 1, na = -1;
	char transn='N', transt='T', uplo='U', side = 'R';
	
	int nb = F77_FUNC(ilaenv,ILAENV)(&ispec,name,opts,&m,&na,&na,&na);
	int itmp, lwork = m*nb, inc = 1;
	double act, dtmp, *wcpy;

	int    *ipiv = (int*)malloc(m*sizeof(int));
	double *work = (double*)malloc(lwork*sizeof(double));
	double *winv = (double*)malloc(m*k*sizeof(double));
	double *sum  = (double*)malloc(n*sizeof(double));
	idxelm  *meanvar = (idxelm*)malloc(m*sizeof(idxelm));
	
	if (eigv) {
/* Compute pseudoinverse of weights*sphere*eigv */
		wcpy = (double*)malloc(m*m*sizeof(double));
		F77_FUNC(dsymm,DSYMM)(&side,&uplo,&m,&m,&alpha,sphere,&m,weights,&m,&beta,wcpy,&m);
		F77_FUNC(dgemm,DGEMM)(&transn,&transt,&m,&k,&m,&alpha,wcpy,&m,eigv,&k,&beta,weights,&m);

		F77_FUNC(dgetrf,DGETRF)(&m,&m,wcpy,&m,ipiv,&info);
		F77_FUNC(dgetri,DGETRI)(&m,wcpy,&m,ipiv,work,&lwork,&info);
		F77_FUNC(dgemm,DGEMM)(&transn,&transn,&k,&m,&m,&alpha,eigv,&k,wcpy,&m,&beta,winv,&k);
		free(wcpy);
	}
	else {
/* Compute inverse of weights*sphere */
		F77_FUNC(dsymm,DSYMM)(&side,&uplo,&m,&m,&alpha,sphere,&m,weights,&m,&beta,winv,&m);

		F77_FUNC(dgetrf,DGETRF)(&m,&m,winv,&m,ipiv,&info);
		F77_FUNC(dgetri,DGETRI)(&m,winv,&m,ipiv,work,&lwork,&info);
	}


/* Compute mean variances for back-projected components */
	for (i=0 ; i<m*k ; i++) winv[i] = winv[i]*winv[i];

	for (i=0,ik=0 ; i<m ; i++,ik+=k) {
		for (j=0,jm=0 ; j<n ; j++,jm+=m) {
			sum[j] = 0;
			act = data[i+jm]*data[i+jm];
			for(l=0 ; l<k ; l++) sum[j] += act*winv[l+ik];
		}
		
		meanvar[i].idx = i;
		meanvar[i].val = F77_FUNC(dsum,DSUM)(&n,sum,&inc);
		if (verbose) printf("%d ",(int)(i+1));
	}
	if (verbose) printf("\n");
	
/* Sort meanvar */
	qsort(meanvar,m,sizeof(idxelm),compar);

	if (verbose) printf("Permuting the activation wave forms ...\n");

/* Perform in-place reordering of weights, data, bias, and signs */
	for (i=0 ; i<m-1 ; i++) {
		j = meanvar[i].idx;
		if (i != j) {
			F77_FUNC(dswap,DSWAP)(&k,&weights[i],&m,&weights[j],&m);
			F77_FUNC(dswap,DSWAP)(&n,&data[i],&m,&data[j],&m);

			if (bias) {
				dtmp = bias[i];
				bias[i] = bias[j];
				bias[j] = dtmp;
			}

			if (signs) {
				itmp = signs[i];
				signs[i] = signs[j];
				signs[j] = itmp;
			}
			
			for (l=i+1 ; i!=meanvar[l].idx ; l++);
			meanvar[l].idx = j;
		}
	}
	
	free(ipiv);
	free(work);
	free(winv);
	free(sum);
	free(meanvar);
}


/******************************* Remove row mean ******************************/
/* Remove row means from data.                                                */
/*                                                                            */
/* data: double array [m,n] (input/output)                                    */
/* m:    int (input)                                                          */
/* n:    int (input)                                                          */

void rmmean(double *data, int m, int n) {
	double *mean, alpha;
	int i, j, one = 1;
	
	mean = (double*)malloc(m*sizeof(double));
	zero(m,mean);
	
	alpha = 1.0;
	for (i=0,j=0 ; i<n ; i++,j+=m) {
		F77_FUNC(daxpy,DAXPY)(&m,&alpha,&data[j],&one,mean,&one);
	}

	alpha = -1.0/(double)n;
	for (i=0,j=0 ; i<n ; i++,j+=m) {
		F77_FUNC(daxpy,DAXPY)(&m,&alpha,mean,&one,&data[j],&one);
	}	
	
	free(mean);
}


/*************************** Compute sphering matrix **************************/
/* Compute sphering matrix based on data.                                     */
/*                                                                            */
/* data: double array [m,n] (input)                                           */
/* m:    int (input)                                                          */
/* n:    int (input)                                                          */
/* sphe: double array [m,m] (output)                                          */

void do_sphere(double *data, int m, int n, double *sphe) {
	char name[] = "SSYTRD\0", opts[] = "U\0";
	double alpha = 1.0/(double)(n-1);
	double beta = 0.0;
	int info = 0, ispec = 1, na = -1;
	
	int nb = F77_FUNC(ilaenv,ILAENV)(&ispec,name,opts,&m,&na,&na,&na);
	int i, im, lwork = (nb+2)*m, inc = 1, mxm = m*m;

	char uplo='U', transn='N', jobz='V';
	double *eigv = (double*)malloc(m*m*sizeof(double));
	double *eigd = (double*)malloc(m*sizeof(double));
	int    *ipiv = (int*)malloc(m*sizeof(int));
	double *work = (double*)malloc(lwork*sizeof(double));

/*
   sphere = 2*inv(sqrtm(cov(data'))) is computes as:

   [v d] = eig(data*data')
   sphere = inv(diag(0.5 * sqrt(d))*v')*v'
*/

	F77_FUNC(dsyrk,DSYRK)(&uplo,&transn,&m,&n,&alpha,data,&m,&beta,sphe,&m);
	F77_FUNC(dsyev,DSYEV)(&jobz,&uplo,&m,sphe,&m,eigd,work,&lwork,&info);

/* Store transpose of sphe in eigv */
	for (i=0,im=0 ; i<m ; i++,im+=m)
		F77_FUNC(dcopy,DCOPY)(&m,&sphe[im],&inc,&eigv[i],&m);
	
/* Copy eigv to sphe */
	F77_FUNC(dcopy,DCOPY)(&mxm,eigv,&inc,sphe,&inc);
	
/* Scale rows of eigv by corresponding eigd values */
	for (i=0 ; i<m ; i++) {
		eigd[i] = 0.5 * sqrt(eigd[i]);
		F77_FUNC(dscal,DSCAL)(&m,&eigd[i],&eigv[i],&m);
	}

/* Solve eigv * sphere = sphe  ~  diag(0.5*sqrt(d))*v' * sphere = v' */
	F77_FUNC(dgesv,DGESV)(&m,&m,eigv,&m,ipiv,sphe,&m,&info);

	free(eigv);
	free(eigd);
	free(ipiv);
	free(work);
}


/***************************** Compute PCA matrix *****************************/
/* Compute PCA decomposition matrix based on data.                            */
/*                                                                            */
/* data: double array [m,n] (input)                                           */
/* m:    int (input)                                                          */
/* n:    int (input)                                                          */
/* trsf: double array [m,m] (output)                                          */

void pca(double *data, int m, int n, double *trsf) {
	char name[] = "SSYTRD\0", opts[] = "U\0";
	double alpha = 1.0/(double)(n-1);
	double beta = 0.0;
	int info = 0, ispec = 1, na = -1;
	
	int nb = F77_FUNC(ilaenv,ILAENV)(&ispec,name,opts,&m,&na,&na,&na);
	int lwork = (nb+2)*m;

	char uplo='U', transn='N', jobz='V';
	double *eigd = (double*)malloc(m*sizeof(double));
	double *work = (double*)malloc(lwork*sizeof(double));

	F77_FUNC(dsyrk,DSYRK)(&uplo,&transn,&m,&n,&alpha,data,&m,&beta,trsf,&m);
	F77_FUNC(dsyev,DSYEV)(&jobz,&uplo,&m,trsf,&m,eigd,work,&lwork,&info);
	
	free(eigd);
	free(work);
}


/*************** Orient components toward positive activations ****************/
/* Project data using trsf. Negate components and their corresponding weights */
/* to assure positive RMS. Returns projections in proj.                       */
/*                                                                            */
/* data: double array [m,n] (input)                                           */
/* trsf: double array [m,m] (input/output)                                    */
/* m:    int (input)                                                          */
/* n:    int (input)                                                          */
/* proj: double array [m,n] (output)                                          */

void posact(double *data, double *trsf, int m, int n, double *proj) {
	char trans='N';
	double alpha = 1.0;
	double beta = 0.0;
	double posrms, negrms;
	int pos, neg, i, j;
	
	F77_FUNC(dgemm,DGEMM)(&trans,&trans,&m,&n,&m,&alpha,trsf,&m,data,&m,&beta,proj,&m);

	if (verbose) printf("Inverting negative activations: ");
	for (i=0 ; i<m ; i++) {
		posrms = 0.0; negrms = 0.0;
		pos = 0; neg = 0;
		for (j=i ; j<m*n ; j+=m)
			if (proj[j] >= 0) {
				posrms += proj[j]*proj[j];
				pos++;
			}
			else {
				negrms += proj[j]*proj[j];
				neg++;
			}

		if (negrms*(double)pos > posrms*(double)neg) {
			if (verbose) printf("-");
			for (j=i ; j<m*n ; j+=m) proj[j] = -proj[j];
			for (j=i ; j<m*m ; j+=m) trsf[j] = -trsf[j];
		}
		if (verbose) printf("%d ",(int)(i+1));
	}
	printf("\n");
}


/******************************* Perform infomax ******************************/
/* Perform infomax or extended-infomax on data. If any elements of weights    */
/* are none-zero use weights as starting weights else use identity matrix.    */
/* If bias not NULL employ biasing. The following externally accessible       */
/* variables are assumed initialized: ext, extblocks, pdfsize, nsub,          */
/* verbose, block, maxsteps, lrate, annealstep, annealdeg, nochange, and      */
/* momentum. If the boolean variable extended is set, signs must be defined   */
/* (i.e. not NULL)                                                            */
/*                                                                            */
/* data:    double array [chans,frames*epoch] (input)                         */
/* weights: double array [chans,chans] (input/output)                         */
/* chans:   int (input)                                                       */
/* frames:  int (input)                                                       */
/* epochs:  int (input)                                                       */
/* bias:    double array [chans] (output) or NULL                             */
/* signs:   int array [chans] (output) or NULL                                */

void runica(double *data, double *weights, int chans, int frames, int epochs, double *bias, int *signs) {
	double alpha = 1.0, beta = 0.0, gamma = -1.0, epsilon;
	double change = nochange, oldchange = nochange, angledelta = 0.0;
	char uplo='U', transn='N', transt='T';
	int i, j, t = 0, inc = 1;
	
	int datalength = frames*epochs;
	int chxch = chans*chans;

	double signsbias = DEFAULT_SIGNSBIAS;
	double extmomentum = DEFAULT_EXTMOMENTUM;
	
	int wts_blowup = 0;
	int signcount = 0;
	int pleft = 0;
	int step = 0;
	int blockno = 1;
	int urextblocks;

/* Allocate arrays */
	double *startweights = (double*)malloc(chxch*sizeof(double));
	double *oldweights = (double*)malloc(chxch*sizeof(double));
	double *tmpweights = (double*)malloc(chxch*sizeof(double));
	double *delta = (double*)malloc(chxch*sizeof(double));
	double *olddelta = (double*)malloc(chxch*sizeof(double));
	double *u = (double*)malloc(chans*block*sizeof(double));
	double *y = (double*)malloc(chans*block*sizeof(double));
	double *yu = (double*)malloc(chxch*sizeof(double));
	double *prevweights, *prevwtchange;
	double *bsum, *kk, *old_kk;
	int *oldsigns, *pdfperm;

#ifndef PROB_WINDOW
	int *dataperm = (int*)malloc(datalength*sizeof(int));
#endif

/* Initialize weights if zero array */
	if (weights[F77_FUNC(idamax,IDAMAX)(&chxch,weights,&inc)-1] == 0.0) eye(chans,weights);

/* Allocate and initialize arrays and variables needed for momentum */
	if (momentum > 0.0) {
		prevweights = (double*)malloc(chxch*sizeof(double));
		prevwtchange = (double*)malloc(chxch*sizeof(double));
		F77_FUNC(dcopy,DCOPY)(&chxch,weights,&inc,prevweights,&inc);
		zero(chxch,prevwtchange);
	}
	else
		prevweights = prevwtchange = NULL;

/* Allocate and initialize arrays and variables needed for biasing */
	if (bias) {
		bsum = (double*)malloc(chans*sizeof(double));
		zero(chans,bias);
	}
	else
		bsum = NULL;

/* Allocate and initialize arrays and variables needed for extended-ICA */
	if (ext) {
		oldsigns = (int*)malloc(chans*sizeof(int));
		for (i=0 ; i<chans ; i++) oldsigns[i] = -1;
		for (i=0 ; i<nsub ; i++) signs[i] = 1;
		for (i=nsub ; i<chans ; i++) signs[i] = 0;
		
		if (pdfsize < datalength)
			pdfperm = (int*)malloc(datalength*sizeof(int));
		else {
			pdfsize = datalength;
			pdfperm = NULL;
		}
		
#ifdef KURTOSIS_ESTIMATE
		kk = (double*)malloc(3*chans*sizeof(double));
#else
		kk = (double*)malloc(4*chans*sizeof(double));
#endif
		old_kk = (double*)malloc(chans*sizeof(double));
		zero(chans,old_kk);

		if (extblocks<0 && verbose) {
			printf("Fixed extended-ICA sign assignments:  ");
			for (i=0 ; i<chans ; i++) printf("%d ",(int)(signs[i]));
			printf("\n");
		}
	}
	else {
		oldsigns = pdfperm = NULL;
		old_kk = kk = NULL;
	}
	urextblocks = extblocks;

/*************************** Initialize ICA training **************************/

	F77_FUNC(dcopy,DCOPY)(&chxch,weights,&inc,startweights,&inc);
	F77_FUNC(dcopy,DCOPY)(&chxch,weights,&inc,oldweights,&inc);
	
	if (verbose) {
		printf("Beginning ICA training ...");
		if (ext) printf(" first training step may be slow ...\n");
		else printf("\n");
	}

#ifdef FIX_SEED	
	srand(1);
#else
	srand((int)time(NULL));
#endif

	while (step < maxsteps) {

#ifndef PROB_WINDOW
		initperm(dataperm,datalength);
#endif

/***************************** ICA training block *****************************/
		for (t=0 ; t<datalength-block && !wts_blowup ; t+=block) {

#ifdef PROB_WINDOW
			probperm(data,weights,bias,chans,block,frames,epochs,u);
#else
			randperm(data,weights,bias,dataperm,chans,block,datalength-t,u);
#endif

			if (!ext) {
/************************* Logistic ICA weight update *************************/
				for (i=0 ; i<chans*block ; i++)
/*					y[i] = 1.0 - 2.0 / (1.0+exp(-u[i]));*/
					y[i] = -tanh(u[i]/2.0);

/*Bias sum for logistic ICA */
				if (bias)
					for (i=0 ; i<chans ; i++)
						bsum[i] = F77_FUNC(dsum,DSUM)(&block,&y[i],&chans);

/* Compute: (1-2*y) * u' */
				F77_FUNC(dgemm,DGEMM)(&transn,&transt,&chans,&chans,&block,&alpha,y,&chans,u,&chans,&beta,yu,&chans);
			}
			else {
/************************* Extended-ICA weight update *************************/
				for (i=0 ; i<chans*block ; i++)
					y[i] = tanh(u[i]);

/* Bias sum for extended-ICA */
				if (bias)
					for (i=0 ; i<chans ; i++)
						bsum[i] = -2*F77_FUNC(dsum,DSUM)(&block,&y[i],&chans);

/* Apply sign matrix */
				for (i=0 ; i<chans ; i++)
					if (signs[i])
						for (j=i ; j<block*chans ; j+=chans)
							y[j] = -y[j];

/* Compute: u * u' */
				F77_FUNC(dsyrk,DSYRK)(&uplo,&transn,&chans,&block,&alpha,u,&chans,&beta,yu,&chans);

				j = chxch - 2;
				for (i=1 ; i<chans ; i++) {
					F77_FUNC(dcopy,DCOPY)(&i,&yu[j],&chans,&yu[j-chans+1],&inc);
					j -= chans+1;
				}

/* Compute: -y * u' -u*u' */
				F77_FUNC(dgemm,DGEMM)(&transn,&transt,&chans,&chans,&block,&gamma,y,&chans,u,&chans,&gamma,yu,&chans);
			}
			
/* Add block identity matix */
			for (i=0 ; i<chxch ; i+=(chans+1))
				yu[i] += (double)block;

/* Apply weight change */
			F77_FUNC(dcopy,DCOPY)(&chxch,weights,&inc,tmpweights,&inc);
			F77_FUNC(dgemm,DGEMM)(&transn,&transn,&chans,&chans,&chans,&lrate,yu,&chans,tmpweights,&chans,&alpha,weights,&chans);

/* Apply bias change */
			if (bias) F77_FUNC(daxpy,DAXPY)(&chans,&lrate,bsum,&inc,bias,&inc);
			
/******************************** Add momentum ********************************/
			if (momentum > 0.0) {
				F77_FUNC(daxpy,DAXPY)(&chxch,&momentum,prevwtchange,&inc,weights,&inc);
				for (i=0 ; i<chxch ; i++)
					prevwtchange[i] = weights[i] - prevweights[i];

				F77_FUNC(dcopy,DCOPY)(&chxch,weights,&inc,prevweights,&inc);
			}
			
			if (abs(weights[F77_FUNC(idamax,IDAMAX)(&chxch,weights,&inc)-1]) > MAX_WEIGHT)
				wts_blowup = 1;

			if (ext && !wts_blowup && extblocks>0 && blockno%extblocks==0) {
				if (pdfperm && pleft < pdfsize) {
					initperm(pdfperm,datalength);
					pleft = datalength;
				}
				
				pdf(data,weights,pdfperm,chans,pdfsize,pleft,kk);
				pleft -= pdfsize;
				
				if (extmomentum > 0.0) {
					epsilon = 1.0-extmomentum;
					F77_FUNC(dscal,DSCAL)(&chans,&epsilon,kk,&inc);
					F77_FUNC(daxpy,DAXPY)(&chans,&extmomentum,old_kk,&inc,kk,&inc);
					F77_FUNC(dcopy,DCOPY)(&chans,kk,&inc,old_kk,&inc);
				}
				
				for (i=0 ; i<chans ; i++)
					signs[i] = kk[i] < -signsbias;
				
				for (i=0 ; i<chans && signs[i]==oldsigns[i] ; i++);
				if (i==chans) signcount++;
				else signcount = 0;
				
				for (i=0 ; i<chans ; i++)
					oldsigns[i] = signs[i];
								
				if (signcount >= SIGNCOUNT_THRESHOLD) {
					extblocks = (int)(extblocks * SIGNCOUNT_STEP);
					signcount = 0;
				}
			}
			blockno++;			
		}

		if (!wts_blowup) {
			step++;
			angledelta = 0.0;
			
			for (i=0 ; i<chxch ; i++)
				delta[i] = weights[i]-oldweights[i];
			
			change = F77_FUNC(ddot,DDOT)(&chxch,delta,&inc,delta,&inc);
		}
		
/************************* Restart if weights blow up *************************/
		if (wts_blowup) {
			printf("\a");
			step = 0;
			change = nochange;
			wts_blowup = 0;
			blockno = 1;
			extblocks = urextblocks;
			lrate = lrate*DEFAULT_RESTART_FAC;
			F77_FUNC(dcopy,DCOPY)(&chxch,startweights,&inc,weights,&inc);
			zero(chxch,delta);
			zero(chxch,olddelta);

			if (bias) zero(chans,bias);
			
			if (momentum > 0.0) {
				F77_FUNC(dcopy,DCOPY)(&chxch,startweights,&inc,oldweights,&inc);
				F77_FUNC(dcopy,DCOPY)(&chxch,startweights,&inc,prevweights,&inc);
				zero(chxch,prevwtchange);
			}
						
			if (ext) {
				for (i=0 ; i<chans ; i++) oldsigns[i] = -1;
				for (i=0 ; i<nsub ; i++) signs[i] = 1;
				for (i=nsub ; i<chans ; i++) signs[i] = 0;
			}
			
			if (lrate > MIN_LRATE)
				printf("Lowering learning rate to %g and starting again.\n",lrate);
			else
				error("QUITTING - weight matrix may not be invertible!\n");
		}
		else {
/*********************** Print weight update information **********************/
			if (step > 2) {
				epsilon = F77_FUNC(ddot,DDOT)(&chxch,delta,&inc,olddelta,&inc);
				angledelta = acos(epsilon/sqrt(change*oldchange));

				if (verbose) {
					if (!ext)
						printf("step %d - lrate %5f, wchange %7.6f, angledelta %4.1f deg\n",step,lrate,change,DEGCONST*angledelta);
					else {
						for (i=0,j=0 ; i<chans ; i++) j += signs[i];
						printf("step %d - lrate %5f, wchange %7.6f, angledelta %4.1f deg, %d subgauss\n",step,lrate,change,DEGCONST*angledelta,j);
					}
				}
			}
			else
				if (verbose) {
					if (!ext)
						printf("step %d - lrate %5f, wchange %7.6f\n",step,lrate,change);
					else {
						for (i=0,j=0 ; i<chans ; i++) j += signs[i];
						printf("step %d - lrate %5f, wchange %7.6f, %d subgauss\n",step,lrate,change,j);
					}
				}
		}

/**************************** Save current values *****************************/
		F77_FUNC(dcopy,DCOPY)(&chxch,weights,&inc,oldweights,&inc);
		
		if (DEGCONST*angledelta > annealdeg) {
			F77_FUNC(dcopy,DCOPY)(&chxch,delta,&inc,olddelta,&inc);
			lrate = lrate*annealstep;
			oldchange = change;
		}
		else
			if (step == 1) {
				F77_FUNC(dcopy,DCOPY)(&chxch,delta,&inc,olddelta,&inc);
				oldchange = change;
			}

		if (step > 2 && change < nochange)
			step = maxsteps;
		else
			if (change > DEFAULT_BLOWUP)
				lrate = lrate*DEFAULT_BLOWUP_FAC;
	}

	if (bsum) free(bsum);
	if (oldsigns) free(oldsigns);
	if (prevweights) free(prevweights);
	if (prevwtchange) free(prevwtchange);
	if (startweights) free(startweights);
	if (oldweights) free(oldweights);
	if (tmpweights) free(tmpweights);
	if (delta) free(delta);
	if (olddelta) free(olddelta);
	if (kk) free(kk);
	if (old_kk) free(old_kk);
	if (yu) free(yu);
	if (y) free(y);
	if (u) free(u);
	if (pdfperm) free(pdfperm);

#ifndef PROB_WINDOW
	if (dataperm) free(dataperm);
#endif
}

