/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#ifndef __IWL_IDI_H__
#define __IWL_IDI_H__

#include <linux/types.h>

#include "iwl-emulation.h"
#include "iwl-em-idi-rx.h"
#include "iwl-em-idi-tx.h"

/*
 * Define for the IDI channel and function input
*/
#define IWL_IDI_RX_CHANNEL	0
#define IDI_PRIMARY_CHANNEL	1
#define IDI_SECONDARY_CHANNEL	2

/* FLAGS */
#define IWL_IDI_DISABLED	0
#define IWL_IDI_ENABLED		1
#define IWL_IDI_NUM_DIR		2

/* S/G List link calculations */
#define IDI_SGLIST_I_BIT	BIT(1)
#define IDI_SGLIST_L_BIT	BIT(0)
#define IDI_SIZE_OFFSET		2
#define IDI_SIZE_MASK		0x3FFFF
#define IDI_NEXT_MASK		(~(IDI_SGLIST_I_BIT | IDI_SGLIST_L_BIT))

/* trash buffer */
#define HIT_AL_TRASH_BUFFER	(0x100000)

/*****************************************************************************
 *
 * IDI block
 *
 ****************************************************************************/

struct iwl_idi_t;
struct idi_sg_desc;

/*
 * IDI Global Emulation struct
 */
extern struct iwl_idi_t *iwl_idi_em;

/**
 * Descriptor of the AL DMA block
 *
 * @src: [31..0] Source base address
 * @dst: [31..0] Destination base address
 * @size: [31..2] The size of the request transaction in bytes
 *           [1] Interrupt enable
 *           [0] Stop DMA processing
 */
struct iwl_al_dma_desc {
	u32 src;
	u32 dst;
	u32 size;
	u32 dummy;
};

/*
 * This struct holds the data required to run the correct
 * interrupt in a channel and direction as requested by the sg list.
 *
 * @channel - The channel number that the interrupt was called in.
 * @dir - The direction that the interrupt was called in.
 * @data: A void pointer to data that will be sent to the interrupt handler
 * when it is called.
 */
struct iwl_idi_inerrupt_data {
	u32 channel;
	u32 dir;
	void *data;
};

/*
 * Callback for the IDI DBB DMA interrupt. The callback must always be called in
 * an async. manner.
 * In the context of the emulation, need to consider the correct context for
 * calling this function.
 * @chan: the channel that the interrupt was called from.
 * @dir: the direction,  IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 * @data: A void pointer to data that was registered in the set irq handler.
*/
typedef void (*iwl_idi_dbb_dma_irq) (__le32 chan, __le32 dir, void *data);

/**
 * Initialize the idi emulation block.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_init(void);

/**
 * Start the IDI Emulation.
 * Used mainly for debugging options since the IDI is considered started
 * already on init.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_start(void);

/**
 * Start the IDI Emulation
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_stop(void);

/**
 * Free the idi emulation block. The API should be called only after all its
 * operation were gracefully stopped. This mean that the free will fail in
 * case that one of the channel is enabled to transmit data.
 */
void iwl_idi_free(void);

/*
 * Returns the state of the RX engine in the DMA.
 * The state will be changes according to DBB requests.
 *
 * Returns true if the rx dma engine is enabled, false otherwise.
 */
int iwl_idi_rx_status_enabled(void);

/**
 *  Set the interrupt handler for the given channel,  in the given direction
 *  The call should be done by the driver transport layer.
 *
 * @chan: the channel
 * @dir: the direction,  IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 * @irq_handler: the function pointer to the irq_handler
 * @data: A void pointer to data that will be sent to the interrupt handler
 * when it is called.
 * Returns 0 on success,  negative value describing the error otherwise.
 */
int iwl_idi_dbb_set_irq_handler(u32 chan, u32 dir,
				iwl_idi_dbb_dma_irq irq_handler,
				void *data);

/*
 * Calls the interrupt that was registered for the given channel in the given
 * direction and to the required module(ABB/DBB).
 *
 * @channel - The channel number that the interrupt was called in.
 * Returns true if this is a DBB interrupt called, false if this is an AL FW
 * interrupt call.*
 */
void iwl_idi_call_inter(u32 channel);

/**
 *  Set a scatter/gather list in the given channel, in the given direction.
 *  The call should be done only when the <chan, dir> is stopped.
 *
 *  @chan: the channel
 *  @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *  @sg_ll: scatter/gather list
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_dbb_set_sg(u32 chan, u32 dir,
		       struct idi_sg_desc *sg_ll);

/**
 * Start the DBB DMA operation on the given channel, in the given direction.
 * The DMA operation should be handled asyncronuously. A setting of the sg
 * list must be done before calling this API
 *
 *  @chan: the channel
 *  @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_dbb_start(u32 chan, u32 dir);

/**
 *  Stop the DBB DMA the given channel, in the given direction.
 *  The call is syncronuous, and or return the DMA is halted.
 *
 *  @chan: the channel
 *  @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_dbb_stop(u32 chan, u32 dir);

/**
 *  Set the address of the ABB buffer for the given channel, in the given direction.
 *  The call should be done by the driver transport layer.
 *
 *  @chan: the channel
 *  @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *  @addr: dummy address as specified above.
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_abb_config(u32 chan, u32 dir, u32 addr);

/******************************************************************************
 *
 * IDI interrupt support
 *
 ******************************************************************************/

/*
 * Callback for the interrupt handler. The callback should be done in an aync
 * manner.
 * In the context of the emulation, need to consider the correct context for
 * calling this function.
 */
typedef void (*iwl_idi_irq) (unsigned long);

/**
 *  Register interrupt handler for the device.
 *
 * @irq_handler: pointer to the funtion that is the interrupt handler
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_dbb_register_irq(iwl_idi_irq irq_handler);

/**
 *  Unregister interrupt handler for the device.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_dbb_unregister_irq(void);

/**
 *  Enable the interrupt.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_dbb_irq_enable(void);

/**
 *  Disable the IDI interrupt.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_dbb_irq_disable(void);

/**
 *  Simulate interrupt generation from the abb to the dbb.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_dbb_gen_irq(void);

/*****************************************************************************
 *
 * AL DMA block
 *
 ****************************************************************************/

/**
 * Al dma irq handler function pointer.
 * Will be called after registration when the interrupt bit in the sg list will
 * be on, by the DMA machine.
 *
 *@channel: the channel identifier.
 *@dir: the direction identifier.
 */
typedef int (*iwl_al_dma_irq) (void);

/**
 * Configuration of AL DMA channel. Assigns an interrupt handler to
 * the given channel.
 * The call should be done by the AL CPU (probably during initialization).
 *
 * @channel: the channel identifier
 * @irq_handler: the function pointer to the irq_handler
 *					(NULL is a valid option)
 * @data: A void pointer to data that will be sent to the interrupt handler
 * when it is called.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_dma_set_irq_handler(u32 chan, u32 dir,
			       iwl_al_dma_irq irq_handler, void *data);

/**
 * Assign a scatter/gather list to the channel. This interface does not start
 * the DMA operation. The AL DMA channel must be stopped before such a
 * configuration can be made.
 *
 *  @channel: the channel identifier
 *  @sg_ll: scatter/gather list to be processed, the address in the sg_ll
 *	  symbolizes the channel
 *  Returns 0 on success,  negative value describing the error otherwise.
 */
int iwl_al_dma_set_sg(u32 chan, u32 dir,
		      struct iwl_al_dma_desc *sg_ll);

/**
 * Enable the AL DMA channel operation.
 * The DMA operation should be handled async. mode. When the processing
 * is complete,
 * the channel operation should be disabled.
 * The call should be done by the AL CPU or AMFH.
 *
 * @channel: the channel identifier
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_dma_start(u32 chan, u32 dir);

/**
 * Disable the AL DMA channel operation.
 * The DMA operation should be handled sync. mode. The call should be done
 * by the AL CPU
 *
 *  @channel: the channel identifier
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_dma_abort(u32 chan, u32 dir);

/**
 * Stops all the DMA operations which are currently running in the IDI.1
 * On all channels in all directions.
 *
 * Can be used as a reset mechanism for the IDI.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_dma_abort_all(void);

/*
 * Print the IDI error data.
 * This will be called when the emulation error data is called to be
 * printed.
 */
void iwl_em_idi_print_error_data(void);

/*****************************************************************************
 *
 * Covertors from DBB SG LL struct to logical fields
 *
 ****************************************************************************/

static inline u32 iwl_idi_calc_dbb_size(struct idi_sg_desc *sg_ll)
{
	u32 size = le32_to_cpu(sg_ll->size);

	WARN_ON(size >> IDI_SIZE_OFFSET & ~IDI_SIZE_MASK);
	return (size >> IDI_SIZE_OFFSET) * sizeof(void *);
}

static inline void *iwl_idi_calc_dbb_addr(
					struct idi_sg_desc *sg_ll)
{
	return phys_to_virt(le32_to_cpu(sg_ll->base));
}

static inline u32 iwl_idi_calc_dbb_EL_bit(struct idi_sg_desc *sg_ll)
{
	return le32_to_cpu(sg_ll->next) & IDI_SGLIST_L_BIT;
}

static inline void *iwl_idi_calc_dbb_next(struct idi_sg_desc *sg_ll)
{
	u32 next = le32_to_cpu(sg_ll->next);

	if (iwl_idi_calc_dbb_EL_bit(sg_ll))
		return NULL;

	/* The next field cannot be NULL if its not the last */
	WARN_ON(!(le32_to_cpu(sg_ll->next) & IDI_NEXT_MASK));

	return phys_to_virt(next & IDI_NEXT_MASK);
}

static inline u32 iwl_idi_calc_dbb_I_bit(struct idi_sg_desc *sg_ll)
{
	/* Alert if the Interrupt bit is on and the Last link bit is off,
	 * Meaning that we have an interrupt request in the middle of the
	 * linked list, which is a mode which we don't support */
	WARN_ON((le32_to_cpu(sg_ll->next)
		& (IDI_SGLIST_I_BIT | IDI_SGLIST_L_BIT)) == IDI_SGLIST_I_BIT);

	return le32_to_cpu(sg_ll->next) & IDI_SGLIST_I_BIT;
}

/*****************************************************************************
 *
 * Logical channel fields Setters and Getters
 *
 ****************************************************************************/

/*
 * Returns the DBB SG list according to given channel.
 */
struct idi_sg_desc *iwl_idi_get_sg_ll(u32 channel);

/*
 * Sets the SG linked list in the DBB channelsaccording to
 * the given value.
 */
void iwl_idi_set_sg_ll(u32 channel, void *value);

/*
 * Returns the DBB dma channel status pointer according to given channel.
 */
atomic_t *iwl_idi_get_dma_state(u32 channel);

#endif /* __IWL_IDI_H__ */
