/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#include <linux/debugfs.h>

#include "iwl-emulation.h"
#include "iwl-trans.h"
#include "iwl-amfh.h"
#include "iwl-modparams.h"
#include "iwl-idi.h"
#include "iwl-em-idi-tx.h"
#include "iwl-em-idi-rx.h"
#include "iwl-em-lls.h"
#include "iwl-em-sfdb.h"
#include "iwl-em-intr.h"

/* State defines */
#define IWL_AL_DISABLED			0
#define IWL_AL_ENABLED			1

/*
 * IWL Emulation Logger
 */
#define EMULATION_LOG_LEVEL_WARN		KERN_ERR
#define EMULATION_LOG_PREFIX			"[IWL_EMULATION]"

#ifdef __IWL_EMULATION_LOG_ENABLED__
#define EMULATION_TRACE_ENTER \
	IWL_EM_TRACE_ENTER(__EMULATION_LOG_LEVEL_TRACE__, EMULATION_LOG_PREFIX)
#define EMULATION_TRACE_EXIT \
	IWL_EM_TRACE_EXIT(__EMULATION_LOG_LEVEL_TRACE__, EMULATION_LOG_PREFIX)
#define EMULATION_TRACE_EXIT_RET_STR(_ret) \
	IWL_EM_LOG(__EMULATION_LOG_LEVEL_TRACE__, EMULATION_LOG_PREFIX, \
							"<<< "_ret)
#define EMULATION_TRACE_EXIT_RET(_ret) \
	IWL_EM_TRACE_EXIT_RET(__EMULATION_LOG_LEVEL_TRACE__, \
				EMULATION_LOG_PREFIX, _ret)
#define IWL_EMULATION_LOG(fmt, args ...) \
	IWL_EM_LOG(__EMULATION_LOG_LEVEL_DEBUG__, \
		EMULATION_LOG_PREFIX, fmt, ## args)
#define IWL_EMULATION_LOG_HEX_DUMP(msg, p, len) \
	IWL_EM_LOG_HEX_DUMP(msg, p, len)
#else

#define EMULATION_TRACE_ENTER
#define EMULATION_TRACE_EXIT
#define EMULATION_TRACE_EXIT_RET_STR(_ret)
#define EMULATION_TRACE_EXIT_RET(_ret)
#define IWL_EMULATION_LOG(fmt, args ...)
#define IWL_EMULATION_LOG_HEX_DUMP(msg, p, len)

#endif

/* Always print error messages */
#define IWL_EMULATION_LOG_ERR(fmt, args ...) \
	IWL_EM_LOG_ERR(fmt, ## args)

/* Forward declerations */
static int iwl_emulation_add_debugfs(struct iwl_trans *trans);
static void iwl_emulation_rm_dbgfs(struct iwl_trans *trans);
static void iwl_al_auto_init_amfh_config(struct iwl_amfh_config *cfg);

/*
 * Global variables that marks the start address of the al->sram
 * will be used as an offset variable.
 * The second one is the virtual address.
 */
u32 AL_SRAM_ADDRESS;
void *AL_SRAM_VIRTUAL_ADDRESS;

/*
 * The struct that respresents the AL layer in the emulation.
 */
struct iwl_al_t {

	/* Reference to the transport in the driver */
	struct iwl_trans *trans;

	/* Reference to a struct with config data and handlers for the AL em.*/
	struct iwl_al_config al_cfg;

	/* A variable to represent the state of the al (Running/Disabled) */
	atomic_t state;

	/* AUTO MODE */
	/* Auto mode configurations for the AMFH */
	struct iwl_amfh_config amfh_start_cfg;

	/* SRAM for the SFDB */
	dma_addr_t sram;
	void *sram_v;

	/* Emulation debugfs */
	struct dentry *debugfs_dir;
};

/*
  * Global emulation struct.
  * And the pointer to the global emulation struct.
  */
static struct iwl_al_t iwl_al_emu_global;
struct iwl_al_t *iwl_al_em = &iwl_al_emu_global;

/*****************************************************************************
 *
 * System flows
 *
 *****************************************************************************/

/**
 * Initialize the emulation block.
 *
 * @config: emulation module configuration
 * @trans: the transport layer
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_init(struct iwl_al_config *config, struct iwl_trans *trans)
{
	u8 ret_error;

	/* Make sure compilation is for 32Bit arch only
	 * if we use the emulation */
	BUILD_BUG_ON(sizeof(u32) != sizeof(void *));

	EMULATION_TRACE_ENTER;
	IWL_EMULATION_LOG("Initializing the emulation of the AL");

	/* Save references to trans */
	WARN_ON(NULL == trans);
	iwl_al_em->trans = trans;

#ifdef LHP_EMULATION_AMFH_CFG_NON_STATIC
	WARN_ON(NULL == config);
#endif /* LHP_EMULATION_AMFH_CFG_NON_STATIC */

	/* Init the SRAM */
	iwl_al_em->sram_v = dma_alloc_coherent(trans->dev,
					       AL_SRAM_SIZE,
					       &(iwl_al_em->sram),
					       GFP_KERNEL);

	if (!iwl_al_em->sram_v) {
		IWL_EMULATION_LOG_ERR("Cannot allocate memory for the AL SRAM");
		ret_error = -ENOMEM;
		goto exit;
	}
	memset(iwl_al_em->sram_v, 0, AL_SRAM_SIZE);
	AL_SRAM_ADDRESS = iwl_al_em->sram;
	AL_SRAM_VIRTUAL_ADDRESS = iwl_al_em->sram_v;

	/************************************
	 * RX ENGINE
	 ************************************/

	/* Init IDI */
	ret_error = iwl_idi_init();
	if (ret_error) {
		IWL_EMULATION_LOG_ERR("Cannot Initialize IDI struct");
		goto free_sram;
	}
	IWL_EMULATION_LOG("Initializing the IDI - Done.");

	/* Init AMFH */
#ifdef LHP_EMULATION_AMFH_CFG_NON_STATIC
	memcpy(iwl_al_em->al_cfg,
	       config,
	       sizeof(struct iwl_al_config));

#else /* LHP_EMULATION_AMFH_CFG_NON_STATIC not set */

	/* Clean the AMFH configurations */
	memset(&iwl_al_em->al_cfg, 0, sizeof(struct iwl_al_config));

	/* configurations are not given by the transport */
	if (config != NULL) {
		iwl_al_em->al_cfg.iwl_al_irq_handler =
			config->iwl_al_irq_handler;
		iwl_al_em->al_cfg.iwl_al_irq_thread = config->iwl_al_irq_thread;
	} else {
		iwl_al_em->al_cfg.iwl_al_irq_handler = NULL;
		iwl_al_em->al_cfg.iwl_al_irq_thread = NULL;
	}

	iwl_al_em->al_cfg.amfh_config.max_burst_size = 15;
	if (iwlwifi_mod_params.amsdu_size_8K)
		iwl_al_em->al_cfg.amfh_config.rb_size = 8 * 1024;
	else
		iwl_al_em->al_cfg.amfh_config.rb_size = 4 * 1024;
	iwl_al_em->al_cfg.amfh_config.num_rbds = 256;

#endif
	/* Init the AMFH */
	ret_error = iwl_amfh_init(&iwl_al_em->al_cfg.amfh_config,
			iwl_al_em->trans);
	if (ret_error) {
		IWL_EMULATION_LOG_ERR("Cannot Initialize AMFH");
		goto free_idi;
	}
	IWL_EMULATION_LOG("Initializing the AMFH - Done. ");

	/* Init AMFH RX */
	iwl_al_auto_init_amfh_config(&iwl_al_em->amfh_start_cfg);
	IWL_EMULATION_LOG("Initializing the Auto Mode - Done.");

	/************************************
	 * TX ENGINE
	 ************************************/

	/* Init the LLS module */
	ret_error = iwl_lls_em_init(trans);
	if (ret_error) {
		IWL_EMULATION_LOG_ERR("Cannot Initialize the LLS");
		goto free_amfh;
	}

	/* Init the SFDB module */
	ret_error = iwl_sfdb_em_init(trans);
	if (ret_error) {
		IWL_EMULATION_LOG_ERR("Cannot Initialize the SFDB");
		goto free_lls;
	}

	/* Init Debugfs */
	iwl_al_em->debugfs_dir = NULL;

	/* Init inta */
	iwl_inta_em_init(trans);

	/* Init the state to disabled */
	atomic_set(&(iwl_al_em->state), IWL_AL_DISABLED);
	goto exit;

free_lls:
	iwl_lls_em_free();
free_amfh:
	iwl_amfh_free();
free_idi:
	iwl_idi_free();
free_sram:
	dma_free_coherent(iwl_al_em->trans->dev,
			  AL_SRAM_SIZE,
			  iwl_al_em->sram_v,
			  iwl_al_em->sram);
exit:
	EMULATION_TRACE_EXIT;
	return ret_error;
}

/**
 * Free the emulation block.
 * Before calling free the block should be stopped.
 */
void iwl_al_free(void)
{
	EMULATION_TRACE_ENTER;

	iwl_emulation_rm_dbgfs(iwl_al_em->trans);

	/* Check Emulation module state */
	if (IWL_AL_DISABLED != atomic_read(&(iwl_al_em->state)))
		IWL_EMULATION_LOG_ERR("AL not disabled");

	/* free inta handler */
	iwl_inta_em_free();
	IWL_EMULATION_LOG("Freed INTA");

	/************************************
	 * RX ENGINE
	 ************************************/
	iwl_amfh_free();
	IWL_EMULATION_LOG("Freed AMFH");

	iwl_idi_free();
	IWL_EMULATION_LOG("Freed IDI");

	/************************************
	 * TX ENGINE
	 ***********************************/
	iwl_sfdb_em_free();
	IWL_EMULATION_LOG("Freed SFDB");

	iwl_lls_em_free();
	IWL_EMULATION_LOG("Freed LLS");

	/* Remove emulation debugfs */
	iwl_al_em->debugfs_dir = NULL;

	/* Release the SRAM*/
	dma_free_coherent(iwl_al_em->trans->dev,
			  AL_SRAM_SIZE,
			  iwl_al_em->sram_v,
			  iwl_al_em->sram);
	IWL_EMULATION_LOG("Released SRAM memory");

	EMULATION_TRACE_EXIT;
}

/**
 * start  the emulation block. The start should be called only immedialtly
 * after init flow, or after stop flow.
 * Starting the block while it is already started will fail with an error
 * return value.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_start(void)
{
	int ret_error = 0;
	EMULATION_TRACE_ENTER;

	/* Check Emulation module state */
	if (IWL_AL_ENABLED == atomic_read(&(iwl_al_em->state))) {
		IWL_EMULATION_LOG_ERR("AL already enabled on start");
		goto exit;
	}

	/* Init Debugfs*/
	iwl_emulation_add_debugfs(iwl_al_em->trans);

	/* Start the irq emulation */
	if (iwl_al_em->al_cfg.iwl_al_irq_handler != NULL &&
	    iwl_al_em->al_cfg.iwl_al_irq_thread != NULL) {
		iwl_inta_em_start
		(iwl_al_em->trans, iwl_al_em->al_cfg.iwl_al_irq_handler,
		 iwl_al_em->al_cfg.iwl_al_irq_thread);
		IWL_EMULATION_LOG("Start Inta - Done.");
	} else {
		IWL_EMULATION_LOG("Inta not started");
	}

	/************************************
	 * RX ENGINE
	 ************************************/

	/* Start the IDI emulation */
	ret_error  = iwl_idi_start();
	if (ret_error) {
		IWL_EMULATION_LOG_ERR("Failed to start IDI");
		goto exit;
	}
	IWL_EMULATION_LOG("Start IDI - Done.");

	/* Start the AMFH */
	ret_error = iwl_amfh_start(&iwl_al_em->amfh_start_cfg);
	if (ret_error) {
		IWL_EMULATION_LOG_ERR("Starting AMFH Failed");
		goto stop_idi;
	}

	/************************************
	 * TX ENGINE
	 ***********************************/

	/* Start the LLS */
	ret_error = iwl_lls_em_start();
	if (ret_error) {
		IWL_EMULATION_LOG_ERR("Starting LLS Failed");
		goto stop_amfh;
	}

	/* Start the SFDB */
	ret_error = iwl_sfdb_em_start();
	if (ret_error) {
		IWL_EMULATION_LOG_ERR("Starting SFDB Failed");
		goto stop_lls;
	}

	/* Change the state machine */
	atomic_set(&(iwl_al_em->state), IWL_AL_ENABLED);
	goto exit;

stop_lls:
	iwl_lls_em_stop();
stop_amfh:
	iwl_amfh_stop();
stop_idi:
	iwl_idi_stop();
exit:
	EMULATION_TRACE_EXIT;
	return ret_error;

}

/**
 * Stop the emulation block.
 * Once the stop flow is complete the block as should be back in its initial
 * state (as after init and before start was called).
 * Stopping a block when it is already stopped will return immediatly with
 * return value indicating success.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_stop(void)
{
	int ret = 0;

	EMULATION_TRACE_ENTER;

	/* Check Emulation module state */
	if (IWL_AL_ENABLED != atomic_read(&(iwl_al_em->state))) {
		EMULATION_TRACE_EXIT;
		return 0;
	}

	/* Stop the AMFH */
	if (iwl_amfh_stop())
		IWL_EMULATION_LOG_ERR("Stopping AMFH - FAILED");

	/* stop the irq emulation if started */
	if (iwl_al_em->al_cfg.iwl_al_irq_handler != NULL &&
	    iwl_al_em->al_cfg.iwl_al_irq_thread != NULL)
		if (iwl_inta_em_stop())
			IWL_EMULATION_LOG_ERR("Stopping INTA FAILED");

	/* Stop the IDI emulation */
	if (iwl_idi_stop())
		IWL_EMULATION_LOG_ERR("Stopping IDI - FAILED");

	/************************************
	 * TX ENGINE
	 ***********************************/
	/* Stop the SFDB */
	if (iwl_sfdb_em_stop())
		IWL_EMULATION_LOG_ERR("Stopping SFDB - FAILED");

	/* Stop the LLS */
	if (iwl_lls_em_stop())
		IWL_EMULATION_LOG_ERR("Stopping LLS - FAILED");

	/* Set emulation state to disabled*/
	atomic_set(&(iwl_al_em->state), IWL_AL_DISABLED);

	EMULATION_TRACE_EXIT;
	return ret;
}

/**
 * Load a given LMAC FW to the device
 *
 * @image: the image to load
 */
int iwl_al_load_given_ucode(struct fw_img *image)
{
	EMULATION_TRACE_ENTER;

	/* Check Emulation module state */
	if (IWL_AL_ENABLED != atomic_read(&(iwl_al_em->state))) {
		EMULATION_TRACE_EXIT_RET_STR("AL State not enabled");
		return -EHWMISUSE;
	}

	IWL_EMULATION_LOG("FW DOWNLOAD");

	/*
	 * FW Download flow TBD
	 */

	EMULATION_TRACE_EXIT;
	return 0;
}

/******************************************************************************
 *
 * IDI interface operations. These include: SG DMA and Interrupt forwarding.
 * Remote register access will be covered buy the IDI bus emulation.
 *
 *****************************************************************************/

/**
 *  Set the interrupt handler for the given channel, in the given direction
 *  The call should be done by the driver transport layer.
 *
 * @chan: the channel
 * @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 * @irq_handler: the function pointer to the irq_handler
 * @data: A void pointer to data that will be sent to the interrupt handler
 * when it is called.
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_dbb_dma_set_irq_handler(__le32 chan, __le32 dir,
				   iwl_idi_al_dbb_dma_irq irq_handler,
				   void *data)
{
	int ret = 0;
	EMULATION_TRACE_ENTER;

	/* Check Emulation module state */
	if (IWL_AL_ENABLED != atomic_read(&(iwl_al_em->state))) {
		EMULATION_TRACE_EXIT_RET_STR("AL State not enabled");
		return -EHWMISUSE;
	}

	ret = iwl_idi_dbb_set_irq_handler(le32_to_cpu(chan),
					le32_to_cpu(dir),
					irq_handler,
					data);

	EMULATION_TRACE_EXIT;
	return ret;
}

/**
 *  Set a scatter/gather list in the given channel, in the given direction.
 *  The call should be done by the driver transport layer.
 *  The call should be done only when the <chan, dir> is stopped.
 *
 *  @chan: the channel
 *  @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *  @sg_ll: scatter/gather list
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_dbb_set_sg(__le32 chan, __le32 dir,
		      struct idi_sg_desc *sg_ll)
{
	int ret = 0;
	EMULATION_TRACE_ENTER;

	/* Check Emulation module state */
	if (IWL_AL_ENABLED != atomic_read(&(iwl_al_em->state))) {
		EMULATION_TRACE_EXIT_RET_STR("AL State not enabled");
		return -EHWMISUSE;
	}

	ret = iwl_idi_dbb_set_sg(le32_to_cpu(chan),
				le32_to_cpu(dir),
				sg_ll);

	EMULATION_TRACE_EXIT;
	return ret;
}

/**
 *  Start the DBB DMA operation on the given channel, in the given direction.
 *  The DMA operation should be handled asyncronuously.
 *  The call should be done by the driver transport layer after it configured
 *  the sg list
 *
 *  @chan: the channel
 *  @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_dbb_start(__le32 chan, __le32 dir)
{
	int ret = 0;
	EMULATION_TRACE_ENTER;

	/* Check Emulation module state */
	if (IWL_AL_ENABLED != atomic_read(&(iwl_al_em->state))) {
		EMULATION_TRACE_EXIT_RET_STR("AL State not enabled");
		return -EHWMISUSE;
	}

	ret = iwl_idi_dbb_start(le32_to_cpu(chan),
				le32_to_cpu(dir));

	EMULATION_TRACE_EXIT;
	return ret;
}

/**
 *  Stop the DBB DMA the given channel, in the given direction.
 *  The call is syncronuous, and on return the DMA is halted.
 *  The call should be done by the driver transport layer.
 *
 *  @chan: the channel
 *  @dir: the direction, IWL_IDI_DBB_TX or IWL_IDI_DBB_RX
 *
 *  Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_al_dbb_stop(__le32 chan, __le32 dir)
{
	int ret = 0;
	EMULATION_TRACE_ENTER;

	/* Check Emulation module state */
	if (IWL_AL_ENABLED != atomic_read(&(iwl_al_em->state))) {
		EMULATION_TRACE_EXIT_RET_STR("AL State not enabled");
		return -EHWMISUSE;
	}

	ret = iwl_idi_dbb_stop(le32_to_cpu(chan),
				le32_to_cpu(dir));

	EMULATION_TRACE_EXIT;
	return ret;
}

/*
 * Initializes the amfh config struct.
 * This struct configures the handler for the amfh RX.
 */
static void iwl_al_auto_init_amfh_config(struct iwl_amfh_config *cfg)
{
	EMULATION_TRACE_ENTER;

	/* Fill AMFH handlers */
	cfg->push = iwl_idi_rx_em_push_to_rxb;
	cfg->available = iwl_idi_rx_em_available_space;

	EMULATION_TRACE_EXIT;
}

/*****************************************************************************
 *
 * Debugfs Framework functions
 *
 ****************************************************************************/

struct dentry *iwl_emulation_get_debugfs_dir(void)
{
	return iwl_al_em->debugfs_dir;
}

#ifdef CPTCFG_IWLWIFI_DEBUGFS

static int iwl_emulation_add_debugfs(struct iwl_trans *trans)
{

	EMULATION_TRACE_ENTER;

	/* If the debugfs was created, Don't create it again */
	if (iwl_al_em->debugfs_dir) {
		IWL_EMULATION_LOG("Debugfs already created, abort");
		goto exit;
	}

	iwl_al_em->debugfs_dir = debugfs_create_dir(IWL_EMULATION_DEBUGFS_DIR,
						    NULL);

	if (ERR_PTR(-ENODEV) == iwl_al_em->debugfs_dir) {
		IWL_EM_LOG_ERR
			("Failed to create debugfs directory for emulation");
		iwl_al_em->debugfs_dir = NULL;
		return -ENOMEM;
	}

exit:
	EMULATION_TRACE_EXIT;
	return 0;
}

static void iwl_emulation_rm_dbgfs(struct iwl_trans *trans)
{
	if (!iwl_al_em->debugfs_dir)
		return;

	debugfs_remove_recursive(iwl_al_em->debugfs_dir);
	iwl_al_em->debugfs_dir = NULL;
}

#else
static int iwl_emulation_add_debugfs(struct iwl_trans *trans)
{
	return 0;
}

static void iwl_emulation_rm_dbgfs(struct iwl_trans *trans)
{
	return 0;
}

#endif
