/*****************************************************************************
 *
 * Copyright(c) 2013 Intel Corporation. All rights reserved.
 * Intel Corporation does NOT grant any right to use this file in any way.
 *
 ****************************************************************************/

#include <linux/list.h>
#include <linux/kthread.h>
#include <linux/debugfs.h>
#include <linux/export.h>

#include "iwl-drv.h"
#include "iwl-dbgm.h"
#include "iwl-io.h"
#include "iwl-prph.h"
#include "iwl-csr.h"

/* DBGM registers */
#define DBGM_REGS_LOWER_BOUND	(PRPH_BASE + 0xa03c00)
#define DBGM_REGS_UPPER_BOUND	(PRPH_END + 0xa03c60)
#define DBGM_SAMPLE_CTL_REG	(DBGM_REGS_LOWER_BOUND + 0x000)
#define DBGM_END_THRESHOLD_REG	(DBGM_REGS_LOWER_BOUND + 0x004)
#define DBGM_SAMPLE_PERIOD_REG	(DBGM_REGS_LOWER_BOUND + 0x008)
#define DBGM_START_MSK_REG	(DBGM_REGS_LOWER_BOUND + 0x00c)
#define DBGM_SAMPLE_MSK_REG	(DBGM_REGS_LOWER_BOUND + 0x010)
#define DBGM_END_MSK_REG	(DBGM_REGS_LOWER_BOUND + 0x014)
#define DBGM_DATA_SEL_CTL_REG	(DBGM_REGS_LOWER_BOUND + 0x018)
#define DBGM_MC_MSK_REG		(DBGM_REGS_LOWER_BOUND + 0x024)
#define DBGM_BUFF_BASE_ADDR_REG	(DBGM_REGS_LOWER_BOUND + 0x03C)
#define DBGM_BUFF_END_ADDR_REG	(DBGM_REGS_LOWER_BOUND + 0x040)
#define DBGM_BUFF_WRPTR_REG	(DBGM_REGS_LOWER_BOUND + 0x044)
#define DBGM_BUFF_CYCLE_CNT_REG	(DBGM_REGS_LOWER_BOUND + 0x048)

/*
 * Marks whether to drop the newest or oldest logs when the allocated space
 * for debug data is exhausted
 */
enum {
	IWL_DBGM_DROP_LOGS_OLDEST = 0,
	IWL_DBGM_DROP_LOGS_NEWEST = 1
};

/* Marks which user interface is active now - two cannot be active together */
enum {
	IWL_DBGM_USER_IF_NONE = 0,
	IWL_DBGM_USER_IF_DEBUGFS  = 1,
	IWL_DBGM_USER_IF_NETLINK = 2
};

/* Bit positions for the status word */
#define IWL_DBGM_STATUS_BIT_ENABLED	(0)
#define IWL_DBGM_STATUS_BIT_AVAILABLE	(1)
#define IWL_DBGM_STATUS_BIT_START_EARLY	(2)

struct iwl_dbgm_cfg {

	/* matching registers in HW */
	u32 test_ctl;
	u32 data_select_ctl;
	u32 mc_msk;
	u32 sample_ctl;
	u32 end_threshold;
	u32 sample_period;
	u32 start_msk;
	u32 sample_msk;
	u32 end_msk;

	/*
	 * These masks are used to enable legacy monitor behavior,
	 * and should be set before enabling it with a HCMD.
	 */
	__le32 enable_msk;
	__le32 prph_msk;

	/*
	 * Controls the maximal number of pages for log buffers which the driver
	 * can hold until the user reads them. When this limit is reached logs
	 * are being discarded according to the drop policy.
	 */
	u32 logs_list_max_pages;

	/*
	 * When activated in cyclic buffer mode, determines the interval between
	 * each check for new logs in the cyclic buffer.
	 */
	u32 cyclic_sample_rate_msecs;

	/* Determines which logs to discard when driver's logs list is full */
	u32 drop_policy;
};

/*
 * Holds a buffer of log data which can be sent to the user, managed as a list.
 * @list: list struct for management purposes
 * @rx_page: if the data comes from the rx_buffer, this variable
 *           holds the poniter to the rx page. Else, its value
 *           must be NULL.
 * @len: length of the buffer allcoated, not of the whole struct.
 * @buf: this pointer has two assignment options
 *       1. data comes from the cyclic buffer:
 *          buf received the value of data_ptr. allocated buffer of
 *          length 'len' with the debug logs.
 *       2. data comes from the rx buffer:
 *          buf points to the rx cmd's data buffer.
 * @data_ptr: starting address of the data buffer, in case of
 *            cyclic buffer use.
 */
struct iwl_dbgm_log {
	struct list_head list;
	struct page *rx_page;
	u32 len;
	u8 *buf;
	u8 data_ptr[0];
};

/* maximal size for iwl_dbgm_log.buf */
#define MAX_LOG_LEN (u32)(PAGE_SIZE - sizeof(struct iwl_dbgm_log))

/*
 * Holds the context of DBGM module.
 */
struct iwl_dbgm {

	/*
	 * Status word containing these bits:
	 * 0: Whether DBGM is enabled in both device and driver
	 * 1: Whether the device is ready to enable DBGM if requested.
	 * 2: Indicates that DBGM in driver should be enabled early as possible.
	 * The reset are not used
	 */
	unsigned long status;
	atomic_t user_if;
	bool user_requested_disable;

	struct iwl_dbgm_cfg cfg;

	/* In cyclic buffer mode, the thread which polls the device for logs */
	struct task_struct *collector_thread;

	/*
	 * In cyclic buffer mode, last values of these regs in order to
	 * determine if a buffer overflow has happened.
	 */
	u32 last_cyclic_write;
	u32 last_cyclic_wraps;

	/* Cyclic buffer virtual and DMA addresses */
	void *cyclic_buff_addr;
	dma_addr_t cyclic_buff_dma;

	struct list_head used_logs_list;
	struct list_head free_logs_list;
	unsigned int num_logs;
	spinlock_t logs_lock; /* Guards the logs lists and num_logs */

	struct iwl_trans *trans;
	struct iwl_dbgm_ops *ops;
};

/* The debug monitor command id and struct */
#define DEBUG_MONITOR_CMD 0xf2
struct iwl_dbgm_cmd {
	__le32 mon_en_msk;
	__le32 prph_only_mon;
	__le32 tfd_info_comp_msk;
	__le32 tfd_info_comp_val;
} __packed;

/* Writing this value to DBGM_SAMPLE_CTL_REG stops logging */
#define DBGM_STOP_SAMPLE_VAL (0x80001100)

/* Default values for part of the DBGM registers */
#define DBGM_TEST_CTL_DEFAULT_VAL	(0x800000D0)
#define DBGM_DATA_SELECT_CTL_DEFAULT_VAL (0x00000082)
#define DBGM_SAMPLE_CTL_DEFAULT_VAL	(0x80001411)

/* size of cyclic buffer for DRAM mode */
#define DBGM_DRAM_CYCLIC_BUFFER_SIZE		(0x20000)
#define DBGM_DRAM_CYCLIC_BUFFER_PADD		(0x2000)
#define DBGM_DRAM_CYCLIC_BUFFER_TOTAL	\
	(DBGM_DRAM_CYCLIC_BUFFER_SIZE + DBGM_DRAM_CYCLIC_BUFFER_PADD)

static const struct iwl_dbgm_cfg iwl_dbgm_default_cfg = {
	.test_ctl = DBGM_TEST_CTL_DEFAULT_VAL,
	.data_select_ctl = DBGM_DATA_SELECT_CTL_DEFAULT_VAL,
	.mc_msk = 0,
	.sample_ctl = DBGM_SAMPLE_CTL_DEFAULT_VAL,
	.end_threshold = 0,
	.sample_period = 0,
	.start_msk = 0xffffffff,
	.sample_msk = 0,
	.end_msk = 0,
	.enable_msk = 0,
	.prph_msk = 0,
	.logs_list_max_pages = 200,
	.cyclic_sample_rate_msecs = 0x14,
	.drop_policy = IWL_DBGM_DROP_LOGS_OLDEST,
};


/**
 * iwl_dbgm_reset_log - resets the fields of a log node
 */
static void iwl_dbgm_reset_log(struct iwl_dbgm_log *log)
{
	log->len = 0;
	/* rx_page must be NULL as default, it is set to a non-NULL
	   value if the log node is storing a rx buf. When done with
	   the log node, the page must be freed. */
	log->rx_page = NULL;
	/* Default value of buffer pointer is in-strcut data (for
	   the cyclic buffer). If the data comes from the rx fifo
	   then log->buf will point to the data buffer in the rx
	   data strcut */
	log->buf = log->data_ptr;
}

/**
 * iwl_dbgm_cyclic_config - allocates and configures the cyclic buffer
 * in DRAM mode (where we periodically read the new logs from it).
 */
static int iwl_dbgm_cyclic_config(struct iwl_dbgm *dbgm)
{
	dbgm->cyclic_buff_addr = dma_alloc_coherent(dbgm->trans->dev,
		DBGM_DRAM_CYCLIC_BUFFER_TOTAL, &(dbgm->cyclic_buff_dma),
		GFP_KERNEL);
	if (!dbgm->cyclic_buff_addr) {
		IWL_ERR(dbgm->trans, "No memory for DBGM cyclic buffer\n");
		return -ENOMEM;
	}

	/* Updating the buffer address on the device */
	iwl_write_prph(dbgm->trans, DBGM_BUFF_BASE_ADDR_REG,
		       (dbgm->cyclic_buff_dma >> 4));
	iwl_write_prph(dbgm->trans, DBGM_BUFF_END_ADDR_REG,
		       ((dbgm->cyclic_buff_dma +
			 DBGM_DRAM_CYCLIC_BUFFER_SIZE) >> 4));

	dbgm->last_cyclic_wraps = 0;
	dbgm->last_cyclic_write = 0;
	IWL_DEBUG_INFO(dbgm->trans, "DBGM cyclic buf addr=0x%llx, len=0x%x\n",
		       (unsigned long long)dbgm->cyclic_buff_dma,
		       DBGM_DRAM_CYCLIC_BUFFER_SIZE);

	return 0;
}

/*
 * iwl_dbgm_get_empty_log - returns an empty log struct in the pointer given.
 * Behavior depends on three variables: have we reached the maximal pages,
 * drop policy (newest or oldest) and state of the free list.
 * All scenarios are in a big if-else to ensure correct unlocking behavior.
 */
static struct iwl_dbgm_log *iwl_dbgm_get_empty_log(struct iwl_dbgm *dbgm)
{
	struct iwl_dbgm_log *log;

	spin_lock_bh(&dbgm->logs_lock);
	if (dbgm->num_logs >= dbgm->cfg.logs_list_max_pages) {
		/* Maximal umber of pages is reached, we must discard a log */
		if (dbgm->cfg.drop_policy == IWL_DBGM_DROP_LOGS_NEWEST)
			/* This is not an error case, so don't return one */
			log = NULL;
		else if (list_empty(&dbgm->used_logs_list)) {
			/* We should replace the oldest log with a new one,
			but there is no such one - probably an error */
			WARN_ON(1);
			log = ERR_PTR(-EFAULT);
		} else {
			/* taking the oldest log (the first) and moving
			it to the end of the list as new */
			log = list_first_entry(&dbgm->used_logs_list,
				struct iwl_dbgm_log, list);
			list_rotate_left(&dbgm->used_logs_list);
		}
	} else {
		/* we can allocate or have some free pages from before */
		if (list_empty(&dbgm->free_logs_list)) {
			log = (struct iwl_dbgm_log *)
				__get_free_page(GFP_ATOMIC);
			if (log == NULL)
				/* This is an error, we shouldn't get so low
				in pages that we can't allocate anything */
				log = ERR_PTR(-ENOMEM);
			else {
				list_add_tail(&log->list,
					      &dbgm->used_logs_list);
				dbgm->num_logs++;
			}
		} else {
			/* Taking a free page and moving to the used list */
			log = list_first_entry(&dbgm->free_logs_list,
				struct iwl_dbgm_log, list);
			list_move_tail(&log->list,
				       &dbgm->used_logs_list);
			dbgm->num_logs++;
		}
	}
	if (!IS_ERR_OR_NULL(log))
		iwl_dbgm_reset_log(log);

	spin_unlock_bh(&dbgm->logs_lock);
	return log;
}

static inline unsigned int iwl_dbgm_copy_to_log(struct iwl_dbgm_log *log,
	void *src, u32 max_size)
{
	unsigned int to_copy;

	to_copy = min(max_size, MAX_LOG_LEN - log->len);
	memcpy(log->buf + log->len, src, to_copy);
	log->len += to_copy;
	return to_copy;
}

/*
 * iwl_dbgm_read_cyclic_buffer - reads the new data from the cyclic buffer.
 * This function reads the new data from the DRAM cyclic buffer (newer than
 * the last call to this function) and copies it as a new log to the used list.
 * It finds cases of buffer overflow (in the cyclic buffer or copy operation)
 * and puts an indication in the logs.
 */
static int iwl_dbgm_read_cyclic_buffer(struct iwl_dbgm *dbgm)
{
	int overflow;
	unsigned int buf_size, wrap_around_size, num_pages;
	unsigned int i, copied_to_page;
	u32 wr_pos, wraps_cnt, new_wraps;
	void *copy_ptr;
	u32 overflow_indication = 0xDEADBEAF;
	struct iwl_dbgm_log *log = NULL;

	/* Addresses and write pointer are shifted 4 bytes right when written,
	so shift it back in order to get a "byte" position. */
	wr_pos = (iwl_read_prph(dbgm->trans, DBGM_BUFF_WRPTR_REG) << 4)
		- (iwl_read_prph(dbgm->trans,
		DBGM_BUFF_BASE_ADDR_REG) << 4);

	wraps_cnt = iwl_read_prph(dbgm->trans, DBGM_BUFF_CYCLE_CNT_REG);
	new_wraps = wraps_cnt - dbgm->last_cyclic_wraps;
	/* overflow holds the number of bytes which are lost due to it */
	overflow = ((wraps_cnt - dbgm->last_cyclic_wraps - 1) *
		DBGM_DRAM_CYCLIC_BUFFER_SIZE) +
		(wr_pos - dbgm->last_cyclic_write);
	if (overflow < 0)
		overflow = 0;
	if (!new_wraps && wr_pos < dbgm->last_cyclic_write) {
		/* Maybe a HW bug, but sometimes the write pointer goes
		back while the wrap number stays the same. Note it
		but don't panic */
		IWL_WARN(dbgm->trans, "DBGM write ptr went back - ignore.\n");
		return 0;
	}

	/* Here we calculate start address and size of useful data to be copied
	 * from cyclic buffer to internal driver storage.
	 * We distinguish 2 basic cases depending on whether wr_pos surpassed
	 * last_cyclic_write or not.
	 */
	if (!new_wraps ||
	    (new_wraps == 1 && wr_pos < dbgm->last_cyclic_write)) {
		/* In case wr_pos never passed last_cyclic_write, start copying
		from last_cyclic_write until you reach wr_pos. */
		copy_ptr = dbgm->cyclic_buff_addr + dbgm->last_cyclic_write;
		buf_size = wr_pos + (DBGM_DRAM_CYCLIC_BUFFER_SIZE * new_wraps) -
			   dbgm->last_cyclic_write;
		wrap_around_size = !new_wraps ? 0 :
			DBGM_DRAM_CYCLIC_BUFFER_SIZE - dbgm->last_cyclic_write;
	} else {
		/* In case wr_pos surpassed last_cyclic_write, start copying
		from wr_pos until full wrap-around. */
		copy_ptr = dbgm->cyclic_buff_addr + wr_pos;
		buf_size = DBGM_DRAM_CYCLIC_BUFFER_SIZE;
		wrap_around_size = DBGM_DRAM_CYCLIC_BUFFER_SIZE - wr_pos;
	}

	if (buf_size == 0)
		return 0;
	/* +4 for overflow indication if needed */
	num_pages = DIV_ROUND_UP(buf_size + sizeof(u32), MAX_LOG_LEN);

	for (i = 0; i < num_pages; i++) {
		log = iwl_dbgm_get_empty_log(dbgm);
		/* NULL pointer is a valid response, meaning that we don't get
		memory according to policy, and we should stop processing. */
		if (IS_ERR_OR_NULL(log))
			return PTR_ERR(log);

		if (wrap_around_size) {
			copied_to_page = iwl_dbgm_copy_to_log(log, copy_ptr,
				wrap_around_size);
			wrap_around_size -= copied_to_page;
			buf_size -= copied_to_page;
			copy_ptr += copied_to_page;
			if (wrap_around_size == 0)
				copy_ptr = dbgm->cyclic_buff_addr;
			else
				continue;
		}

		/* get here after all wrap around data is read */
		copied_to_page = iwl_dbgm_copy_to_log(log, copy_ptr,
			buf_size);
		copy_ptr += copied_to_page;
		buf_size -= copied_to_page;
	}

	dbgm->last_cyclic_wraps = wraps_cnt;
	dbgm->last_cyclic_write = wr_pos;

	/* This is a sanity check, to see if the logs we have copied were
	  * overwritten while we were copying them.
	  * Only relevant if no overflow before. */
	if (!overflow) {
		wr_pos = (iwl_read_prph(dbgm->trans, DBGM_BUFF_WRPTR_REG) << 4)
			- (iwl_read_prph(dbgm->trans,
			DBGM_BUFF_BASE_ADDR_REG) << 4);
		wraps_cnt = iwl_read_prph(dbgm->trans,
			DBGM_BUFF_CYCLE_CNT_REG);
		overflow = ((wraps_cnt - dbgm->last_cyclic_wraps - 1) *
			DBGM_DRAM_CYCLIC_BUFFER_SIZE) +
			(wr_pos - dbgm->last_cyclic_write);
	}
	if (overflow > 0)
		iwl_dbgm_copy_to_log(log, &overflow_indication, 4);

	return 0;
}

/*
 * iwl_dbgm_write_config - writes configuration to the device.
 */
static int iwl_dbgm_write_config(struct iwl_dbgm *dbgm)
{
	struct iwl_trans *trans = dbgm->trans;
	struct iwl_dbgm_cfg *cfg = &(dbgm->cfg);
	int ret;

	iwl_write_prph(trans, DBGM_SAMPLE_CTL_REG, DBGM_STOP_SAMPLE_VAL);

	ret = iwl_dbgm_cyclic_config(dbgm);
	if (ret)
		return ret;

	iwl_write_prph(trans, HBUS_TARG_TEST_REG, cfg->test_ctl);
	iwl_write_prph(trans, DBGM_DATA_SEL_CTL_REG, cfg->data_select_ctl);
	iwl_write_prph(trans, DBGM_MC_MSK_REG, cfg->mc_msk);
	iwl_write_prph(trans, DBGM_SAMPLE_MSK_REG, cfg->sample_msk);
	iwl_write_prph(trans, DBGM_SAMPLE_CTL_REG, cfg->sample_ctl);
	iwl_write_prph(trans, DBGM_START_MSK_REG, cfg->start_msk);
	iwl_write_prph(trans, DBGM_END_MSK_REG, cfg->end_msk);
	iwl_write_prph(trans, DBGM_END_THRESHOLD_REG, cfg->end_threshold);
	iwl_write_prph(trans, DBGM_SAMPLE_PERIOD_REG, cfg->sample_period);

	return 0;
}

/*
 * iwl_dbgm_collector_thread - thread function for DRAM mode.
 */
static int iwl_dbgm_collector_thread(void *data)
{
	struct iwl_dbgm *dbgm = (struct iwl_dbgm *)data;
	int ret = 0;

	while (!kthread_should_stop()) {
		ret = iwl_dbgm_read_cyclic_buffer(dbgm);
		if (ret) {
			IWL_ERR(dbgm->trans, "DBGM thread term with err=%d\n",
				ret);
			return ret;
		}
		msleep(dbgm->cfg.cyclic_sample_rate_msecs);
	}

	/* If the user requested to stop the thread, then we should read the
	last logs for the user. Otherwise it's not important */
	if (dbgm->user_requested_disable)
		ret = iwl_dbgm_read_cyclic_buffer(dbgm);
	return ret;
}

/*
 * iwl_dbgm_enable - enables DBGM by sending a HCMD.
 * This function writes the configuration currently stored to the device,
 * clear any data from previous sessions and enables the device monitor.
 */
static int iwl_dbgm_enable(struct iwl_dbgm *dbgm)
{
	int ret;
	struct iwl_dbgm_cmd cmd = {
		.mon_en_msk = dbgm->cfg.enable_msk,
		.prph_only_mon = dbgm->cfg.prph_msk,
		.tfd_info_comp_msk = 0,
		.tfd_info_comp_val = 0,
	};
	struct iwl_host_cmd cmd_out = {
		.id = DEBUG_MONITOR_CMD,
		.len = { sizeof(struct iwl_dbgm_cmd), },
		.flags = CMD_SYNC,
		.data = { &cmd, },
	};

	/* if there are leftovers logs from a previous session - clear */
	spin_lock_bh(&dbgm->logs_lock);
	list_splice_tail_init(&dbgm->used_logs_list, &dbgm->free_logs_list);
	dbgm->num_logs = 0;
	spin_unlock_bh(&dbgm->logs_lock);

	ret = iwl_dbgm_write_config(dbgm);
	if (ret) {
		IWL_ERR(dbgm->trans, "DBGM failed write config to dev [%d]\n",
			ret);
		return ret;
	}

	dbgm->collector_thread = kthread_create(iwl_dbgm_collector_thread,
		dbgm, "iwlwifi DBGM");
	if (IS_ERR(dbgm->collector_thread)) {
		IWL_ERR(dbgm->trans, "DBGM failed to create a thread.\n");
		goto enable_err;
	}

	ret = dbgm->ops->send_cmd(dbgm->trans->op_mode, &cmd_out);
	if (ret)
		goto enable_err;
	wake_up_process(dbgm->collector_thread);
	set_bit(IWL_DBGM_STATUS_BIT_ENABLED, &dbgm->status);

	IWL_DEBUG_INFO(dbgm->trans, "DBGM enable success\n");
	IWL_DEBUG_INFO(dbgm->trans, "DBGM enable msk=0x%x prph msk=0x%x\n",
		       cmd.mon_en_msk, cmd.prph_only_mon);
	return 0;

enable_err:
	dma_free_coherent(dbgm->trans->dev, DBGM_DRAM_CYCLIC_BUFFER_TOTAL,
			  dbgm->cyclic_buff_addr, dbgm->cyclic_buff_dma);
	return ret;
}

/*
 * iwl_dbgm_disable - disables the DBGM monitor.
 * Disables the HW monitor by writing a "stop sample" value.
 * If the user requested the stop, then he/she probably wants to see the
 * last logs as well, so pass this parameter to the thread.
 * Otherwise, the disable flow should be quicker.
 * Note that it doesn't free the pages of logs, as they might be needed
 * by the user.
 */
static int iwl_dbgm_disable(struct iwl_dbgm *dbgm, bool user_request)
{
	int ret;

	clear_bit(IWL_DBGM_STATUS_BIT_ENABLED, &dbgm->status);
	iwl_write_prph(dbgm->trans, DBGM_SAMPLE_CTL_REG,
		       DBGM_STOP_SAMPLE_VAL);
	dbgm->user_requested_disable = user_request;

	/* The thread may exit due to error before */
	if (!task_is_dead(dbgm->collector_thread))
		ret = kthread_stop(dbgm->collector_thread);
	else
		ret = dbgm->collector_thread->exit_code;

	WARN_ON(ret); /* should exit cleanly, but it's not that important */

	/* At this stage the kernel thread should have collected all logs */
	dma_free_coherent(dbgm->trans->dev, DBGM_DRAM_CYCLIC_BUFFER_TOTAL,
			  dbgm->cyclic_buff_addr, dbgm->cyclic_buff_dma);
	IWL_DEBUG_INFO(dbgm->trans, "DBGM was disabled by %s\n",
		       (user_request ? "user" : "driver flow"));
	return ret;
}

#ifdef CPTCFG_IWLWIFI_DEBUGFS

#define MAX_CONFIG_LINE_SIZE 1024

/*
 * iwl_dbgm_get_full_log - extracts the oldest valid log from the list.
 * This function is used to extract logs when the user reads them.
 * It checks that the log is smaller than max_size before extracting,
 * in order to prevent extracting and re-inserting of the log after that.
 */
static struct iwl_dbgm_log *iwl_dbgm_get_full_log(struct iwl_dbgm *dbgm,
	unsigned int max_size)
{
	struct iwl_dbgm_log *log;

	spin_lock_bh(&dbgm->logs_lock);
	if (list_empty(&dbgm->used_logs_list)) {
		log = NULL;
	} else {
		log = list_first_entry(&dbgm->used_logs_list,
			struct iwl_dbgm_log, list);
		if (log->len > max_size) {
			log = NULL;
		} else {
			list_del(&log->list);
			dbgm->num_logs--;
		}
	}
	spin_unlock_bh(&dbgm->logs_lock);
	return log;
}

/*
 * iwl_dbgm_free_log - puts a log in the free list.
 */
static void iwl_dbgm_free_log(struct iwl_dbgm *dbgm, struct iwl_dbgm_log *log)
{
	if (log->rx_page)
		put_page(log->rx_page);

	spin_lock_bh(&dbgm->logs_lock);
	list_add(&log->list, &dbgm->free_logs_list);
	spin_unlock_bh(&dbgm->logs_lock);
}

static int iwl_dbgm_validate_config(struct iwl_dbgm *dbgm)
{
	/* TODO: implement */
	return 0;
}

/*
 * iwl_dbgm_parse_config - updates config, and return 0/1 according to
 * the "enabled" flag (default 0) or -1 on error.
 */
static int iwl_dbgm_parse_config(struct iwl_dbgm *dbgm, char *buf)
{
	const char *seps = " =\n\t";
	char *param, *value_str;
	int value;
	int enable = 0;

	while ((param = strsep(&buf, seps))) {
		if (strlen(param) == 0)
			continue;
		value_str = strsep(&buf, seps);
		if (strlen(value_str) == 0)
			continue;
		sscanf(value_str, "%x", &value);
		if (!strcmp(param, "test_ctl"))
			dbgm->cfg.test_ctl = value;
		else if (!strcmp(param, "data_select_ctl"))
			dbgm->cfg.data_select_ctl = value;
		else if (!strcmp(param, "mc_msk"))
			dbgm->cfg.mc_msk = value;
		else if (!strcmp(param, "sample_ctl"))
			dbgm->cfg.sample_ctl = value;
		else if (!strcmp(param, "end_threshold"))
			dbgm->cfg.end_threshold = value;
		else if (!strcmp(param, "sample_period"))
			dbgm->cfg.sample_period = value;
		else if (!strcmp(param, "start_msk"))
			dbgm->cfg.start_msk = value;
		else if (!strcmp(param, "end_msk"))
			dbgm->cfg.end_msk = value;
		else if (!strcmp(param, "logs_list_max_pages"))
			dbgm->cfg.logs_list_max_pages = value;
		else if (!strcmp(param, "cyclic_sample_rate_msecs"))
			dbgm->cfg.cyclic_sample_rate_msecs = value;
		else if (!strcmp(param, "drop_policy"))
			dbgm->cfg.drop_policy = value;
		else if (!strcmp(param, "enable_msk"))
			dbgm->cfg.enable_msk = cpu_to_le32(value);
		else if (!strcmp(param, "prph_msk"))
			dbgm->cfg.prph_msk = cpu_to_le32(value);
		else if (!strcmp(param, "enable"))
			enable = value;
		else /* not a valid parameter name */
			return -EINVAL;
	}

	if (iwl_dbgm_validate_config(dbgm))
		return -1;
	return enable;
}

/*
 * iwl_dbgm_debugfs_config_read - returns the user a config line.
 * This line shows the current configuration and can be used to restore it.
 */
static ssize_t iwl_dbgm_debugfs_config_read(struct file *file,
	char __user *user_buf,
	size_t count, loff_t *ppos)
{
	struct iwl_dbgm *dbgm = file->private_data;
	char *out_str;
	int out_len, ret;

	if (IS_ERR_OR_NULL(dbgm))
		return -ENOMSG;

	out_str = kzalloc(MAX_CONFIG_LINE_SIZE, GFP_KERNEL);
	if (!out_str)
		return -ENOMEM;

	out_len = snprintf(out_str, MAX_CONFIG_LINE_SIZE,
		"test_ctl=0x%x data_select_ctl=0x%x "
		"mc_msk=0x%x sample_ctl=0x%x end_threshold=0x%x "
		"sample_period=0x%x start_msk=0x%x end_msk=0x%x "
		"logs_list_max_pages=0x%x "
		"cyclic_sample_rate_msecs=0x%x drop_policy=0x%x "
		"enable_msk=0x%x prph_msk=0x%x enable=%d\n",
		dbgm->cfg.test_ctl, dbgm->cfg.data_select_ctl,
		dbgm->cfg.mc_msk, dbgm->cfg.sample_ctl,
		dbgm->cfg.end_threshold, dbgm->cfg.sample_period,
		dbgm->cfg.start_msk, dbgm->cfg.end_msk,
		dbgm->cfg.logs_list_max_pages,
		dbgm->cfg.cyclic_sample_rate_msecs, dbgm->cfg.drop_policy,
		le32_to_cpu(dbgm->cfg.enable_msk),
		le32_to_cpu(dbgm->cfg.prph_msk),
		(test_bit(IWL_DBGM_STATUS_BIT_ENABLED, &dbgm->status) ? 1 : 0));

	ret = simple_read_from_buffer(user_buf, count, ppos, out_str,
		out_len);
	kfree(out_str);
	return ret;
}

/*
 * iwl_dbgm_debugfs_log_read - returns as much logs as possible to the user.
 * The logs are returned in binary format until the buffer of debugfs is
 * exhausted. If a log can't be copied to user (due to general error, not
 * a size problem) it is totally discarded and lost.
 */
static ssize_t iwl_dbgm_debugfs_log_read(struct file *file,
			char __user *user_buf,
			size_t count, loff_t *ppos)
{
	struct iwl_dbgm *dbgm = file->private_data;
	struct iwl_dbgm_log *log;
	ssize_t total = 0, ret = 0;

	if (IS_ERR_OR_NULL(dbgm))
		return -EFAULT;

	if (atomic_read(&dbgm->user_if) != IWL_DBGM_USER_IF_DEBUGFS) {
		ret = atomic_add_unless(&dbgm->user_if,
			IWL_DBGM_USER_IF_DEBUGFS, IWL_DBGM_USER_IF_NETLINK);
		/* returns true if the add really succeeded */
		if (!ret)
			return -EFAULT;
	}

	log = iwl_dbgm_get_full_log(dbgm, count);
	while (log) {
		ret = copy_to_user(user_buf + total, log->buf, log->len);
		if (ret) {
			iwl_dbgm_free_log(dbgm, log);
			break;
		} else {
			total += log->len;
			iwl_dbgm_free_log(dbgm, log);
			log = iwl_dbgm_get_full_log(dbgm, count - total);
		}
	}

	*ppos = total;
	return total;
}

/*
 * iwl_dbgm_debugfs_config_write - updates the configuration of DBGM.
 * parses the user string for parameters in the form of:
 * "par1[ =]val1[ =]par2[ =]val2 ... " (spaces or equation marks).
 * writes the configuration and enables/disables DBGM if requested to.
 */
static ssize_t iwl_dbgm_debugfs_config_write(struct file *file,
		const char __user *user_buf,
		size_t count, loff_t *ppos)
{
	struct iwl_dbgm *dbgm = file->private_data;
	char buf[512];
	int ret;
	int buf_size;

	if (IS_ERR_OR_NULL(dbgm))
		return -EFAULT;

	if (atomic_read(&dbgm->user_if) != IWL_DBGM_USER_IF_DEBUGFS) {
		ret = atomic_add_unless(&dbgm->user_if,
			IWL_DBGM_USER_IF_DEBUGFS, IWL_DBGM_USER_IF_NETLINK);
		/* returns true if the add really succeeded */
		if (!ret)
			return -EFAULT;
	}

	memset(buf, 0, sizeof(buf));
	buf_size = min(count, sizeof(buf) -  1);
	if (copy_from_user(buf, user_buf, buf_size))
		return -EFAULT;

	ret = iwl_dbgm_parse_config(dbgm, buf);
	if (ret < 0)
		return ret;

	if (ret == 0) {
		clear_bit(IWL_DBGM_STATUS_BIT_START_EARLY, &dbgm->status);
		if (test_bit(IWL_DBGM_STATUS_BIT_ENABLED, &dbgm->status))
			ret = iwl_dbgm_disable(dbgm, true);
	} else if (ret == 1) {
		/* Marks that DBGM should start automatically when possible */
		set_bit(IWL_DBGM_STATUS_BIT_START_EARLY, &dbgm->status);
		if (test_bit(IWL_DBGM_STATUS_BIT_AVAILABLE, &dbgm->status) &&
		    !test_bit(IWL_DBGM_STATUS_BIT_ENABLED, &dbgm->status))
			ret = iwl_dbgm_enable(dbgm);
	}

	/* If an error occured, return it. */
	if (ret)
		return ret;
	else
		return count;
}

static const struct file_operations iwl_dbgm_debugfs_config_ops = {
	.write = iwl_dbgm_debugfs_config_write,
	.read = iwl_dbgm_debugfs_config_read,
	.open = simple_open,
	.llseek = generic_file_llseek,
};

static const struct file_operations iwl_dbgm_debugfs_log_ops = {
	.read = iwl_dbgm_debugfs_log_read,
	.open = simple_open,
	.llseek = generic_file_llseek,
};

int iwl_dbgm_debugfs_register(struct iwl_dbgm *dbgm, struct dentry *dir)
{
	struct dentry *dir_dbgm;

	/*
	 * init may fail but driver will still continue its flows,
	 * so we need to check if this is the case and exit silently.
	 */
	if (IS_ERR_OR_NULL(dbgm))
		return 0;

	dir_dbgm = debugfs_create_dir("dbgm", dir);
	if (!dir_dbgm)
		return -EFAULT;

	if (!debugfs_create_file("config", S_IWUSR | S_IRUSR, dir_dbgm,
				 dbgm, &iwl_dbgm_debugfs_config_ops))
		return -EFAULT;

	if (!debugfs_create_file("log", S_IRUSR, dir_dbgm,
				 dbgm, &iwl_dbgm_debugfs_log_ops))
		return -EFAULT;
	return 0;
}
IWL_EXPORT_SYMBOL(iwl_dbgm_debugfs_register);

#endif /* CPTCFG_IWLWIFI_DEBUGFS */

struct iwl_dbgm *iwl_dbgm_init(struct iwl_trans *trans,
			       struct iwl_dbgm_ops *ops)
{
	struct iwl_dbgm *dbgm;

	dbgm = kzalloc(sizeof(struct iwl_dbgm), GFP_KERNEL);
	if (!dbgm)
		return ERR_PTR(-ENOMEM);

	/*
	 * Only the minimum initializations are done here, in order to save
	 * memory if DBGM is not enabled.
	 */
	memcpy(&dbgm->cfg, &iwl_dbgm_default_cfg, sizeof(struct iwl_dbgm_cfg));
	INIT_LIST_HEAD(&dbgm->used_logs_list);
	INIT_LIST_HEAD(&dbgm->free_logs_list);
	spin_lock_init(&dbgm->logs_lock);

	dbgm->trans = trans;
	dbgm->ops = ops;

	IWL_DEBUG_INFO(trans, "DBGM initialized successfully\n");
	return dbgm;
}
IWL_EXPORT_SYMBOL(iwl_dbgm_init);

void iwl_dbgm_free(struct iwl_dbgm *dbgm)
{
	struct iwl_dbgm_log *log, *next;
	struct iwl_trans *trans;

	/*
	 * init may fail but driver will still continue its flows,
	 * so we need to check if this is the case and exit silently.
	 */
	if (!dbgm)
		return;
	trans = dbgm->trans;

	spin_lock_bh(&dbgm->logs_lock);
	list_for_each_entry_safe(log, next, &dbgm->used_logs_list, list)
		free_page((unsigned long)log);
	list_for_each_entry_safe(log, next, &dbgm->free_logs_list, list)
		free_page((unsigned long)log);
	spin_unlock_bh(&dbgm->logs_lock);
	kfree(dbgm);
	IWL_DEBUG_INFO(trans, "DBGM was freed\n");
	return;
}
IWL_EXPORT_SYMBOL(iwl_dbgm_free);

void iwl_dbgm_set_available(struct iwl_dbgm *dbgm, bool available)
{
	/*
	 * init may fail but driver will still continue its flows,
	 * so we need to check if this is the case and exit silently.
	 */
	if (!dbgm)
		return;

	if (!available &&
	    test_bit(IWL_DBGM_STATUS_BIT_ENABLED, &dbgm->status)) {
		/* Marking DBGM to start early next time, because it's not a
		user-requested disable - we want to return to this state */
		set_bit(IWL_DBGM_STATUS_BIT_START_EARLY, &dbgm->status);
		iwl_dbgm_disable(dbgm, false);
	}

	if (available &&
	    test_bit(IWL_DBGM_STATUS_BIT_START_EARLY, &dbgm->status))
		iwl_dbgm_enable(dbgm);

	if (available)
		set_bit(IWL_DBGM_STATUS_BIT_AVAILABLE, &dbgm->status);
	else
		clear_bit(IWL_DBGM_STATUS_BIT_AVAILABLE, &dbgm->status);
	IWL_DEBUG_INFO(dbgm->trans, "DBGM was set to %s\n", (available ?
		"available" : "unavaialble"));
	return;
}
IWL_EXPORT_SYMBOL(iwl_dbgm_set_available);

int iwl_dbgm_log_rx_debug_msg(struct iwl_dbgm *dbgm,
		 struct iwl_rx_cmd_buffer *rxb,
		 struct iwl_device_cmd *cmd)
{
	struct iwl_rx_packet *pkt;
	struct iwl_dbgm_log *log;

	log = iwl_dbgm_get_empty_log(dbgm);
	/* NULL pointer is a valid response, meaning that we don't get
	memory according to policy, and we should stop processing. */
	if (IS_ERR_OR_NULL(log))
		return PTR_ERR(log);

	log->rx_page = rxb_steal_page(rxb);
	pkt = rxb_addr(rxb);

	log->buf = pkt->data;
	log->len = (le32_to_cpu(pkt->len_n_flags) & FH_RSCSR_FRAME_SIZE_MSK)
		    - sizeof(struct iwl_cmd_header);

	return 0;
}
IWL_EXPORT_SYMBOL(iwl_dbgm_log_rx_debug_msg);
