#include "iwl-io.h"
#include "iwl-fh.h"
#include "iwl-op-mode.h"
#include "idi_al.h"
#include "idi_internal.h"
#include "idi_utils.h"
#include "idi_tx.h"
#include "idi_tx_policy.h"
#include "iwl-agn-hw.h"
#include "idi_host_csr.h"

/* FIXME: remove this include after AMPDU support is added */
#include "iwl-modparams.h"

/* FIXME: need to abstract out TX command (once we know what it looks like) */
#include "dvm/commands.h"

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
#include "iwl-idi.h"
#endif

#ifdef CPTCFG_IWLWIFI_UT_TX_STORE
#include "UT/iwl-idi-ut.h"
#endif

#define IWL_IDI_POOL_TFD_SIZE_ORDER (6)
#define IWL_IDI_POOL_TB_SIZE_ORDER (8)

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
#define IWL_IDI_TX_QUEUE_PTR_MASK (IDI_TX_TFD_PER_Q - 1)
#define INC_TFD_IDX(_idx) ((_idx) = ((_idx)+1) & IWL_IDI_TX_QUEUE_PTR_MASK)

#define IWL_GET_SG_IDX_FROM_IDI_CHAN(_chan) \
	(_chan == IDI_SECONDARY_CHANNEL ? TX_SG_LP_CH : TX_SG_HP_CH)

#define IWL_GET_IDI_CHAN_FROM_SG_IDX(_trans_tx, _idx) \
	(_trans_tx->sg_list_to_chan[_idx])

#else
/* the LLS determine the SCD write-ptr by the tfd_idx, hence it shold be
 * warped to the TFD queue size */
#define INC_TFD_IDX(_idx) ((_idx) = ((_idx)+1) & (TFD_QUEUE_SIZE_MAX - 1))

#define IWL_GET_SG_IDX_FROM_IDI_CHAN(_chan) \
	(_chan == IDI_PRIMARY_CHANNEL ? TX_SG_HP_CH : TX_SG_LP_CH)

#define IWL_GET_IDI_CHAN_FROM_SG_IDX(_trans_tx, _idx) \
	(_trans_tx->sg_list_to_chan[_idx] - 1)
#endif

/* FIXME: which part of the SRAM can be used in FW download? */
#define IWL_SRAM_SIZE_FOR_FW (30 * 1024)
#define IWL_IDI_TXBU_CAPACITY (IWL_IDI_MAX_TXC_COUNT_IN_TXBU *\
			       IDI_TX_PAYLOAD_PAGE_SIZE)
#define IWL_IDI_NUM_FH_REG_TO_CONFIG 6
#define IWL_IDI_FH_REG_SIZE sizeof(u32)

#define IWL_IDI_TXH_RES_NAME	"txh fifo"
#define IWL_IDI_TXL_RES_NAME	"txl fifo"

static struct iwl_trans *g_trans;

static const u8 idi_tx_sg_to_chan[] = {
	IDI_SECONDARY_CHANNEL,
	IDI_PRIMARY_CHANNEL,
};

static const u8 idi_tx_queue_to_sg_list[] = {
	TX_SG_LP_CH,
	TX_SG_LP_CH,
	TX_SG_LP_CH,
	TX_SG_LP_CH,
	TX_SG_LP_CH,
	TX_SG_LP_CH,
	TX_SG_LP_CH,
	TX_SG_LP_CH,
	TX_SG_LP_CH,
	TX_SG_HP_CH,
	TX_SG_LP_CH,
	TX_SG_LP_CH,
	TX_SG_LP_CH,
	TX_SG_LP_CH,
	TX_SG_LP_CH,
	TX_SG_LP_CH,
};

static void iwl_idi_tx_fix_sg_list(struct iwl_trans *trans, u32 idx,
				   bool is_fw_download);

/**
 * iwl_idi_tx_dma_arm - start IDI dma transaction
 * @trans_tx: tx transport
 * @chan: channel index
 */
int iwl_idi_tx_dma_arm(struct iwl_idi_trans_tx *trans_tx, u8 chan)
{
#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans *trans = IWL_TRANS_TX_GET_TRANS(trans_tx);
#endif
	int ret = 0;

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	u8 idi_chan = IWL_GET_IDI_CHAN_FROM_SG_IDX(trans_tx, chan);

	ret = iwl_al_dbb_set_sg(cpu_to_le32(idi_chan),
				cpu_to_le32(IWL_IDI_DBB_TX),
				(struct idi_sg_desc *)
				trans_tx->sg_list[chan].addr);
	if (ret)
		goto error;

	ret = iwl_al_dbb_start(cpu_to_le32(idi_chan),
			       cpu_to_le32(IWL_IDI_DBB_TX));
	if (ret)
		goto error;

#else
	ret = idi_async_write(trans_tx->trans_idi->pdev,
			      trans_tx->transaction[chan]);

	if (ret) {
		IWL_ERR(trans, "idi_async_write returns with error, ret = %d\n",
			ret);
		goto error;
	}
#endif

error:
	/* FIXME: Need to define error handling */
	return ret;
}

static inline void iwl_idi_tx_stop_ch(struct iwl_trans *trans, u8 channel)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	int ret;
	struct iwl_idi_trans_tx *trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);
	u8 chan = IWL_GET_IDI_CHAN_FROM_SG_IDX(trans_tx, channel);

	ret = iwl_al_dbb_stop(cpu_to_le32(chan), cpu_to_le32(IWL_IDI_DBB_TX));
	if (ret)
		IWL_ERR(trans, "Tx DBB stop failed, ret = %d\n", ret);
#endif
}

/**
 * iwl_idi_tx_stop_device - stop tx, don't free resources yet.
 */
void iwl_idi_tx_stop(struct iwl_trans *trans)
{
#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
	struct iwl_trans_idi *trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);
#endif
	BUG_ON(trans == NULL);

	iwl_idi_tx_stop_ch(trans, TX_SG_HP_CH);
	iwl_idi_tx_stop_ch(trans, TX_SG_LP_CH);

#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
	/* flash IDI TX channels */
	idi_peripheral_flush(trans_idi->pdev,
			     IDI_PRIMARY_CHANNEL | IDI_TX_CHANNEL);
	idi_peripheral_flush(trans_idi->pdev,
			     IDI_SECONDARY_CHANNEL | IDI_TX_CHANNEL);
#endif
	iwl_slv_tx_stop(trans);
}

static inline u16 iwl_idi_get_txbu_hdr_size(u8 txcs_num)
{
	return sizeof(struct iwl_idi_txbu_header) +
		txcs_num * sizeof(struct iwl_idi_txc_header);
}

static inline u16 iwl_idi_tx_get_txbu_hdr_size(u8 txcs_num)
{
	return iwl_idi_get_txbu_hdr_size(txcs_num) +
		sizeof(struct iwl_idi_tfd);
}

static inline struct iwl_idi_txc_header *iwl_idi_tx_get_txcs_from_txbu(
					struct iwl_idi_txbu_header *txbu)
{
	return (struct iwl_idi_txc_header *)((u8 *)txbu +
			sizeof(struct iwl_idi_txbu_header));
}

static inline struct iwl_idi_tfd *iwl_idi_tx_get_tfd_from_txbu(
					struct iwl_idi_txbu_header *txbu,
					u8 txcs_num)
{
	return (struct iwl_idi_tfd *)
		((u8 *)txbu + sizeof(struct iwl_idi_txbu_header) +
		txcs_num * sizeof(struct iwl_idi_txc_header));
}

static inline int iwl_idi_set_sg_desc(struct iwl_idi_trans_tx *trans_tx,
				      u8 chan, int idx, dma_addr_t phys_addr,
				      u32 len)
{
	struct idi_sg_desc *sg_desc;

	sg_desc = (struct idi_sg_desc *)trans_tx->sg_list[chan].addr;

	sg_desc[idx].base = cpu_to_le32(phys_addr);
	sg_desc[idx].size = cpu_to_le32(len);
	return 0;
}

static int iwl_idi_build_fw_chunk(struct iwl_idi_trans_tx *trans_tx,
				  struct iwl_idi_dma_ptr *hdrs,
				  u32 byte_cnt,
				  dma_addr_t phy_addr,
				  int *hdr_pos)
{
	struct iwl_idi_txbu_header *txbu;
	struct iwl_idi_txc_header *txcs;
	dma_addr_t dma_addr;
	u8 txcs_num, j, leftover_txcs;
	u16 txc_len, seq_num, leftover_bytes;
	u32 sram_dest, txbu_total_len;
	int txcs_in_chunk, txbus_num;
	int sg_idx;
	int i;

	txcs_in_chunk = DIV_ROUND_UP(byte_cnt, IDI_TX_PAYLOAD_PAGE_SIZE);
	leftover_bytes = byte_cnt % IDI_TX_PAYLOAD_PAGE_SIZE;

	txbus_num = DIV_ROUND_UP(txcs_in_chunk, IWL_IDI_MAX_TXC_COUNT_IN_TXBU);
	leftover_txcs = txcs_in_chunk % IWL_IDI_MAX_TXC_COUNT_IN_TXBU;

	sg_idx = trans_tx->sg_list_meta[TX_SG_LP_CH].used_count;
	seq_num = trans_tx->txbu_seq_num;

	sram_dest = IDI_AL_SFDB_BASE_ADDR;
	*hdr_pos = 0;

	for (i = 0; i < txbus_num; i++) {
		txbu = (struct iwl_idi_txbu_header *)((u8 *)hdrs->addr +
						      *hdr_pos);
		txcs = iwl_idi_tx_get_txcs_from_txbu(txbu);
		txcs_num = (i == (txbus_num - 1) && leftover_txcs != 0) ?
			leftover_txcs : IWL_IDI_MAX_TXC_COUNT_IN_TXBU;
		txbu_total_len = 0;
		txc_len = IDI_TX_PAYLOAD_PAGE_SIZE;

		for (j = 0; j < txcs_num; j++) {
			/* image sections are aligned, no need to add pa, pb */
			txcs->dest_and_flags = cpu_to_le32(sram_dest);
			if (i == (txbus_num - 1) && j == (txcs_num - 1) &&
			    leftover_bytes != 0) {
				/* len of the last TXC in the last TXBU */
				txc_len = leftover_bytes;
			}
			txcs->len = cpu_to_le16(txc_len);
			sram_dest += txc_len;
			txbu_total_len += txc_len;
			txcs++;
		}

		txbu->signature = cpu_to_le16(IWL_IDI_TXBU_SIGNATURE);
		txbu->txc_count = txcs_num;
		txbu->seq_num = cpu_to_le16(seq_num++);

		/* first descriptor for headers, second for data */
		dma_addr = (dma_addr_t)(hdrs->dma + *hdr_pos);

		iwl_idi_set_sg_desc(trans_tx, TX_SG_LP_CH, sg_idx++, dma_addr,
				    iwl_idi_get_txbu_hdr_size(txcs_num));
		iwl_idi_set_sg_desc(trans_tx, TX_SG_LP_CH, sg_idx++, phy_addr,
				    txbu_total_len);

		/* check if S/G ll is full */
		if (WARN_ON(sg_idx > IWL_IDI_TX_SG_FRAGS_MAX))
			return -ENOSR;

		phy_addr += txbu_total_len;
		*hdr_pos += iwl_idi_get_txbu_hdr_size(txcs_num);
	}

	trans_tx->sg_list_meta[TX_SG_LP_CH].used_count = sg_idx;
	trans_tx->txbu_seq_num = seq_num;

	return 0;
}

static inline void iwl_idi_set_fh_txc(struct iwl_idi_txc_header *txc, u32 dest)
{
#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	/* emulation work-around in order to provide access
	 * to non-SRAM registers in the AL */
	dest = IWL_AL_ADD_SRAM_SIZE(dest);
#endif
	txc->dest_and_flags = cpu_to_le32(dest);
	txc->len = cpu_to_le16(IWL_IDI_FH_REG_SIZE);
}

static int iwl_idi_config_fh(struct iwl_idi_trans_tx *trans_tx,
			     struct iwl_idi_dma_ptr *headers,
			     u32 byte_cnt, u32 dst_addr, int *hdr_pos)
{
	struct iwl_idi_txbu_header *txbu;
	struct iwl_idi_txc_header *txcs;
	dma_addr_t dma_addr;

	int sg_idx;
	u32 phy_addr, len;
	__le32 *data;

	phy_addr = IWL_IDI_AL_SRAM_DMA_ADDRESS;

	txbu = (struct iwl_idi_txbu_header *)((u8 *)headers->addr + *hdr_pos);
	txbu->signature = cpu_to_le16(IWL_IDI_TXBU_SIGNATURE);
	txbu->txc_count = IWL_IDI_NUM_FH_REG_TO_CONFIG;
	txbu->seq_num = cpu_to_le16(trans_tx->txbu_seq_num);
	trans_tx->txbu_seq_num++;

	txcs = iwl_idi_tx_get_txcs_from_txbu(txbu);
	data = (__le32 *)(txcs + txbu->txc_count); /* FH registers content */

	iwl_idi_set_fh_txc(txcs++, FH_TCSR_CHNL_TX_CONFIG_REG(FH_SRVC_CHNL));
	data[0] = cpu_to_le32(FH_TCSR_TX_CONFIG_REG_VAL_DMA_CHNL_PAUSE);

	iwl_idi_set_fh_txc(txcs++, FH_SRVC_CHNL_SRAM_ADDR_REG(FH_SRVC_CHNL));
	data[1] = cpu_to_le32(dst_addr);

	/* phy_addr is the address inside device's SRAM, so it's u32 */
	iwl_idi_set_fh_txc(txcs++, FH_TFDIB_CTRL0_REG(FH_SRVC_CHNL));
	data[2] = cpu_to_le32(phy_addr & FH_MEM_TFDIB_DRAM_ADDR_LSB_MSK);

	/* it's always 0 as long as we use an address inside SRAM */
	iwl_idi_set_fh_txc(txcs++, FH_TFDIB_CTRL1_REG(FH_SRVC_CHNL));
	data[3] = cpu_to_le32((iwl_get_dma_hi_addr(phy_addr)
		   << FH_MEM_TFDIB_REG1_ADDR_BITSHIFT) | byte_cnt);

	iwl_idi_set_fh_txc(txcs++, FH_TCSR_CHNL_TX_BUF_STS_REG(FH_SRVC_CHNL));
	data[4] = cpu_to_le32(1 << FH_TCSR_CHNL_TX_BUF_STS_REG_POS_TB_NUM |
		  1 << FH_TCSR_CHNL_TX_BUF_STS_REG_POS_TB_IDX |
		  FH_TCSR_CHNL_TX_BUF_STS_REG_VAL_TFDB_VALID);

	iwl_idi_set_fh_txc(txcs, FH_TCSR_CHNL_TX_CONFIG_REG(FH_SRVC_CHNL));
	data[5] = cpu_to_le32(FH_TCSR_TX_CONFIG_REG_VAL_DMA_CHNL_ENABLE |
		  FH_TCSR_TX_CONFIG_REG_VAL_DMA_CREDIT_DISABLE |
		  FH_TCSR_TX_CONFIG_REG_VAL_CIRQ_HOST_ENDTFD);

	len = iwl_idi_get_txbu_hdr_size(IWL_IDI_NUM_FH_REG_TO_CONFIG) +
		IWL_IDI_NUM_FH_REG_TO_CONFIG * IWL_IDI_FH_REG_SIZE;

	sg_idx = trans_tx->sg_list_meta[TX_SG_LP_CH].used_count;
	/* check if S/G ll is full */
	if (WARN_ON(sg_idx >= IWL_IDI_TX_SG_FRAGS_MAX))
		return -ENOSR;

	dma_addr = (dma_addr_t)(headers->dma + *hdr_pos);
	iwl_idi_set_sg_desc(trans_tx, TX_SG_LP_CH, sg_idx, dma_addr, len);

	trans_tx->sg_list_meta[TX_SG_LP_CH].used_count++;
	return 0;
}

static int iwl_idi_load_fw_chunk(struct iwl_trans *trans,
				 u32 dst_addr, struct iwl_idi_dma_ptr *headers,
				 dma_addr_t phy_addr, u32 byte_cnt)
{
	struct iwl_idi_trans_tx *trans_tx;
	struct idi_sg_desc *sg_desc;
	int sg_idx, hdr_pos;
	int ret, time_left;

	trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);

	/* Build burst's TXBUs */
	ret = iwl_idi_build_fw_chunk(trans_tx, headers, byte_cnt,
				     phy_addr, &hdr_pos);
	if (ret)
		return ret;
	ret = iwl_idi_config_fh(trans_tx, headers, byte_cnt, dst_addr,
				&hdr_pos);
	if (ret)
		return ret;

	/* seal burst and arm DMA */
	trans_tx->ucode_write_complete = false;

	sg_idx = trans_tx->sg_list_meta[TX_SG_LP_CH].used_count;
	sg_desc = (struct idi_sg_desc *)trans_tx->
						 sg_list[TX_SG_LP_CH].addr;
	sg_desc[sg_idx - 1].next |= cpu_to_le32(IWL_IDI_SG_LIST_END);

	ret = iwl_idi_tx_dma_arm(trans_tx, TX_SG_LP_CH);
	if (ret) {
		IWL_ERR(trans, "[%d] %s: iwl_idi_tx_dma_arm failed, ret %d\n",
			get_current()->tgid, __func__, ret);
		return ret;
	}

	IWL_DEBUG_FW(trans, "uCode chunk being loaded...\n");

	/* Wait until the FH will finish to copy this chunk to the device SRAM
	 * FIXME: timeout value
	 */
	time_left = wait_event_timeout(trans_tx->ucode_write_waitq,
				       trans_tx->ucode_write_complete, 5 * HZ);
	if (!time_left) {
		IWL_ERR(trans, "Could not load uCode chunk\n");
		return -ETIMEDOUT;
	}

	/* cleanup */
	iwl_idi_tx_fix_sg_list(trans, TX_SG_LP_CH, true);
	return 0;
}

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
/**
 * FW download is done using the LLS, with TXBUs in basic mode.This function
 * splits the FW image to chunks that can fit in the AL SRAM and constructs
 * TXBUs which describes them. FH configuration is done on the same burst,
 * by adding an additional TXBU which configures the service channel of the
 * FH to copy the data to the LMAC SRAM. Upon completion, the FH fires
 * interrupt which indicates that all HW components are done and the current
 * chunk has arrived to its position in LMAC SRAM.
 * @trans - the transport
 * @fw_img - the firmware image to load
 */
int iwl_idi_load_given_ucode(struct iwl_trans *trans, const struct fw_img *img)
{
	struct iwl_idi_trans_tx *trans_tx;
	struct iwl_idi_dma_ptr chunk_headers;
	int ret, i, j, chunks_num;
	int max_txbu_hdr_size, txbus_per_chunk;
	u8 *buf_v_addr, *cur_data;
	dma_addr_t buf_p_addr;
	u32 dst_addr, byte_cnt;
	u32 leftover;

	BUG_ON(trans == NULL);
	trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);
	ret = 0;

	max_txbu_hdr_size =
		iwl_idi_get_txbu_hdr_size(IWL_IDI_MAX_TXC_COUNT_IN_TXBU);
	txbus_per_chunk = DIV_ROUND_UP(IWL_SRAM_SIZE_FOR_FW,
				       IWL_IDI_TXBU_CAPACITY);

	/* FIXME: Consider implementing a fallback for smaller size in case
	 * the allocation fails.
	 */
	buf_v_addr = dma_alloc_coherent(trans->dev, IWL_SRAM_SIZE_FOR_FW,
					&buf_p_addr, GFP_KERNEL);
	if (!buf_v_addr) {
		IWL_ERR(trans, "alloc failure for dma buffer.\n");
		return -ENOMEM;
	}

	/* Allocate room for TXBU headers and FH configuration data of each
	 * chunk. There are "txbus_per_chunk" data TXBUs, plus one for FH
	 * configuration. In addition, this memory will also hold the content
	 * of FH configuration TXBU (registers values).
	 */
	chunk_headers.size = max_txbu_hdr_size * (txbus_per_chunk + 1) +
		IWL_IDI_NUM_FH_REG_TO_CONFIG * IWL_IDI_FH_REG_SIZE;

	chunk_headers.addr = dma_alloc_coherent(trans->dev, chunk_headers.size,
						&chunk_headers.dma, GFP_KERNEL);
	if (!chunk_headers.addr) {
		IWL_ERR(trans, "alloc failure.\n");
		goto free;
	}

	/* DBB DMA expects 4-aligned addresses; this is guaranteed by
	 * dma_alloc_coherent.
	 */

	for (i = 0; i < IWL_UCODE_SECTION_MAX; i++) {
		if (!img->sec[i].data)
			break;

		cur_data = (u8 *)img->sec[i].data;
		dst_addr = img->sec[i].offset;

		if (ALIGN(img->sec[i].len, 4) != img->sec[i].len) {
			ret = -EFAULT;
			goto free;
		}

		chunks_num = DIV_ROUND_UP(img->sec[i].len,
					  IWL_SRAM_SIZE_FOR_FW);
		leftover = img->sec[i].len % IWL_SRAM_SIZE_FOR_FW;

		for (j = 0; j < chunks_num; j++) {
			IWL_DEBUG_FW(trans,
				     "processing uCode chunk %d from section %d...\n",
				     j, i);
			memset(chunk_headers.addr, 0, chunk_headers.size);
			byte_cnt = (j == (chunks_num - 1) && leftover != 0) ?
				leftover : IWL_SRAM_SIZE_FOR_FW;

			memcpy(buf_v_addr, cur_data, byte_cnt);
			ret = iwl_idi_load_fw_chunk(trans, dst_addr,
						    &chunk_headers, buf_p_addr,
						    byte_cnt);
			if (ret)
				goto free;
			cur_data += byte_cnt;
			dst_addr += byte_cnt;
		}
	}
free:
	if (chunk_headers.addr)
		dma_free_coherent(trans->dev, chunk_headers.size,
				  chunk_headers.addr, chunk_headers.dma);

	if (buf_v_addr)
		dma_free_coherent(trans->dev, IWL_SRAM_SIZE_FOR_FW,
				  buf_v_addr, buf_p_addr);

	return ret;
}

#else

int iwl_idi_load_given_ucode(struct iwl_trans *trans,
			     const struct fw_img *image)
{
	u32 target;
	int i, ret = 0;
	struct fw_desc *section;

	if (image == NULL)
		return -EINVAL;

	/* stall both target access channels in the AL */
	idi_al_write(trans, AMFH_TG1_BUS_WAIT_EN_REG, AMFH_TG1_BUS_STALL);
	idi_al_write(trans, AMFH_TG2_BUS_WAIT_EN_REG, AMFH_TG2_BUS_STALL);

	/* Iterate over all the FW sections and load each section */
	for (i = 0; i < IWL_UCODE_SECTION_MAX; i++) {
		if (!image->sec[i].data)
			break;

		section = &image->sec[i];
		target = section->offset;

		/* section.len in bytes and iwl_trans_idi_write_mem expects len in dwords */
		ret = iwl_trans_write_mem(trans, target, section->data, (section->len / sizeof(u32)));
		if (ret)
			goto unstall_bus;
	}

	/* release LMAC ARC from reset */
	idi_al_write(trans, POWER_CFG_W1C_REG, POWER_CFG_ARC_REST_REQ_MSK);

unstall_bus:
	idi_al_write(trans, AMFH_TG1_BUS_WAIT_EN_REG, AMFH_TG1_BUS_IGNORE);
	idi_al_write(trans, AMFH_TG2_BUS_WAIT_EN_REG, AMFH_TG2_BUS_IGNORE);

	return ret;
}

#endif

/**
 * iwl_idi_tx_sg_init - initialize list of sg descriptors for a given channel.
 */
static int iwl_idi_tx_sg_init(struct iwl_idi_trans_tx *trans_tx, int chan)
{
	struct iwl_trans *trans;
	size_t sg_byte_size;
	int i, ret;
	struct idi_sg_desc *cur_virt;
	dma_addr_t cur_dma;

	BUG_ON(trans_tx == NULL);

	trans = IWL_TRANS_TX_GET_TRANS(trans_tx);

	/* Data Tx always requires 2 DMA descriptors per frame and HCmd needs
	 * at most IWL_MAX_CMD_TBS_PER_TFD + 1 descriptors. Each bus aggregation
	 * could contain up to sg_max_txbus elements. Allocating now the maximal
	 * needed size.
	 * FIXME: In case of separate channel for hcmds only - need to adjust
	 * the size accordingly.
	 */
	sg_byte_size = IWL_IDI_TX_SG_FRAGS_MAX *
			sizeof(struct idi_sg_desc);

	ret = iwl_idi_alloc_sg_list(trans,
			&trans_tx->sg_list[chan], sg_byte_size);
	if (unlikely(ret)) {
		IWL_WARN(trans,
			 "%s (%d): alloc_sg_list failed, ret %d\n",
			 __func__, __LINE__, ret);
		return ret;
	}
	IWL_INFO(trans, "%s: sg_list offset %d\n", __func__,
		 trans_tx->sg_list[chan].align_offset);

	trans_tx->sg_list_meta[chan].max_txbus = IWL_IDI_TX_CH_SG_SIZE;
	trans_tx->sg_list_meta[chan].used_count = 0;
	clear_bit(chan, &trans_tx->sg_list_loaded);

	cur_virt = (struct idi_sg_desc *)trans_tx->sg_list[chan].addr;
	cur_dma = trans_tx->sg_list[chan].dma;

	/* set up next of each descriptor to point on the
	 * following descriptor */
	for (i = 0; i < IWL_IDI_TX_SG_FRAGS_MAX; i++) {
		cur_virt->next = cpu_to_le32(cur_dma +
					sizeof(struct idi_sg_desc));
		cur_dma += sizeof(struct idi_sg_desc);
		cur_virt++;
	}

	return 0;
}

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
void iwl_idi_tx_irq_handler(__le32 chan, __le32 dir, void *ctx)
{
	struct iwl_trans *trans = (struct iwl_trans *)ctx;
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);
	struct iwl_idi_trans_tx *trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);
	u32 sg_list_idx;

	sg_list_idx = IWL_GET_SG_IDX_FROM_IDI_CHAN(le32_to_cpu(chan));
	spin_lock_bh(&trans_tx->slv_tx.mem_rsrc_lock);

	iwl_idi_tx_fix_sg_list(trans, sg_list_idx, false);
	clear_bit(sg_list_idx, &trans_tx->sg_list_loaded);

	spin_unlock_bh(&trans_tx->slv_tx.mem_rsrc_lock);

	queue_work(trans_slv->policy_wq, &trans_slv->policy_trigger);
}

#else

void iwl_idi_tx_complete(struct idi_transaction *transaction)
{
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(g_trans);
	struct iwl_idi_trans_tx *trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(g_trans);
	struct iwl_trans *trans = IWL_TRANS_TX_GET_TRANS(trans_tx);
	u32 sg_list_idx;

	sg_list_idx = IWL_GET_SG_IDX_FROM_IDI_CHAN(transaction->idi_xfer.channel_opts);

	spin_lock_bh(&trans_tx->slv_tx.mem_rsrc_lock);

	iwl_idi_tx_fix_sg_list(g_trans, sg_list_idx, false);
	clear_bit(sg_list_idx, &trans_tx->sg_list_loaded);

	spin_unlock_bh(&trans_tx->slv_tx.mem_rsrc_lock);

	queue_work(trans_slv->policy_wq, &trans_slv->policy_trigger);
}

int iwl_idi_tx_set_channel_config(struct idi_peripheral_device *pdev)
{
	struct idi_channel_config conf = {0};
	struct idi_resource *idi_res;
	struct resource *res;
	int ret, i;
	struct {
		const char *name;
		unsigned channel;
	} iwl_idi_tx_ch_map[] = {
		{ .name = IWL_IDI_TXL_RES_NAME, .channel = IDI_PRIMARY_CHANNEL},
		{ .name = IWL_IDI_TXH_RES_NAME,
		  .channel = IDI_SECONDARY_CHANNEL},
	};

	idi_res = &pdev->resources;

	for (i = 0; i < ARRAY_SIZE(iwl_idi_tx_ch_map); i++) {
		const char *ch_name = iwl_idi_tx_ch_map[i].name;
		unsigned ch_id = iwl_idi_tx_ch_map[i].channel;

		dev_err(&pdev->device, "Getting resources %s for ch %d\n",
			ch_name, ch_id);
		res = idi_get_resource_byname(idi_res, IORESOURCE_MEM, ch_name);

		if (!res) {
			dev_err(&pdev->device, "Failed to get resource %s\n",
				ch_name);
			return -EINVAL;
		}

		conf.tx_or_rx = 1;
		conf.priority = IDI_NORMAL_PRIORITY;
		conf.channel_opts = IDI_TX_EARLY_IRQ | ch_id;

		conf.dst_addr = res->start;
		conf.hw_fifo_size = resource_size(res);

		ret = idi_set_channel_config(pdev, &conf);
		if (ret) {
			dev_err(&pdev->device,
				"Failed in idi_set_channel_config %s, err %d\n",
								ch_name, ret);
			return ret;
		}
	}

	return 0;
}

static int iwl_idi_tx_alloc_transaction(struct iwl_idi_trans_tx *trans_tx,
					int idx)
{
	struct iwl_trans *trans = IWL_TRANS_TX_GET_TRANS(trans_tx);
	struct idi_channel_config conf = {0};
	struct idi_resource *idi_res;
	struct idi_transaction *idi_trans;
	struct resource *res;
	int ret;

	idi_res = &trans_tx->trans_idi->pdev->resources;


	idi_trans = idi_alloc_transaction(GFP_KERNEL);
	if (!idi_trans) {
		IWL_ERR(trans, "%s failed to allocate IDI transaction\n",
			__func__);
		return -ENOMEM;
	}

	idi_trans->complete = iwl_idi_tx_complete;
	idi_trans->context = trans;
	idi_trans->idi_xfer.channel_opts = IDI_TX_EARLY_IRQ;
	idi_trans->idi_xfer.channel_opts |= (idx == TX_SG_LP_CH) ?
						IDI_PRIMARY_CHANNEL :
							IDI_SECONDARY_CHANNEL;

	/* The API of IDI driver will be updated, should be phys addr */
	idi_trans->idi_xfer.desc = trans_tx->sg_list[idx].dma;
	trans_tx->transaction[idx] = idi_trans;
	return 0;
}

#endif /* CPTCFG_IWLWIFI_IDI_OVER_PCI */

/**
* iwl_idi_tx_sg_init_active - initilailize active S/G lists. A list is
* considered active if it appears in trans_tx->idi_tx_sg_to_chan
* only works if each channel appears only once (which is the case here).
*/
static int iwl_idi_tx_sg_init_active(struct iwl_idi_trans_tx *trans_tx)
{
	struct iwl_trans *trans;
	int i, j, ret;

	BUG_ON(trans_tx->sg_list_to_chan == NULL);

	for (i = 0; i < TX_SG_CH_MAX; i++) {
		ret = iwl_idi_tx_sg_init(trans_tx, i);
		if (unlikely(ret))
			goto error;

#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
		ret = iwl_idi_tx_alloc_transaction(trans_tx, i);
		if (unlikely(ret))
			goto error;
#endif
	}

	return 0;
error:
	trans = IWL_TRANS_TX_GET_TRANS(trans_tx);

	for (j = 0; j < i; j++) {
		iwl_idi_free_sg_list(trans, &trans_tx->sg_list[j]);
#ifndef CPTCFG_IWLWIFI_IDI_OVER_PCI
		if (trans_tx->transaction[i])
			idi_free_transaction(trans_tx->transaction[i]);
#endif
	}
	IWL_ERR(trans, "%s failed, ret %d\n", __func__, ret);
	return ret;
}

void iwl_idi_tx_free_txbu_mem(struct iwl_trans *trans, void **data)
{
	struct iwl_idi_trans_tx *trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);
	struct iwl_idi_tx_reclaim_info *reclaim_info =
		(struct iwl_idi_tx_reclaim_info *)*data;
	int i;

	for (i = 0; i < reclaim_info->blocks_count; i++) {
		if (dma_unmap_len(&reclaim_info->map_data[i], len) > 0) {
			dma_unmap_single(trans->dev,
					 dma_unmap_addr(
						&reclaim_info->map_data[i],
						mapping),
					 dma_unmap_len(
						&reclaim_info->map_data[i],
						len),
					 DMA_TO_DEVICE);
		} else {
			/* mapping fragments is done sequentially */
			break;
		}
	}

	if (reclaim_info) {
		kmem_cache_free(trans_tx->reclaim_info_pool, reclaim_info);
		reclaim_info = NULL;
	}
}

void iwl_idi_tx_clean_txbu(struct iwl_trans *trans, void *data)
{
	struct iwl_idi_trans_tx *trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);
	struct iwl_idi_tx_reclaim_info *reclaim_info =
		(struct iwl_idi_tx_reclaim_info *)data;
	struct iwl_idi_tfd *tfd;
	int i, ret;

	tfd = iwl_idi_tx_get_tfd_from_txbu(reclaim_info->txbu,
					   reclaim_info->txbu->txc_count);
	spin_lock_bh(&trans_tx->slv_tx.mem_rsrc_lock);
	for (i = 0; i < tfd->num_tbs; i++) {
		ret = iwl_slv_al_mem_pool_free(&trans_tx->slv_tx,
					       &trans_tx->slv_tx.tb_pool,
					       tfd->tbs[i].tb_idx);
		if (ret)
			IWL_WARN(trans, "%s failed to free tb pool resources\n",
				 __func__);
	}

	ret = iwl_slv_al_mem_pool_free(&trans_tx->slv_tx,
				       &trans_tx->slv_tx.tfd_pool,
				       reclaim_info->txbu->lut_value);
	if (ret)
		IWL_WARN(trans, "%s failed to free tfd pool resources\n",
			 __func__);
	spin_unlock_bh(&trans_tx->slv_tx.mem_rsrc_lock);

	iwl_idi_tx_free_txbu_mem(trans, (void **)&reclaim_info);
}

/**
* iwl_idi_tx_init - initialize IDI transport Tx (called from start device).
*/
int iwl_idi_tx_init(struct iwl_trans *trans)
{
	struct iwl_idi_trans_tx *trans_tx;
	int ret;

	/* ensure TB size is power of 2 */
	BUILD_BUG_ON_NOT_POWER_OF_2(IDI_TX_PAYLOAD_PAGE_SIZE);
	/* ensure Tx queue size is power of 2 */
	BUILD_BUG_ON_NOT_POWER_OF_2(IDI_TX_TFD_PER_Q);

	BUG_ON(trans == NULL);
	BUG_ON(trans->trans_specific == NULL);

	trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);
	trans_tx->trans_idi = IWL_TRANS_GET_IDI_TRANS(trans);

	g_trans = trans;

	ret = iwl_slv_init(trans);
	if (ret)
		goto error;

	spin_lock_init(&trans_tx->slv_tx.mem_rsrc_lock);
	trans_tx->queue_to_sg_list = idi_tx_queue_to_sg_list;
	trans_tx->sg_list_to_chan = idi_tx_sg_to_chan;

	/* initialize policy data, it doesn't run anything yet */
	trans_tx->txbus_added_policy[TX_SG_HP_CH] = 0;
	trans_tx->txbus_added_policy[TX_SG_LP_CH] = 0;

	ret = iwl_idi_tx_sg_init_active(trans_tx);
	if (ret)
		goto error_free;

	ret = iwl_slv_al_mem_pool_init(&trans_tx->slv_tx.tfd_pool,
				       IDI_TX_TFD_PER_Q);
	if (ret)
		goto error_free;

	ret = iwl_slv_al_mem_pool_init(&trans_tx->slv_tx.tb_pool,
				       IDI_TX_TBS_IN_POOL);
	if (ret)
		goto error_free;

	trans_tx->reclaim_info_pool =
		kmem_cache_create("iwl_idi_reclaim_info",
				  sizeof(struct iwl_idi_tx_reclaim_info),
				  sizeof(void *), 0, NULL);
	if (unlikely(!trans_tx->reclaim_info_pool)) {
		ret = -ENOMEM;
		goto error_free;
	}

#ifdef CPTCFG_IWLWIFI_IDI_OVER_PCI
	{
	int i;
	u8 chan;

	for (i = 0; i < TX_SG_CH_MAX; i++) {
		chan = IWL_GET_IDI_CHAN_FROM_SG_IDX(trans_tx, i);
		iwl_al_dbb_dma_set_irq_handler
				(cpu_to_le32(chan),
				 cpu_to_le32(IWL_IDI_DBB_TX),
				 iwl_idi_tx_irq_handler, (void *)trans);
	}
	}
#endif
	trans_tx->txbu_seq_num = 0;
	init_waitqueue_head(&trans_tx->ucode_write_waitq);

	/* While IDI doesn't support AMPDU, need to force that
	 * TX aggregations are disabled.
	 */
	if (!(iwlwifi_mod_params.disable_11n & IWL_DISABLE_HT_TXAGG)) {
		IWL_DEBUG_TX(trans, "No AMPDU support - force TX agg disable");
		iwlwifi_mod_params.disable_11n |= IWL_DISABLE_HT_TXAGG;
	}

	return 0;

error_free:
	/* free all resources;
	 * it can handle the case of zeroed but not initialized resource
	 */

	/* FIXME: verify all un-allocated resources are zeroed */
	iwl_idi_tx_free(trans);

error:
	IWL_ERR(trans, "%s failed, ret %d\n", __func__, ret);
	return ret;
}

/**
 * iwl_idi_tx_fix_sg_list - resets descriptors of a S/G list up to used_count
 * @trans - the transport
 * @idx - the index of the required S/G list in the array of sg lists.
 * @is_fw_download - whether this function called from load ucode flow
 */
static void iwl_idi_tx_fix_sg_list(struct iwl_trans *trans, u32 idx,
				   bool is_fw_download)
{
	struct iwl_idi_trans_tx *trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);
	struct idi_sg_desc *sg_desc;
	u32 next;
	int i;

	if (!is_fw_download)
		lockdep_assert_held(&trans_tx->slv_tx.mem_rsrc_lock);

	sg_desc = (struct idi_sg_desc *)trans_tx->sg_list[idx].addr;
	next = trans_tx->sg_list[idx].dma;

	for (i = 0; i < trans_tx->sg_list_meta[idx].used_count; i++) {
		sg_desc[i].base = 0;
		sg_desc[i].size = 0;

		/* reset the next, even though it shouldn't be changed by
		 * IDI DMA except the last used one.
		 */
		if (i < IWL_IDI_TX_SG_FRAGS_MAX - 1) {
			next += sizeof(struct idi_sg_desc);
			sg_desc[i].next = cpu_to_le32(next);
		} else {
			/* set also INT in the last desc to prevent
			 * a possiblity of a stuck list.
			 */
			sg_desc[i].next = cpu_to_le32(IWL_IDI_SG_LIST_END);

			if (!is_fw_download)
				sg_desc[i].next |=
					cpu_to_le32(IWL_IDI_SG_LIST_INT);
		}
	}

	trans_tx->sg_list_meta[idx].used_count = 0;
}

/**
* iwl_idi_tx_free - free all the resources, assumes tx is stopped.
*/
void iwl_idi_tx_free(struct iwl_trans *trans)
{
	struct iwl_idi_trans_tx *trans_tx;

	BUG_ON(trans == NULL);

	trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);
	BUG_ON(trans_tx == NULL);

	if (trans_tx->trans_idi == NULL)
		return;

	iwl_slv_tx_free(trans);

	if (trans_tx->reclaim_info_pool) {
		kmem_cache_destroy(trans_tx->reclaim_info_pool);
		trans_tx->reclaim_info_pool = NULL;
	}

	iwl_idi_free_sg_list(trans, &trans_tx->sg_list[TX_SG_HP_CH]);
	iwl_idi_free_sg_list(trans, &trans_tx->sg_list[TX_SG_LP_CH]);
}

void iwl_idi_tx_close_sg_list(struct iwl_idi_trans_tx *trans_tx, u8 chan)
{
	struct idi_sg_desc *sg_desc;

	if (unlikely(!trans_tx->sg_list_meta[chan].used_count))
		return;

	sg_desc = (struct idi_sg_desc *)trans_tx->sg_list[chan].addr;
	sg_desc[trans_tx->sg_list_meta[chan].used_count - 1].next =
			cpu_to_le32(IWL_IDI_SG_LIST_END|IWL_IDI_SG_LIST_INT);
}

#ifdef CPTCFG_IWLWIFI_UT_TX_STORE
/* For TX Store test */
static u8	*ut_source;
#endif

static void
iwl_idi_tx_free_resources(struct iwl_idi_trans_tx *trans_tx,
			  u8 * const pool_idx_set, u8 tbs_num)
{
	int i;

	spin_lock_bh(&trans_tx->slv_tx.mem_rsrc_lock);

	iwl_slv_al_mem_pool_free(&trans_tx->slv_tx,
				 &trans_tx->slv_tx.tfd_pool,
				 pool_idx_set[0]);

	for (i = 1; i <= tbs_num; i++) {
		iwl_slv_al_mem_pool_free(&trans_tx->slv_tx,
					 &trans_tx->slv_tx.tb_pool,
					 pool_idx_set[i]);
	}
	spin_unlock_bh(&trans_tx->slv_tx.mem_rsrc_lock);
}

static int
iwl_idi_tx_get_resources(struct iwl_idi_trans_tx *trans_tx, u8 txq_id,
			 u8 * const pool_idx_set, u8 txcs_num)
{
	struct iwl_trans *trans = IWL_TRANS_TX_GET_TRANS(trans_tx);
	struct iwl_trans_slv *trans_slv = IWL_TRANS_GET_SLV_TRANS(trans);

	int free_tfds, free_tbs, i, ret;
	u8 idx;

	i = 0;
	spin_lock_bh(&trans_tx->slv_tx.mem_rsrc_lock);

	/* check resources availability */
	free_tbs = iwl_slv_al_mem_pool_free_count(&trans_tx->slv_tx,
						  &trans_tx->slv_tx.tb_pool);
	free_tfds = iwl_slv_al_mem_pool_free_count(&trans_tx->slv_tx,
						   &trans_tx->slv_tx.tfd_pool);
	if (!iwl_idi_tx_policy_check_alloc(trans_slv, txq_id, free_tfds,
					   free_tbs, 1, txcs_num)) {
		IWL_DEBUG_TX(trans, "%s failed, no free SRAM.\n", __func__);
		ret = -ENOMEM;
		goto error;
	}

	/* allocate TFD, store it in the first entry */
	idx = iwl_slv_al_mem_pool_alloc(&trans_tx->slv_tx,
					&trans_tx->slv_tx.tfd_pool);
	if (idx < 0) {
		ret = -IWL_SLV_TX_ALLOC_ERR;
		goto error;
	}

	/* TFD should be stored at index 0 */
	pool_idx_set[0] = idx;

	/* allocate TBs for each TXC (index 0 is reserved for TFD) */
	for (i = 1; i <= txcs_num; i++) {
		idx = iwl_slv_al_mem_pool_alloc(&trans_tx->slv_tx,
						&trans_tx->slv_tx.tb_pool);
		if (idx < 0) {
			ret = -IWL_SLV_TX_ALLOC_ERR;
			goto error;
		}
		pool_idx_set[i] = idx;
	}

	spin_unlock_bh(&trans_tx->slv_tx.mem_rsrc_lock);
	return 0;

error:
	spin_unlock_bh(&trans_tx->slv_tx.mem_rsrc_lock);

	if (i > 0)
		/* If arrived here, TFD was allocated - thus always freeing
		 * entry at idx 0*/
		iwl_idi_tx_free_resources(trans_tx, pool_idx_set, i);

	return ret;
}

static int iwl_idi_tx_attach_tfd(struct iwl_idi_trans_tx *trans_tx,
				 u8 * const pool_idx_set,
				 struct iwl_idi_txc_header *txc,
				 struct iwl_idi_tfd *tfd,
				 u16 len)
{
	u32 dest;
	int ret = 0;

	/* first entry is the TFD */
	tfd->tbs[tfd->num_tbs].tb_idx = pool_idx_set[tfd->num_tbs + 1];

	dest = IDI_AL_SFDB_PAYLOAD_MEM_ADDR +
	       tfd->tbs[tfd->num_tbs].tb_idx * IDI_TX_PAYLOAD_PAGE_SIZE;

	if (unlikely(dest & ~IWL_IDI_TXC_DEST_MASK))
		return -IWL_SLV_TX_GEN_ERR;

	txc->dest_and_flags = cpu_to_le32(dest);
	txc->len = cpu_to_le16(len);

	tfd->tbs[tfd->num_tbs].tb_len = cpu_to_le16(len);

#ifdef CPTCFG_IWLWIFI_UT_TX_STORE
	/* Copy the TB data to the mock AL SRAM */
	memcpy(IWL_IDI_UT_SRAM_VIRTUAL + dest,
	       ut_source,
	       len);
	ut_source += len;
#endif

	tfd->num_tbs++;

	return ret;
}

static
int iwl_idi_tx_add_frag_to_sg(struct iwl_trans *trans,
			      struct iwl_idi_tx_reclaim_info *reclaim_info,
			      u8 *addr, u16 len, u8 chan)
{
	struct iwl_idi_trans_tx *trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);
	struct idi_sg_desc *sg_desc;
	dma_addr_t phys_addr;
	int sg_idx;

	phys_addr = dma_map_single(trans->dev, addr, len, DMA_TO_DEVICE);

	if (unlikely(dma_mapping_error(trans->dev, phys_addr))) {
		IWL_ERR(trans, "%s (%d): dma_map_single failed.\n",
			__func__, __LINE__);
		return -ENOMEM;
	}
	sg_idx = trans_tx->sg_list_meta[chan].used_count;

	sg_desc = &((struct idi_sg_desc *)
		   trans_tx->sg_list[chan].addr)[sg_idx];

	trans_tx->sg_list_meta[chan].used_count++;

	sg_desc->base = cpu_to_le32(phys_addr);
	sg_desc->size = cpu_to_le32(len);

	dma_unmap_addr_set(&reclaim_info->map_data[reclaim_info->blocks_count],
			   mapping, phys_addr);
	dma_unmap_len_set(&reclaim_info->map_data[reclaim_info->blocks_count],
			  len, len);
	reclaim_info->blocks_count++;
	return 0;
}

static inline void iwl_idi_tx_align_addr(u8 **addr, u8 *pb)
{
	u8 *aligned_addr;
	*pb = 0;

	aligned_addr = PTR_ALIGN(*addr, 4);
	if (aligned_addr != *addr) {
		/* 4's complement of the alignment difference */
		*pb = 4 - (aligned_addr - *addr);
		*addr -= *pb;
	}
}

/**
 * iwl_idi_tx_set_txc_chunk - set TXC and TFD entry for a single continuous
 * memory chunk.
 * Note, since alignment is a requirement of IDI DBB DMA only, pb and pa
 * should be set only for the first and the last TXCs respectively.
 * @trans: IDI transport
 * @chan: channel
 * @tfd: a ponter to tfd which will be updtated
 * @txbu_info: addr/len/txcs num for the chunk
 * @txcs: array of txcs, starting at the first free TXC
 *
 * Returns length including alignments if success, negative error otherwise.
 */
static
int iwl_idi_tx_set_txc_chunk(struct iwl_trans *trans, u8 chan,
			     struct iwl_idi_tfd *tfd,
			     struct iwl_idi_txc_header *txcs,
			     struct iwl_slv_tx_chunk_info * const txbu_info,
			     u8 * const pool_idx_set)
{
	struct iwl_idi_trans_tx *trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);
	u8 pb, pa;
	int curr_txc, i, cur_len, ret;

	curr_txc = 0;
	iwl_idi_tx_align_addr(&txbu_info->addr, &pb);

	/* pb should be set only for the first txc of the memory chunk */
	txcs[curr_txc].pb = pb;

	/* check length alignment and set pa of the last txc for this chunk */
	pa = ALIGN(txbu_info->len + pb, 4) - txbu_info->len - pb;
	txcs[txbu_info->desc_num - 1].pa = pa;

#ifdef CPTCFG_IWLWIFI_UT_TX_STORE
	/* For TX Store test - save the source address */
	ut_source = txbu_info->addr;
#endif

	/* the default is to occupy a full TB */
	cur_len = IDI_TX_PAYLOAD_PAGE_SIZE;

	for (i = 0; i < txbu_info->desc_num; i++) {
		/* the last txc can occupy less than IDI_TX_PAYLOAD_PAGE_SIZE */
		if (i == txbu_info->desc_num - 1) {
			cur_len = txbu_info->len &
				IWL_IDI_PAYLOAD_PAGE_SIZE_MASK;
			if (!cur_len)
				cur_len = IDI_TX_PAYLOAD_PAGE_SIZE;
		}

		ret = iwl_idi_tx_attach_tfd(trans_tx, pool_idx_set,
					    &txcs[curr_txc], tfd,
					    cur_len);
		if (ret) {
			IWL_ERR(trans, "%s failed to attach TFD.\n", __func__);
			return ret;
		}
		curr_txc++;
	}

	/* return the length with all the alignments */
	return txbu_info->len + pa + pb;
}

/**
 * iwl_idi_set_txbu_and_tfd - fill TXBU header and TXC describing the TFD
 */
static
int iwl_idi_set_txbu_and_tfd(struct iwl_trans *trans,
			     struct iwl_idi_txbu_header *txbu,
			     struct iwl_idi_txc_header *txcs,
			     u8 txq_id,
			     struct iwl_slv_tx_dtu_meta *txbu_meta,
			     u8 * const pool_idx_set)
{
	struct iwl_idi_trans_tx *trans_tx = IWL_TRANS_GET_IDI_TRANS_TX(trans);
	u32 dest;

	/* fill in TXBU */
	txbu->signature = cpu_to_le16(IWL_IDI_TXBU_SIGNATURE);

	/* Note, txbu->lut_value was allocated at the start */

	/* FIXME: wrap around behavior should be defined */
	txbu->lut_value = pool_idx_set[0];
	txbu->seq_num = cpu_to_le16(trans_tx->txbu_seq_num);
	txbu->txc_count = txbu_meta->total_desc_num;
	txbu->flags = IWL_IDI_TXBU_EX_BIT;
	txbu->tfd_index = trans_tx->next_tfd_index[txq_id];
	txbu->queue = txq_id;
	txbu->byte_cnt_value = cpu_to_le16(txbu_meta->total_len);

	/* TXC for compressed TFD */
	dest = IDI_AL_SFDB_TFD_POOL_BASE_ADDR +
	       txbu->lut_value * sizeof(struct iwl_idi_tfd);
	if (unlikely(dest & ~IWL_IDI_TXC_DEST_MASK)) {
		spin_lock_bh(&trans_tx->slv_tx.mem_rsrc_lock);
		iwl_slv_al_mem_pool_free(&trans_tx->slv_tx,
					 &trans_tx->slv_tx.tfd_pool,
					 txbu->lut_value);
		spin_unlock_bh(&trans_tx->slv_tx.mem_rsrc_lock);
		/* FIXME: free resources */
		IWL_ERR(trans, "%s failed, invalid destination 0x%x\n",
			__func__, dest);
		return -IWL_SLV_TX_GEN_ERR;
	}

	/* set the first TXC which always references TFD */
	txcs[0].dest_and_flags = cpu_to_le32(dest);
	txcs[0].len = cpu_to_le16(sizeof(struct iwl_idi_tfd));

	return 0;
}

static
int iwl_idi_tx_process_txbu_info(struct iwl_idi_trans_tx *trans_tx,
				 u8 chan, u8 txq_id,
				 struct iwl_slv_tx_dtu_meta *txbu_meta,
				 struct iwl_idi_tx_reclaim_info *reclaim_info,
				 u8 * const pool_idx_set)
{
	struct iwl_trans *trans = IWL_TRANS_TX_GET_TRANS(trans_tx);
	struct iwl_idi_txbu_header *txbu;
	struct iwl_idi_txc_header *txcs;
	struct iwl_idi_tfd *tfd;
	u16 total_hdr_size;
	u8 i, pb;
	int ret, chunk_len;

	/* set up txbu headers */
	total_hdr_size =
		iwl_idi_tx_get_txbu_hdr_size(txbu_meta->total_desc_num);
	txbu = (struct iwl_idi_txbu_header *)
			(txbu_meta->chunk_info[0].addr - total_hdr_size);
	iwl_idi_tx_align_addr((u8 **)&txbu, &pb);
	total_hdr_size += pb;
	memset(txbu, 0, total_hdr_size);

	txcs = iwl_idi_tx_get_txcs_from_txbu(txbu);
	tfd = iwl_idi_tx_get_tfd_from_txbu(txbu, txbu_meta->total_desc_num);

	reclaim_info->txbu = txbu;

	/* set up txbu header and the txc for tfd */
	ret = iwl_idi_set_txbu_and_tfd(trans, txbu, txcs, txq_id,
				       txbu_meta, pool_idx_set);
	if (ret) {
		IWL_ERR(trans, "%s failed to set txbu and txc-tfd\n", __func__);
		goto error_free;
	}

	/* skip the first TXC which is always refers to the TFD */
	txcs++;
	for (i = 0; i < txbu_meta->chunks_num; i++) {
		u8 *addr;

		chunk_len = iwl_idi_tx_set_txc_chunk(trans, chan, tfd, txcs,
						     &txbu_meta->chunk_info[i],
						     pool_idx_set);
		if (i == 0) {
			/* for the first chunk need to map header as well */
			chunk_len += total_hdr_size;
			addr = (u8 *)txbu;
		} else {
			addr = txbu_meta->chunk_info[i].addr;
		}

		ret = iwl_idi_tx_add_frag_to_sg(trans, reclaim_info, addr,
						chunk_len, chan);
		if (ret) {
			IWL_ERR(trans, "%s failed to set sg descriptor\n",
				__func__);
			goto error_free;
		}

		/* move to the first free txc */
		txcs += txbu_meta->chunk_info[i].desc_num;
	}

	trans_tx->txbu_seq_num++;

	/* increment index w.r.t. wrapping at queue size */
	INC_TFD_IDX(trans_tx->next_tfd_index[txq_id]);

	return 0;

error_free:
	/* FIXME */
	iwl_idi_tx_clean_txbu(trans, reclaim_info);
	return ret;
}

/**
 * iwl_idi_tx_add_burst - single entry point for policy. Send cmd/data to device
 * and move from waiting to sent queue if sent with success.
 * @trans_tx: the transport
 * @chan: channel to send
 * @txq_id: the queue
 * Returns 0 if sent, error code otherwise.
 */
int iwl_idi_tx_add_burst(struct iwl_trans_slv *trans_slv, u8 chan, u8 txq_id)
{
	struct iwl_idi_trans_tx *trans_tx =
		IWL_TRANS_SLV_GET_IDI_TRANS_TX(trans_slv);

	struct iwl_slv_tx_queue *txq = &trans_slv->txqs[txq_id];
	u8 pool_idx_set[IWL_IDI_MAX_TXC_COUNT_IN_TXBU + 1];
	struct list_head *element;
	struct iwl_slv_txq_entry *txq_entry;
	int ret;

	spin_lock_bh(&trans_slv->txq_lock);

	if (list_empty(&txq->waiting)) {
		spin_unlock_bh(&trans_slv->txq_lock);
		return -IWL_SLV_TX_Q_EMPTY;
	}
	/* get the frame from the list, don't delete it yet */
	element = txq->waiting.next;
	txq_entry = list_entry(element, struct iwl_slv_txq_entry, list);

	spin_unlock_bh(&trans_slv->txq_lock);

	/* In IDI TXBU, the first txc is describing the TFD.
	 * total_desc_num counts only the number of descriptors
	 * needed for data, hence we need to add one */
	txq_entry->dtu_meta.total_desc_num =
		txq_entry->dtu_meta.total_desc_num + 1;

	/* pass txcs_num not including the one for TFD */
	ret = iwl_idi_tx_get_resources(trans_tx, txq_id,
				       pool_idx_set,
				       txq_entry->dtu_meta.total_desc_num - 1);
	/* if no resources, just return - it's valid state */
	if (ret)
		return ret;

	txq_entry->reclaim_info =
		kmem_cache_alloc(trans_tx->reclaim_info_pool, GFP_ATOMIC);
	if (unlikely(!txq_entry->reclaim_info)) {
		ret = -ENOMEM;
		goto error_free_rsc;
	}

	memset(txq_entry->reclaim_info, 0,
	       sizeof(struct iwl_idi_tx_reclaim_info));

	ret = iwl_idi_tx_process_txbu_info(trans_tx, chan, txq_id,
					   &txq_entry->dtu_meta,
					   (struct iwl_idi_tx_reclaim_info *)
						txq_entry->reclaim_info,
					   pool_idx_set);
	if (ret) {
		/* process_txbu_info free all resoures in case of failure */
		return ret;
	}

	/* move this entry from waiting to sent only if succeeded to allocate */
	spin_lock_bh(&trans_slv->txq_lock);
	list_del(element);
	list_add_tail(&txq_entry->list, &txq->sent);
	txq->waiting_count--;
	spin_unlock_bh(&trans_slv->txq_lock);

	return 0;

error_free_rsc:
	iwl_idi_tx_free_resources(trans_tx,
				  pool_idx_set,
				  txq_entry->dtu_meta.total_desc_num - 1);
	return ret;
}
