/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#include <linux/interrupt.h>
#include <linux/kfifo.h>
#include <linux/kthread.h>

#include "iwl-em-lls.h"
#include "iwl-em-idi-tx.h"
#include "iwl-emulation.h"
#include "iwl-em-sfdb.h"
#include "iwl-target-access.h"

/*
 * IWL LLS Emulation Logger
 */
#define LLS_LOG_PREFIX	"[LLS]"

#ifdef __IWL_LLS_LOG_ENABLED__
#define LLS_TRACE_ENTER \
	IWL_EM_TRACE_ENTER(__LLS_LOG_LEVEL_TRACE__, LLS_LOG_PREFIX)
#define LLS_TRACE_EXIT \
	IWL_EM_TRACE_EXIT(__LLS_LOG_LEVEL_TRACE__, LLS_LOG_PREFIX)
#define LLS_TRACE_EXIT_RET_STR(_ret) \
	IWL_EM_LOG(__LLS_LOG_LEVEL_TRACE__, LLS_LOG_PREFIX, "<<< "_ret)
#define LLS_TRACE_EXIT_RET(_ret) \
	IWL_EM_TRACE_EXIT_RET(__LLS_LOG_LEVEL_TRACE__, LLS_LOG_PREFIX,\
			      _ret)
#define IWL_LLS_LOG(fmt, args ...) \
	IWL_EM_LOG(__LLS_LOG_LEVEL_DEBUG__, LLS_LOG_PREFIX, fmt, ## args)
#define IWL_LLS_LOG_HEX_DUMP(msg, p, len) \
	IWL_EM_LOG_HEX_DUMP(msg, p, len)
#else

#define LLS_TRACE_ENTER
#define LLS_TRACE_EXIT
#define LLS_TRACE_EXIT_RET_STR(_ret)
#define LLS_TRACE_EXIT_RET(_ret)
#define IWL_LLS_LOG(fmt, args ...)
#define IWL_LLS_LOG_HEX_DUMP(msg, p, len)

#endif

/* Always print error messages */
#define IWL_LLS_LOG_ERR(fmt, args ...) \
	IWL_EM_LOG_ERR(fmt, ## args)

/* Forward declarations */
static int iwl_lls_em_parser(void *data);
static void iwl_lls_em_inter_disp(unsigned long data);

/*
 * The LLS structure
 */
struct iwl_lls_t {

	struct iwl_trans *trans;

	/* Thread for the LLS job*/
	struct task_struct *lls_em_thread[IWL_IDI_TX_NUM_CHANNELS];

	/* Wait queue for the LLS threads */
	wait_queue_head_t iwl_lls_wq[IWL_IDI_TX_NUM_CHANNELS];

	/*
	 * The TX buffers fifos - one for High priority and one for low.
	 * The LLS will pull the burst from this buffer
	 */
	struct kfifo tx_pipe[IWL_IDI_TX_NUM_CHANNELS];

	/* Interrupt tasklet */
	struct tasklet_struct processing_done_int;

	/* Irq handlers */
	iwl_idi_lls_processing_irq irq_handlers[IWL_IDI_TX_NUM_CHANNELS];

	/* Irq data */
	void *irq_data[IWL_IDI_TX_NUM_CHANNELS];
};

/* Global LLS struct */
static struct iwl_lls_t iwl_lls_em_global;
static struct iwl_lls_t *iwl_lls_em = &iwl_lls_em_global;

/* LLS Defines */
/* The buffer size relays on the fact that there is room for 25K in the AL
 * SRAM for TBs and at any given time, at max there can be one SG list
 * in the LLS, one that just finished copy and one in the copy process.
 * in any other situation this will mean an error.
 * So the buffer is set to support the 3 maxed size SG lists */
#define IWL_EM_LLS_PIPE_SIZE	(75 * 1024)
#define IWL_LLS_EM_TXC_MAX_LEN	(256)
#define IWL_LLS_MAX_TXC		(31)

/* LLS Channel definitions */
#define IWL_LLS_HP_CHANNEL	(0)
#define IWL_LLS_LP_CHANNEL	(1)
#define IWL_LLS_MAX_TX_CHANNEL	(2)
#define GET_LLS_CHANNEL(channel)	(channel - 1)

/* So there will be no confusion with the IDI channel numbers */
#undef IDI_PRIMARY_CHANNEL
#undef IDI_SECONDARY_CHANNEL

/*
 * Initialize the LLS TX component.
 * Should be called as part of the global AL intialization flow.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_lls_em_init(struct iwl_trans *trans)
{
	int ret = 0;
	LLS_TRACE_ENTER;

	/* Init the LLS struct */
	memset(iwl_lls_em , 0, sizeof(struct iwl_lls_t));

	/* Store trans */
	iwl_lls_em->trans = trans;

	/* Allocate the High priority TX buffer fifo */
	ret = kfifo_alloc(&iwl_lls_em->tx_pipe[IWL_LLS_HP_CHANNEL],
			  IWL_EM_LLS_PIPE_SIZE,
			  GFP_KERNEL);
	if (ret) {
		IWL_LLS_LOG_ERR("Cannot allocate fifo for HP TX buffer");
		goto out;
	}
	IWL_LLS_LOG("LLS HP pipe Initialized");

	/* Allocate the High priority TX buffer fifo */
	ret = kfifo_alloc(&iwl_lls_em->tx_pipe[IWL_LLS_LP_CHANNEL],
			  IWL_EM_LLS_PIPE_SIZE,
			  GFP_KERNEL);
	if (ret) {
		IWL_LLS_LOG_ERR("Cannot allocate fifo for LP TX buffer");
		goto release_hp;
	}
	IWL_LLS_LOG("LLS LP pipe Initialized");

	/* Waitqueue init */
	init_waitqueue_head(&(iwl_lls_em->iwl_lls_wq[IWL_LLS_HP_CHANNEL]));
	init_waitqueue_head(&(iwl_lls_em->iwl_lls_wq[IWL_LLS_LP_CHANNEL]));
	IWL_LLS_LOG("LLS Wait queue initialized");

	IWL_LLS_LOG("LLS Initialized");
	goto out;

release_hp:
	kfifo_free(&iwl_lls_em->tx_pipe[IWL_LLS_HP_CHANNEL]);
out:
	LLS_TRACE_EXIT;
	return ret;
}

/**
 * Free the LLS TX emulation block.
 * Synchronously returns when the LLS work for current job is finished.
 */
void iwl_lls_em_free(void)
{
	LLS_TRACE_ENTER;

	/* Free the HP TX Buffer fifo*/
	kfifo_free(&(iwl_lls_em->tx_pipe[IWL_LLS_HP_CHANNEL]));

	/* Free the LP TX Buffer fifo*/
	kfifo_free(&(iwl_lls_em->tx_pipe[IWL_LLS_LP_CHANNEL]));

	IWL_LLS_LOG("LLS Freed");
	LLS_TRACE_EXIT;
}

/**
 * Start the LLS TX Emulation.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_lls_em_start(void)
{
	int ret = 0;
	LLS_TRACE_ENTER;

	/* Start LLS HP Emulator */
	iwl_lls_em->lls_em_thread[IWL_LLS_HP_CHANNEL] =
					kthread_run(iwl_lls_em_parser,
						    (void *)IWL_LLS_HP_CHANNEL,
						    "LLS_TX_HP_CH_EM");
	if (IS_ERR(iwl_lls_em->lls_em_thread[IWL_LLS_HP_CHANNEL])) {
		IWL_LLS_LOG_ERR("Cannot create thread for LLS HP");
		return -ENOMEM;
		goto out;
	}
	IWL_LLS_LOG("LLS HP thread - STARTED");

	/* Start LLS HP Emulator */
	iwl_lls_em->lls_em_thread[IWL_LLS_LP_CHANNEL] =
					kthread_run(iwl_lls_em_parser,
						    (void *)IWL_LLS_LP_CHANNEL,
						    "LLS_TX_LP_CH_EM");
	if (IS_ERR(iwl_lls_em->lls_em_thread[IWL_LLS_LP_CHANNEL])) {
		IWL_LLS_LOG_ERR("Cannot create thread for LLS LP");
		return -ENOMEM;
		goto kill_HP_thread;
	}
	IWL_LLS_LOG("LLS LP thread - STARTED");

	/* Reset the kfifo */
	kfifo_reset(&iwl_lls_em->tx_pipe[IWL_LLS_HP_CHANNEL]);
	kfifo_reset(&iwl_lls_em->tx_pipe[IWL_LLS_LP_CHANNEL]);

	/* Init LLS to dbb interrupt tasklet */
	tasklet_init(&(iwl_lls_em->processing_done_int),
		     iwl_lls_em_inter_disp, 0);

	IWL_LLS_LOG("LLS STARTED");
	goto out;

kill_HP_thread:
	kthread_stop(iwl_lls_em->lls_em_thread[IWL_LLS_HP_CHANNEL]);
out:
	LLS_TRACE_EXIT;
	return ret;
}

/**
 * Stop the LLS TX Emulation.
 * Used to stop all parsing jobs.
 * Synchronious function - returns only after all parsing operations are
 * stopped.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_lls_em_stop(void)
{
	int ret = 0;
	LLS_TRACE_ENTER;

	/* Stop the LLS HP TX parser work */
	if (iwl_lls_em->lls_em_thread[IWL_LLS_HP_CHANNEL])
		kthread_stop(iwl_lls_em->lls_em_thread[IWL_LLS_HP_CHANNEL]);
	iwl_lls_em->lls_em_thread[IWL_LLS_HP_CHANNEL] = NULL;

	/* Stop the LLS LP TX parser work */
	if (iwl_lls_em->lls_em_thread[IWL_LLS_LP_CHANNEL])
		kthread_stop(iwl_lls_em->lls_em_thread[IWL_LLS_LP_CHANNEL]);
	iwl_lls_em->lls_em_thread[IWL_LLS_LP_CHANNEL] = NULL;

	/* Stop the LLS interrupts work */
	tasklet_kill(&(iwl_lls_em->processing_done_int));

	IWL_LLS_LOG("LLS STOPPED");
	LLS_TRACE_EXIT;
	return ret;
}

/**
 * Calls the handler registered by the IDI to call when there is an
 * interrupt call required to the IDI.
 * Called from tasklet context.
 *
 * @param data - the channel number on which the interrupt is required.
 */
static inline void iwl_lls_em_inter_disp(unsigned long data)
{
	u8 channel = (u8)data;
	IWL_LLS_LOG("Interrupt called on channel %d", channel);

	/* Run the irq handler in the dbb in this tasklet context
	 * With the pre registered data by the IDI */
	if (iwl_lls_em->irq_handlers[channel])
		iwl_lls_em->irq_handlers[channel](cpu_to_le32((u32) channel),
						iwl_lls_em->irq_data[channel]);
}

/**
 * Scheduals a call to the interrupt dispatcher in order to call
 * the IDI interrupt handler registered for lls interrupts dor
 * this channel.
 *
 * @param channel - the channel number on which the interrupt was called.
 */
static inline void iwl_lls_em_call_inter(u8 channel)
{
	LLS_TRACE_ENTER;

	/* Set channel data */
	iwl_lls_em->processing_done_int.data = (unsigned long)channel;

	/* Schedule tasklet */
	tasklet_schedule(&iwl_lls_em->processing_done_int);

	LLS_TRACE_EXIT;
}

/**
 * Registers an interrupt handler for LLs interrupts to the IDI.
 *
 * @param irq_handler - interrupt handler to register.
 * @param channel - the channel number on which the interrupt will be called.
 * @param data  void pointer which will be given to the interrupt handler.
 */
void iwl_lls_em_register_inter(iwl_idi_lls_processing_irq irq_handler,
			      u8 channel, void *data)
{
	LLS_TRACE_ENTER;

	BUG_ON(channel > IWL_IDI_TX_NUM_CHANNELS);
	IWL_LLS_LOG("Interrupt register on channel %d", channel);

	/* Register irq handler */
	iwl_lls_em->irq_handlers[GET_LLS_CHANNEL(channel)] = irq_handler;

	/* Register irq data */
	iwl_lls_em->irq_data[GET_LLS_CHANNEL(channel)] = data;

	LLS_TRACE_EXIT;
}

/*
 * Push data to the LLS pipe for the target channel.
 *
 * @param buffer - the target address to push from tp the pipe.
 * @param size - The amount of data in bytes to push to the pipe.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_lls_em_push_to_pipe(u32 channel, void *buffer, u32 size)
{
	u32 chan = GET_LLS_CHANNEL(channel);

	LLS_TRACE_ENTER;
	/* Check max size */
	if (WARN_ON(kfifo_avail(&iwl_lls_em->tx_pipe[chan]) < size)) {
		IWL_LLS_LOG_ERR("Not more room in the pipe on channel %d",
				chan);
		return -ENOMEM;
	}
	IWL_LLS_LOG("Pushing to pipe %d bytes for channel %d", size, chan);

	/* Insert the given buffer to the fifo */
	kfifo_in(&iwl_lls_em->tx_pipe[chan], buffer, size);

	/* Wake up parsers */
	wake_up_interruptible(&(iwl_lls_em->iwl_lls_wq[chan]));

	LLS_TRACE_EXIT;
	return 0;
}

/*
 * TXBU Parser helper functions
 */
static inline bool iwl_lls_em_txbu_verify_signature(struct iwl_lls_txbu *txbu)
{
	return (le16_to_cpu(txbu->signature) == IWL_LLS_TXBU_SIGNATURE);
}

static inline bool iwl_lls_em_is_ex_mode(struct iwl_lls_txbu *txbu)
{
	return (txbu->flags & IWL_LLS_EX_MODE_MASK) == IWL_LLS_EX_MODE_MASK;
}

static inline bool iwl_lls_em_is_inter_enabled(struct iwl_lls_txbu *txbu)
{
	return (txbu->flags & IWL_LLS_INTEN_MASK) == IWL_LLS_INTEN_MASK;
}

static inline bool iwl_lls_em_is_txc_sc(struct iwl_lls_txbu *txbu)
{
	return (txbu->flags & IWL_LLS_TXC_SC_MASK) == IWL_LLS_TXC_SC_MASK;
}

/*
 * TXC Parser helper functions - Extract bit fileds data
 */
static inline u8 iwl_lls_em_txc_get_pb(struct iwl_lls_txc *txc)
{
	return txc->pb & IWL_LLS_TXC_PB_MASK;
}

static inline u8 iwl_lls_em_txc_get_pa(struct iwl_lls_txc *txc)
{
	return txc->pa & IWL_LLS_TXC_PA_MASK;
}

static inline void *iwl_lls_em_txc_get_dst(struct iwl_lls_txc *txc)
{
	return (void *)((u8 *)AL_SRAM_VIRTUAL_ADDRESS +
	       (le32_to_cpu(txc->dest_and_flags) & IWL_LLS_TXC_DST_MASK));
}

static inline u16 iwl_lls_em_txc_get_size(struct iwl_lls_txc *txc)
{
	return le16_to_cpu(txc->len);
}

#define LLS_THREAD_STOP_COND (kfifo_len(&iwl_lls_em->tx_pipe[channel]) || \
			      kthread_should_stop())
/**
 * Reads data from the channel pipe.
 * Sleeps until is is awakend with more data on the pipe.
 *
 * Can also return if kthread stop was called
 */
static void iwl_lls_em_pipe_read(u32 channel, void *target, u32 size)
{
	int read_left = size, size_read;
	wait_queue_head_t *wq = iwl_lls_em->iwl_lls_wq;

	if (size == 0)
		return;

	IWL_LLS_LOG("Setting read for channel pipe %d, target 0x%x, size %d",
		    channel,
		    (unsigned int)target,
		    size);

	/* Try to read and sleep in case the pipe is empty */
	while (!kthread_should_stop() && read_left) {
		/* Sleep until there is data on the pipe */
		wait_event_interruptible(wq[channel], LLS_THREAD_STOP_COND);
		if (!kthread_should_stop()) {
			size_read = kfifo_out(&iwl_lls_em->tx_pipe[channel],
					       target,
					       read_left);
			read_left -= size_read;
			target = (u8 *)target + size_read;
		}
	}

	/* Check for kthread stop call */
	if (read_left)
		IWL_LLS_LOG("KTHREAD STOP called on %d", channel);

	else
		IWL_LLS_LOG("Reading Complete on channel %d of %d bytes",
			    channel,
			    size);
}

static void iwl_lls_em_cpy_data_to_dst(u32 channel, struct iwl_lls_txc *txc,
				       bool ex_mode)
{
	u32 size;
	u32 target;
	__le32 reg_data;

	target = le32_to_cpu(txc->dest_and_flags);
	size = iwl_lls_em_txc_get_size(txc);

	/* Write the data to the SRAM in extended mode or when the address
	 * mapped to the SRAM. Otherwise, the address represent a register
	 * in the device, so target access should be used. */
	if (ex_mode || (target >= IDI_AL_SFDB_BASE_ADDR && target <
			IDI_AL_SFDB_BASE_ADDR + AL_SRAM_SIZE)) {
		iwl_lls_em_pipe_read(channel, iwl_lls_em_txc_get_dst(txc),
				     size);
	} else {
		target = IWL_AL_SUB_SRAM_SIZE(target);
		WARN_ON(size != sizeof(u32));
		iwl_lls_em_pipe_read(channel, (void *)&reg_data, size);
		iwl_idi_tg_write32(iwl_lls_em->trans, target,
				   le32_to_cpu(reg_data));
	}
}

/**
 * Parses the TX buffers that contain the TXBU streams given by the IDI TX.
 * Called in tasklet context.
 * Pull as many buffers as possible from the fifo since the SG LL delivers many
 * TXBU, and each one is put into one buffer.
 *
 *@param data - unused - tasklet API.
 */
static int iwl_lls_em_parser(void *data)
{
	struct iwl_lls_txbu txbu;
	struct iwl_lls_txc txcs[IWL_LLS_MAX_TXC];
	char trash_buffer[4];
	u8 txc_num, txc_itr;
	bool ex_mode;
	u32 channel = (u32) data;

	LLS_TRACE_ENTER;

	while (!kthread_should_stop()) {
		IWL_LLS_LOG("Processing Data");

		/* Get the TXBU */
		iwl_lls_em_pipe_read(channel,
				     &txbu,
				     sizeof(struct iwl_lls_txbu));
		if (kthread_should_stop())
			goto stop_thread;
		IWL_LLS_LOG_HEX_DUMP("TXBU-LLS: ",
				     &txbu,
				     sizeof(struct iwl_lls_txbu));

		/* Verify TXBU Signature */
		BUG_ON(!iwl_lls_em_txbu_verify_signature(&txbu));
		IWL_LLS_LOG("Signature verifyed");

		/* Get number of TXCs */
		txc_num = txbu.txc_count;
		IWL_LLS_LOG("TXC count = %d", txc_num);
		BUG_ON(txc_num <= 0 ||	txc_num > IWL_LLS_MAX_TXC);

		/* If ex mode - print tx queue and tfd index */
		ex_mode = iwl_lls_em_is_ex_mode(&txbu);
		if (ex_mode) {
			IWL_LLS_LOG("EX MODE: tx queue = %d, tfd_index = %d",
				    txbu.queue,
				    txbu.tfd_index);
		}

		/*Get the TXCs*/
		iwl_lls_em_pipe_read(channel,
				     txcs,
				     (sizeof(struct iwl_lls_txc) * txc_num));
		if (kthread_should_stop())
			goto stop_thread;
		IWL_LLS_LOG_HEX_DUMP("TXCS-LLS: ",
				     txcs,
				     sizeof(struct iwl_lls_txc) * txc_num);

		/* Copy data - Iterate over all TXCs */
		for (txc_itr = 0; txc_itr < txc_num; txc_itr++) {

			/* Check TXC */
			if (iwl_lls_em_is_txc_sc(&txbu))
				WARN_ON(iwl_lls_em_txc_get_size(&txcs[txc_itr])
					> IWL_LLS_EM_TXC_MAX_LEN);

			/* Pull BEFORE padding from pipe to trash buffer */
			iwl_lls_em_pipe_read(channel,
					     trash_buffer,
					iwl_lls_em_txc_get_pb(&txcs[txc_itr]));
			if (kthread_should_stop())
				goto stop_thread;

			/* Copy the data from pipe */
			iwl_lls_em_cpy_data_to_dst(channel, &txcs[txc_itr],
						   ex_mode);
			if (ex_mode)
				IWL_LLS_LOG_HEX_DUMP("AL SRAM: ",
						     iwl_lls_em_txc_get_dst(
							&txcs[txc_itr]),
						     iwl_lls_em_txc_get_size(
							&txcs[txc_itr]));

			if (kthread_should_stop())
				goto stop_thread;

			/* Pull AFTER padding from pipe to trash buffer */
			iwl_lls_em_pipe_read(channel,
					     trash_buffer,
					iwl_lls_em_txc_get_pa(&txcs[txc_itr]));
			if (kthread_should_stop())
				goto stop_thread;

		}

		/* Extended mode */
		if (ex_mode) {
			/* Write BC Value */
			iwl_sfdb_em_write_bc(
				txbu.queue,
				txbu.tfd_index,
				txbu.byte_cnt_value);

			/* Write LUT value */
			iwl_sfdb_em_write_lut_value(
				txbu.queue,
				txbu.tfd_index,
				txbu.lut_value);

			/* Update Write Pointer */
			iwl_sfdb_em_attach_tfd(txbu.queue, txbu.lut_value);
			iwl_sfdb_em_inc_write_ptr(txbu.queue);
		}

		/* Call interrupt */
		if (iwl_lls_em_is_inter_enabled(&txbu))
			iwl_lls_em_call_inter(channel);
	}

stop_thread:
	IWL_LLS_LOG("LLS THREAD %d - STOPPED", channel);
	LLS_TRACE_EXIT;
	return 0;
}
