/******************************************************************************
 *
 * Copyright(c) 2005 - 2013 Intel Corporation.
 * All rights reserved.
 *
 * LICENSE PLACE HOLDER
 *
 *****************************************************************************/

#include <linux/kthread.h>

#include "iwl-emulation.h"
#include "iwl-idi.h"
#include "iwl-em-idi-tx.h"
#include "iwl-em-lls.h"

/*
 * IWL IDI Emulation Logger
 */
#define IDI_TX_LOG_PREFIX	"[IDI_TX]"

#ifdef __IWL_IDI_TX_LOG_ENABLED__
#define IDI_TX_TRACE_ENTER \
	IWL_EM_TRACE_ENTER(__IDI_TX_LOG_LEVEL_TRACE__, IDI_TX_LOG_PREFIX)
#define IDI_TX_TRACE_EXIT \
	IWL_EM_TRACE_EXIT(__IDI_TX_LOG_LEVEL_TRACE__, IDI_TX_LOG_PREFIX)
#define IDI_TX_TRACE_EXIT_RET_STR(_ret) \
	IWL_EM_LOG(__IDI_TX_LOG_LEVEL_TRACE__, IDI_TX_LOG_PREFIX, "<<< "_ret)
#define IDI_TX_TRACE_EXIT_RET(_ret) \
	IWL_EM_TRACE_EXIT_RET(__IDI_TX_LOG_LEVEL_TRACE__, IDI_TX_LOG_PREFIX,\
			      _ret)
#define IWL_IDI_TX_LOG(fmt, args ...) \
	IWL_EM_LOG(__IDI_TX_LOG_LEVEL_DEBUG__, IDI_TX_LOG_PREFIX, fmt, ## args)
#define IWL_IDI_TX_LOG_HEX_DUMP(msg, p, len) \
	IWL_EM_LOG_HEX_DUMP(msg, p, len)
#else

#define IDI_TX_TRACE_ENTER
#define IDI_TX_TRACE_EXIT
#define IDI_TX_TRACE_EXIT_RET_STR(_ret)
#define IDI_TX_TRACE_EXIT_RET(_ret)
#define IWL_IDI_TX_LOG(fmt, args ...)
#define IWL_IDI_TX_LOG_HEX_DUMP(msg, p, len)

#endif
/* Always print error messages */
#define IWL_IDI_TX_LOG_ERR(fmt, args ...) \
	IWL_EM_LOG_ERR(fmt, ## args)

/* IDI TX Workqueue definitons */
#define IWL_IDI_TX_WQ_FLAGS		(WQ_HIGHPRI | WQ_NON_REENTRANT)
#define IWL_IDI_TX_MAX_ACTIVE_WORKERS	IWL_IDI_TX_NUM_CHANNELS

/* IDI TX Buffer Definitions */
#define IWL_IDI_TX_BUFFER_SIZE		(4 * 1024) /*4K*/
#define IWL_IDI_TX_BUFFER_FLAGS	0
#define IDI_TX_TXBU_LINK_NUM	0
#define IDI_TX_PAYLOAD_LINK_NUM	1

/*
 * The IDI and yDMA TX flow struct.
 */
struct iwl_idi_tx_em_t {

	/*
	 * Workqueue and work struct for each IDI TX DMA channel.
	 */
	struct workqueue_struct *idi_tx_wq;
	struct work_struct idi_tx_high_worker;
	struct work_struct idi_tx_low_worker;

};

/* Global IDI TX struct */
static struct iwl_idi_tx_em_t iwl_idi_tx_em_global;
static struct iwl_idi_tx_em_t *iwl_idi_tx_em = &iwl_idi_tx_em_global;

/* Forward declarations */
static void iwl_idi_tx_em_high_work(struct work_struct *data);
static void iwl_idi_tx_em_low_work(struct work_struct *data);

/*
 * Initialize the IDI TX component.
 * Should be called as part of the global IDI intialization flow.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_tx_em_init(void)
{
	int ret = 0;
	IDI_TX_TRACE_ENTER;

	/* Allocate the wrokqueue and init workers */
	iwl_idi_tx_em->idi_tx_wq =
			alloc_workqueue("iwl_idi_tx_wq",
					IWL_IDI_TX_WQ_FLAGS,
					IWL_IDI_TX_MAX_ACTIVE_WORKERS);
	if (!iwl_idi_tx_em->idi_tx_wq) {
		IWL_IDI_TX_LOG_ERR("Cannot allocate wq");
		ret = -ENOMEM;
		goto out;
	}
	INIT_WORK(&iwl_idi_tx_em->idi_tx_high_worker, iwl_idi_tx_em_high_work);
	INIT_WORK(&iwl_idi_tx_em->idi_tx_low_worker, iwl_idi_tx_em_low_work);

	IWL_IDI_TX_LOG("IDI TX Initialized succesfuly");

out:
	IDI_TX_TRACE_EXIT;
	return ret;
}

/**
 * Free the IDI TX emulation block.
 * All DMA copies should be stopped before the call to this function.
 */
void iwl_idi_tx_em_free(void)
{
	IDI_TX_TRACE_ENTER;

	/* Free TX buffers */
	if (iwl_idi_tx_em->idi_tx_wq)
		destroy_workqueue(iwl_idi_tx_em->idi_tx_wq);
	IWL_IDI_TX_LOG("workqueue destroyed");

	IWL_IDI_TX_LOG("IDI TX freed succesfuly");
	IDI_TX_TRACE_EXIT;
}

/**
 * Start the IDI TX Emulation.
 *
 * Used mainly for debugging options since the IDI is considered started
 * already on init.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_tx_em_start(void)
{
	int ret = 0;
	IDI_TX_TRACE_ENTER;

	IWL_IDI_TX_LOG("Started IDI TX DMA");

	IDI_TX_TRACE_EXIT;
	return ret;
}

/**
 * Stop the IDI TX Emulation.
 * Used to stop all DMA work on the TX channels.
 * Synchronious function - returns only after all DMA operations are stopped.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_tx_em_stop(void)
{
	int ret = 0;
	IDI_TX_TRACE_ENTER;

	/* Cancel work on DMA channels */
	cancel_work_sync(&iwl_idi_tx_em->idi_tx_high_worker);
	cancel_work_sync(&iwl_idi_tx_em->idi_tx_low_worker);
	IWL_IDI_TX_LOG("IDI TX Stopped");

	IDI_TX_TRACE_EXIT;
	return ret;
}

/**
 * Start the  DMA operation on the given TX channel.
 * Since this is called only when a SG linked list is set by the IDI and then
 * the IDI waits for interrupt telling it that the processing has finished,
 * there is no risk of race here - per channel.
 *
 *@channel - The channel to enable the dma in.
 * Returns 0 on success, negative value describing the error otherwise.
 */
int iwl_idi_tx_em_dma_enable(u32 channel)
{
	int ret = 0;
	IDI_TX_TRACE_ENTER;

	if (IDI_PRIMARY_CHANNEL == channel)
		ret = queue_work_on(IWL_IDI_TX_WORK_CPU,
				    iwl_idi_tx_em->idi_tx_wq,
				    &iwl_idi_tx_em->idi_tx_high_worker);
	else
		ret = queue_work_on(IWL_IDI_TX_WORK_CPU,
				    iwl_idi_tx_em->idi_tx_wq,
				    &iwl_idi_tx_em->idi_tx_low_worker);
	IWL_IDI_TX_LOG("Enabled TX DMA work on channel %d", channel);

	IDI_TX_TRACE_EXIT;
	return ret;
}

/**
 * Runs the DMA transaction of copying from the SG linked list given by the
 * IDI DBB to the TX buffer which will be given for LLS prcessing.
 *
 * Returns 0 on success, negative value describing the error otherwise.
 */
static int iwl_idi_tx_em_work(u32 channel)
{
	int ret = 0;
	struct idi_sg_desc *sg_ll_tx = iwl_idi_get_sg_ll(channel);
	atomic_t *dma_enabled_tx = iwl_idi_get_dma_state(channel);
	bool call_inter;

	IDI_TX_TRACE_ENTER;
	/* Iterate over the sg list and copy to given tx buffer */
	while (NULL != sg_ll_tx) {

		/* Check that DMA is enabled in this channel -
		 * used in case this channel is disabled at run time */
		if (atomic_read(dma_enabled_tx) != IWL_IDI_ENABLED) {
			IDI_TX_TRACE_EXIT_RET_STR("TX DMA Disabled");
			ret = -EDMADISABLED;
			goto out;
		}

		/* DMA Copy */
		IWL_IDI_TX_LOG("TX: Pushing to pipe from 0x%p size %d",
			       iwl_idi_calc_dbb_addr(sg_ll_tx),
			       iwl_idi_calc_dbb_size(sg_ll_tx));

		ret = iwl_lls_em_push_to_pipe(channel,
					iwl_idi_calc_dbb_addr(sg_ll_tx),
					iwl_idi_calc_dbb_size(sg_ll_tx));
		if (ret) {
			IDI_TX_TRACE_EXIT_RET_STR("COPY ERROR");
			goto out;
		}

		/* Stop DMA processing and channel if this is the last link */
		if (iwl_idi_calc_dbb_EL_bit(sg_ll_tx)) {
			iwl_idi_set_sg_ll(channel, NULL);
			atomic_set(dma_enabled_tx, IWL_IDI_DISABLED);
		}

		/* Save interrupt call if needed */
		call_inter = !!iwl_idi_calc_dbb_I_bit(sg_ll_tx);

		/*advance to the next linked list */
		sg_ll_tx = iwl_idi_calc_dbb_next(sg_ll_tx);

		/* Now call the interrupt after the SG list has advanced
		 * and if this is the last link then sg_ll is null */
		if (call_inter)
			iwl_idi_call_inter(channel);
	}

out:
	IDI_TX_TRACE_EXIT;
	return ret;
}

/*
 * WORKER function for the IDI DMA TX LOW priority channel.
 */
static void iwl_idi_tx_em_low_work(struct work_struct *data)
{
	IDI_TX_TRACE_ENTER;

	/* Call the generic worker */
	iwl_idi_tx_em_work(IDI_SECONDARY_CHANNEL);

	IDI_TX_TRACE_EXIT;
}

/*
 * WORKER function for the IDI DMA TX HIGH priority channel.
 */
static void iwl_idi_tx_em_high_work(struct work_struct *data)
{
	IDI_TX_TRACE_ENTER;

	/* Call the generic worker */
	iwl_idi_tx_em_work(IDI_PRIMARY_CHANNEL);

	IDI_TX_TRACE_EXIT;
}
