/*
 * $Id: ScannerThread.java,v 1.4 2002/05/20 13:10:00 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2002 iKu Netzwerkl&ouml;sungen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openantivirus.virushammer;

import java.io.*;
import java.util.*;

import org.openantivirus.scanner.*;
import org.openantivirus.scanner.scanobject.*;

/**
 * Thread that scans the files, so that the GUI does not block
 *
 * Pattern-Roles:
 * @author  Kurt Huwig
 * @version $Revision: 1.4 $
 */
public class ScannerThread implements Runnable {
    public static final String VERSION =
        "$Id: ScannerThread.java,v 1.4 2002/05/20 13:10:00 kurti Exp $";
    
    /** Utility field holding list of ScannerListeners. */
    private transient ArrayList scannerListenerList;
    private transient ArrayList recursiveFileFilterListenerList;
    
    /** Holds value of property scanTargetList. */
    private ScanTargetList scanTargetList;
    
    /** Holds value of property trie. */
    private Trie trie;
    
    /** Holds value of property scanning. */
    private volatile boolean scanning;
    
    /** If the Thread should abort scanning */
    private volatile boolean abortRequested;
    
    private RecursiveFileFilter recursiveFileFilter;
    
    /** Listeners */
    private final RecursiveFileFilterListener recursiveFileFilterListener =
            new RecursiveFileFilterListener() {
        public void startingScan() {
            fireStartingScan();
        }
        
        public void scanning(File file) throws ScanningAbortedException {
            if (abortRequested) {
                throw new ScanningAbortedException("Aborted");
            }
            fireScanning(file);
        }
        
        public void virusFound(VirusFoundException virusFoundException) {
            fireVirusFound(virusFoundException);
        }
        
        public void finishedScan() {
            fireFinishedScan();
        }
        
        public void exceptionThrown(Exception exception) {}
    };
    
    private final ScannerConfiguration scannerConfiguration;
    private final Scanner scanner;
    
    public ScannerThread(ScannerConfiguration scannerConfiguration) {
        this.scannerConfiguration = scannerConfiguration;
        scanner = new Scanner(scannerConfiguration);
    }
    
    public void run() {
        scanning = false;
        while (true) {
            synchronized (this) {
                try {
                    wait();
                } catch (InterruptedException ie) {
                    // that's how it works
                }
                scanning       = true;
                abortRequested = false;
            } 
            fireStartingScan();
            for (int i = 0; i < scanTargetList.getSize(); i++) {
                final ScanTarget scanTarget =
                        (ScanTarget) scanTargetList.getElementAt(i);
                final File scanFile = scanTarget.getPath();
                try {
                    scanner.scan(new FileScanObject(scanFile));
//                    recursiveFileFilter.filter(scanTarget.getPath());
                } catch (ScanningAbortedException sae) {
                    abortRequested = false;
                    break;
                } catch (VirusFoundException vfe) {
                    vfe.setFile(scanFile);
                    fireVirusFound(vfe);
                } catch (Exception e) {
                    fireExceptionThrown(e);
                }
            }
            synchronized (this) {
                scanning = false;
            }
            fireFinishedScan();
        }
    }
    
    /** starts the scanning */
    public synchronized void startScanning() {
        notify();
    }
    
    /** notifies, that the scanning should be stopped */
    public synchronized void stopScanning() {
        abortRequested = true;
    }
    
    /** Registers ScannerListener to receive events.
     * @param listener The listener to register.
     */
    public synchronized void addScannerListener(ScannerListener listener) {
        if (scannerListenerList == null ) {
            scannerListenerList = new ArrayList();
        }
        scannerListenerList.add(listener);
    }
    
    /** Registers ScannerListener to receive events.
     * @param listener The listener to register.
     */
    public synchronized void addRecursiveFileFilterListener(
            RecursiveFileFilterListener listener) {
        if (recursiveFileFilterListenerList == null ) {
            recursiveFileFilterListenerList = new ArrayList();
        }
        recursiveFileFilterListenerList.add(listener);
        addScannerListener(listener);
    }
    
    /** Removes ScannerListener from the list of listeners.
     * @param listener The listener to remove.
     */
    public synchronized void removeScannerListener(ScannerListener listener) {
        if (scannerListenerList != null ) {
            scannerListenerList.remove(listener);
        }
    }
    
    /** Removes ScannerListener from the list of listeners.
     * @param listener The listener to remove.
     */
    public synchronized void removeRecursiveFileFilterListener(
            RecursiveFileFilterListener listener) {
        if (recursiveFileFilterListenerList != null ) {
            recursiveFileFilterListenerList.remove(listener);
        }
    }
    
    /** Setter for property scanTargetList.
     * @param scanTargetList New value of property scanTargetList.
     */
    public void setScanTargetList(ScanTargetList scanTargetList) {
        this.scanTargetList = scanTargetList;
    }
    
    /** Setter for property trie.
     * @param trie New value of property trie.
     */
    public void setTrie(Trie trie) {
        this.trie = trie;
        if (recursiveFileFilter != null) {
            recursiveFileFilter.removeRecursiveFileFilterListener(
                    recursiveFileFilterListener);
        }
        recursiveFileFilter = new RecursiveFileFilter(
                scannerConfiguration,new StreamFileFilter(trie));
        recursiveFileFilter.addRecursiveFileFilterListener(
                recursiveFileFilterListener);
    }
    
    /** Getter for property scanning.
     * @return Value of property scanning.
     */
    public synchronized boolean isScanning() {
        return this.scanning;
    }
    
    /** notifies the listeners, that scanning has started */
    protected synchronized void fireStartingScan() {
        if (scannerListenerList != null) {
            for (Iterator it = scannerListenerList.iterator();
                 it.hasNext(); ) {
                ((ScannerListener) it.next()).startingScan();
            }
        }
    }
    
    /** notifies the listeners, that scanning has finished */
    protected synchronized void fireFinishedScan() {
        if (scannerListenerList != null) {
            for (Iterator it = scannerListenerList.iterator();
                 it.hasNext(); ) {
                ((ScannerListener) it.next()).finishedScan();
            }
        }
    }
    
    /** notifies the listeners, that a virus has been found */
    protected synchronized void fireVirusFound(VirusFoundException vfe) {
        if (scannerListenerList != null) {
            for (Iterator it = scannerListenerList.iterator();
                 it.hasNext(); ) {
                ((ScannerListener) it.next()).virusFound(vfe);
            }
        }
    }
    
    /** notifies the listeners, that a file is about to be scanned */
    protected synchronized void fireScanning(File file)
            throws ScanningAbortedException {
        if (recursiveFileFilterListenerList != null) {
            for (Iterator it = recursiveFileFilterListenerList.iterator();
                 it.hasNext(); ) {
                ((RecursiveFileFilterListener) it.next()).scanning(file);
            }
        }
    }
    
    /** notifies the listeners, that an exception has been thrown */
    protected synchronized void fireExceptionThrown(Exception exception) {
        if (recursiveFileFilterListenerList != null) {
            for (Iterator it = recursiveFileFilterListenerList.iterator();
                 it.hasNext(); ) {
                ((ScannerListener) it.next()).exceptionThrown(exception);
            }
        }
    }
}
