/*
 * $Id: TemporaryScanObject.java,v 1.3 2002/05/20 13:10:35 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2002 iKu Netzwerkl&ouml;sungen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openantivirus.scanner.scanobject;

import java.io.*;

/**
 * Creates a temporary file that is deleted on 'dispose'
 *
 * Pattern-Roles:
 * @author  Kurt Huwig
 * @version $Revision: 1.3 $
 */
public class TemporaryScanObject extends ScanObject {
    public static final String VERSION =
        "$Id: TemporaryScanObject.java,v 1.3 2002/05/20 13:10:35 kurti Exp $";
    
    private static final String
            TEMP_PREFIX = "file",
            TEMP_SUFFIX = "oav";
    
    private static final int BUFFER_SIZE = 8192;
    private final File file;
    private byte[] start = null;

    public TemporaryScanObject(File directory) throws IOException {
        file = createTempFile(directory);
    }
    
    public TemporaryScanObject(InputStream is, File directory)
                throws IOException {
        file = createTempFile(directory);
        
        final OutputStream os = new FileOutputStream(file);
        
        // read the start of the stream
        int pos = 0;
        byte[] buffer = new byte[BUFFER_SIZE];
        int read;
        while ((read = is.read(buffer, pos, MAX_START_SIZE - pos))
               != -1) {
            os.write(buffer, pos, read);
            pos += read;
            if (pos == MAX_START_SIZE) {
                break;
            }
        }
        
        // copy the start into the start-array
        start = new byte[pos];
        for (int i = 0; i < pos; i++) {
            start[i] = buffer[i];
        }
        
        // read the rest of the stream
        while ((read = is.read(buffer)) != -1) {
            os.write(buffer, 0, read);
        }
        os.close();
    }

    protected File createTempFile(File directory) throws IOException {
        File tmpFile = File.createTempFile(TEMP_PREFIX, TEMP_SUFFIX, directory);
//        tmpFile.deleteOnExit(); // just in case
        return tmpFile;
    }
    
    /** called if the ScanObject is not needed any more  */
    public void dispose() throws IOException {
        file.delete();
    }    
    
    /** @return File for the object; the caller HAS TO close all streams to it  */
    public File getFile() throws IOException {
        return file;
    }
    
    /** @return the first block of bytes from the object; should be at least 2kB
     */
    public byte[] getStart() throws IOException {
        if (start == null) {
            final RandomAccessFile raf = new RandomAccessFile(file, "r");
            start = new byte[Math.min(MAX_START_SIZE, (int) file.length())];
            raf.readFully(start);
        }
            
        return start;
    }
    
}
