/*
 * $Id: UninfectableFilterFactory.java,v 1.1 2002/05/13 22:21:00 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2002 iKu Netzwerkl&ouml;sungen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openantivirus.scanner.filter;

import java.io.*;
import org.openantivirus.scanner.scanobject.*;

/**
 * This factory detects file formats, that cannot contain a virus, e.g. they
 * are too small or a graphic
 *
 * Pattern-Roles:
 * @author  Kurt Huwig
 * @version $Revision: 1.1 $
 */
public class UninfectableFilterFactory implements FilterFactory {
    public static final String VERSION =
        "$Id: UninfectableFilterFactory.java,v 1.1 2002/05/13 22:21:00 kurti Exp $";
    
    /** this one surprisingly is thread safe */
    private final Filter nullFilter = NullFilter.getInstance();
    
    private static final byte[][] MAGICS = {
        {0x00, 0x01, 0x00, 0x00, 0x00}, // MS true type font
        {'?', '_', 0x03, 0x00},         // MS help data
        {'%', 'P', 'D', 'F', '-'},      // PDF
        {'B', 'M'},                     // BMP image
        {'G', 'I', 'F', '8'},           // GIF image
        {'I', 'I', 0x2a, 0x00},         // TIFF image little-endian
        {'M', 'M', 0x00, 0x2a},         // TIFF image big-endian
        {'R', 'I', 'F', 'F'},           // RIFF multimedia
        {(byte) 0x89, 'P', 'N', 'G'},   // Portable Network Graphic
        {(byte) 0xff, (byte) 0xd8}      // JPEG image
    };
    
    /** @return Filter for the ScanObject or null if the factory does not know
     *         how to filter it
     */
    public Filter getFilterFor(ScanObject scanObject) throws IOException {
        final byte[] start = scanObject.getStart();
        if (start.length < 40) {
            // this is the size of EICAR
            return nullFilter;
        }
        
mag:    for (int magicNr = 0; magicNr < MAGICS.length; magicNr++) {
            byte[] magic = MAGICS[magicNr];
            for (int pos = 0; pos < magic.length; pos++) {
                if (magic[pos] != start[pos]) {
                    continue mag;
                }
            }
            return nullFilter;
        }

        return null;
    }
}
