/*
 * $Id: Trie.java,v 1.5 2002/10/16 11:01:44 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001 iKu Netzwerkl&ouml;sungen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openantivirus.scanner;

import java.util.*;

/**
 * Trie
 *
 * Pattern-Roles:
 * @author  Kurt Huwig
 * @version $Revision: 1.5 $
 */
public class Trie {
    public static final String VERSION =
        "$Id: Trie.java,v 1.5 2002/10/16 11:01:44 kurti Exp $";
    
    public final static int
        MINIMUM_LENGTH = 4;
    
    private Node nRoot = new Node();
    
    private int[] nodeDepthCount = new int[MINIMUM_LENGTH];
    
    public void addString(byte[] abPattern, PositionFoundListener pfl) {
        if (abPattern.length < MINIMUM_LENGTH) {
            throw new IllegalArgumentException("String too short");
        }
        
        // start at rootnode
        Node nPos = nRoot;
        
        // add nodes into the tree for the prefix with length MINIMUM_LENGTH
        for (int i = 0; i < MINIMUM_LENGTH; i++) {
            int iCharacter = (abPattern[i] + 256) & 0xff;
            
            Node next = (nPos.isLastNode() ? null : nPos.getTrans(iCharacter));
            if (next == null) {
                next = new Node();
                nPos.setTrans(iCharacter, next);
                nodeDepthCount[i]++;
            }
            
            nPos = next;
        }
        nPos.addPositionFoundListener(pfl);
    }
    
    public Node getRootNode() {
        return nRoot;
    }
    
    /**
     * Prepares the Trie for usage. This method has to be called before the
     * trie can be used
     */
    public void prepare() {
        // initialize the root node
        final LinkedList children = new LinkedList();
        nRoot.setFailure(null); // null = top node; to end following
        for (int i = 0; i < nRoot.NUM_CHILDS; i++) {
            final Node child = nRoot.getTrans(i);
            if (child == null) {
                nRoot.setTrans(i, nRoot);
            } else {
                child.setFailure(nRoot);
                children.addLast(child);
            }
        }
        
        while (!children.isEmpty()) {
            final Node node = (Node) children.removeFirst();
            if (node.isLastNode()) {
                continue;
            }
            for (int i = 0; i < node.NUM_CHILDS; i++) {
                final Node child = node.getTrans(i);
                if (child == null) {
                    node.setTrans(i, node.getFailure().getTrans(i));
                } else {
                    child.setFailure(node.getFailure().getTrans(i));
                    children.addLast(child);
                }
            }
        }
    }
    
    public int[] getNodeDepths() {
        return nodeDepthCount;
    }
}
