/*
 * $Id: ScannerDaemon.java,v 1.15 2002/10/16 11:01:44 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001-02 iKu Netzwerkl&ouml;sungen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */ 

package org.openantivirus.scanner;

import java.net.*;
import java.io.*;
import java.util.StringTokenizer;
import org.openantivirus.credo.*;

/**
 * Listens on a port to incoming scan request. Request is of the form
 *
 *  SCAN <filename><newline>
 *
 * Answers with
 *
 *  OK<newline>
 *
 * or
 *
 *  FOUND: <virusname> <virusname> <virusname>...<newline>
 *
 * @author  Kurt Huwig
 * @version $Revision: 1.15 $
 */
public class ScannerDaemon {
    public static final String VERSION =
        "$Id: ScannerDaemon.java,v 1.15 2002/10/16 11:01:44 kurti Exp $";
    
    public static final String REVISION = "0.5.2";
    
    public static final String BINDNAME = "localhost";
    public static final int PORT = 8127;
    
    private static final int MAX_BACKLOG = 50;
    
    private boolean isRunning;
    
    private final ScannerConfiguration scannerConfiguration;
    private final Scanner scanner;
    
    public ScannerDaemon(ScannerConfiguration scannerConfiguration)
                throws CredoException, IOException {
        scanner = new Scanner(scannerConfiguration);
        this.scannerConfiguration = scannerConfiguration;
        
        new CredoParser(scannerConfiguration).load();
        final Trie trie = scannerConfiguration.getTrie();
        trie.prepare();
        
        System.out.println("Nodes: " + Node.getInstanceCount());
        int[] depthNodeCount = trie.getNodeDepths();
        for (int i = 0; i < depthNodeCount.length; i++) {
            System.out.println(i + ": " + depthNodeCount[i]);
        }
    }
    
    public void start() {
        ServerSocket serverSocket = null;
        try {
            serverSocket = new ServerSocket(PORT, MAX_BACKLOG,
                                            InetAddress.getByName(BINDNAME));
        } catch (IOException ioe) {
            ioe.printStackTrace();
            return;
        }
        
        isRunning = true;
        while (isRunning) {
            try {
                new Thread(new RequestHandler(serverSocket.accept(), scanner))
                        .start();
            } catch(IOException ioe) {
                ioe.printStackTrace();
            }
        }
    }
    
    public static void main(String[] args) {
        System.out.println("OpenAntivirus ScannerDaemon v" + REVISION + "\n"
                + "(c) 2001-2002 iKu Netzwerklsungen http://www.iku-netz.de/\n"
                + "ScannerDaemon comes with ABSOLUTELY NO WARRANTY; for "
                + "details read 'COPYING'.\n"
                + "This is free software, and you are welcome to redistribute "
                + "it under certain\nconditions; for details read 'COPYING'.");
        
        final ScannerConfiguration scannerConfiguration =
                new ScannerConfiguration();
        
        for (int i = 0; i < args.length; i++) {
            if (args[i].charAt(0) == '-') {
                final String sParameter = args[i].substring(1);
                if ("nosignature".equals(sParameter)) {
                    scannerConfiguration.setCredoLevel(CredoParser.NO_VERIFY);
                } else if ("credolevel".equals(sParameter)) {
                    scannerConfiguration.setCredoLevel(
                            Integer.parseInt(args[++i]));
                } else if ("followsymlinks".equals(sParameter)) {
                    scannerConfiguration.setFollowSymlinks(true);
                } else if ("tempdir".equals(sParameter)) {
                    scannerConfiguration.setTempDirectory(new File(args[++i]));
                } else {
                    System.err.println("Unknown parameter: " + args[i]);
                }
            } else {
                scannerConfiguration.setCredoPath(args[i]);
            }
        }
        
        try {
            new ScannerDaemon(scannerConfiguration).start();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
