/*
 * $Id: ScannerConfiguration.java,v 1.4 2002/05/20 13:10:35 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001-02 iKu Netzwerkl&ouml;sungen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openantivirus.scanner;

import java.io.*;
import java.util.*;
import org.openantivirus.credo.*;

/**
 * Contains all information for the scanner itself, e.g. Trie. It is used as
 * a container, so that the API does not need to be changed if additional
 * parameters are needed.
 *
 * Pattern-Roles: Container
 * @author  Kurt Huwig
 * @version $Id: ScannerConfiguration.java,v 1.4 2002/05/20 13:10:35 kurti Exp $
 */

public class ScannerConfiguration {
    public static final String VERSION =
        "$Id: ScannerConfiguration.java,v 1.4 2002/05/20 13:10:35 kurti Exp $";
    
    private Trie trie = new Trie();
    
    /** Holds value of property followSymlinks; defaults to false */
    private boolean followSymlinks = false;
    
    /**
     * Holds value of property credoLevel; defaults to CredoParser.DEFAULT_LEVEL
     */
    private int credoLevel = CredoParser.DEFAULT_LEVEL;
    
    /** Holds value of property credoPath; defaults to 'credo' */
    private String credoPath = "credo";
    
    /** Holds value of property tempDirectory. */
    private File tempDirectory;
    
    private List listScannerListener = new ArrayList();
    
    public Trie getTrie() {
        return trie;
    }
    
    /** Getter for property followSymlinks.
     * @return Value of property followSymlinks.
     */
    public boolean getFollowSymlinks() {
        return this.followSymlinks;
    }
    
    /** Setter for property followSymlinks.
     * @param followSymlinks New value of property followSymlinks.
     */
    public synchronized void setFollowSymlinks(boolean followSymlinks) {
        this.followSymlinks = followSymlinks;
    }
    
    /** Getter for property credoLevel.
     * @return Value of property credoLevel.
     */
    public int getCredoLevel() {
        return this.credoLevel;
    }
    
    /** Setter for property credoLevel.
     * @param credoLevel New value of property credoLevel.
     */
    public synchronized void setCredoLevel(int credoLevel) {
        this.credoLevel = credoLevel;
    }
    
    /** Getter for property credoPath.
     * @return Value of property credoPath.
     */
    public String getCredoPath() {
        return this.credoPath;
    }
    
    /** Setter for property credoPath.
     * @param credoPath New value of property credoPath.
     */
    public synchronized void setCredoPath(String credoPath) {
        this.credoPath = credoPath;
    }
    
    public synchronized Object clone() {
        final ScannerConfiguration newConfiguration =
                new ScannerConfiguration();
        newConfiguration.followSymlinks = followSymlinks;
        newConfiguration.credoLevel     = credoLevel;
        newConfiguration.credoPath      = credoPath;
        newConfiguration.listScannerListener = listScannerListener;
        return newConfiguration;
    }
    
    public synchronized void setConfiguration(
                ScannerConfiguration newConfiguration) {
        this.followSymlinks = newConfiguration.followSymlinks;
        this.credoLevel     = newConfiguration.credoLevel;
        this.credoPath      = newConfiguration.credoPath;
        this.trie           = newConfiguration.trie;
        this.listScannerListener = newConfiguration.listScannerListener;
    }
    
    /** Getter for property tempDirectory.
     * @return Value of property tempDirectory.
     */
    public File getTempDirectory() {
        return this.tempDirectory;
    }
    
    /** Setter for property tempDirectory.
     * @param tempDirectory New value of property tempDirectory.
     */
    public synchronized void setTempDirectory(File tempDirectory) {
        this.tempDirectory = tempDirectory;
    }
    
    public synchronized void addScannerListener(ScannerListener listener) {
        listScannerListener.add(listener);
    }
    
    public synchronized void removeScannerListener(ScannerListener listener) {
        listScannerListener.remove(listener);
    }
    
    public synchronized void fireVirusFoundException(VirusFoundException vfe) {
        for (Iterator i = listScannerListener.iterator(); i.hasNext(); ) {
            ((ScannerListener) i.next()).virusFound(vfe);
        }
    }
    
    public synchronized void fireException(Exception e) {
        for (Iterator i = listScannerListener.iterator(); i.hasNext(); ) {
            ((ScannerListener) i.next()).exceptionThrown(e);
        }
    }
}
