package org.openantivirus.credo;

import java.util.*;
import org.openantivirus.scanner.*;
import java.io.*;

/**
 * Reads in Credo-files and initializes the corresponding Finders
 *
 * Pattern-Roles:
 * @author  Kurt Huwig
 * @version $Id: CredoParser.java,v 1.5 2002/04/23 08:54:00 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001-02 iKu Netzwerkl&ouml;sungen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
public class CredoParser {
    public static final String VERSION =
        "$Id: CredoParser.java,v 1.5 2002/04/23 08:54:00 kurti Exp $";
    
    public static final int NO_VERIFY     = -1;
    public static final int DEFAULT_LEVEL =  3;
    
    private final ScannerConfiguration scannerConfiguration;
    
    /**
     * @param verify if the digital signature of the Credo-files should be
     *               verified
     */
    public CredoParser(ScannerConfiguration scannerConfiguration) {
        this.scannerConfiguration = scannerConfiguration;
    }
    
    /**
     * Loads the Credo files in the path contained in the ScannerConfiguration
     */
    public void load() throws CredoException, IOException {
        parse(new File(scannerConfiguration.getCredoPath()));
    }
    
    /**
     * Recursively parses all credo files in this directory and subdirectories
     * if this is a directory; otherwise the file itself
     */
    public void parse(File file) throws CredoException, IOException {
        if (file.isDirectory()) {
            final File[] afFiles = file.listFiles(new FilenameFilter() {
                public boolean accept(File directory, String name) {
                    return name.endsWith(CredoFile.EXTENSION);
                }
            });
            for (int i = 0; i < afFiles.length; i++) {
                parse(afFiles[i]);
            }
        } else {
            doParse(new CredoFile(file));
        }
    }
    
    public void parse(InputStream is) throws CredoException, IOException {
        doParse(new CredoFile(is));
    }
    
    protected void doParse(CredoFile credoFile) throws CredoException,
                                                       IOException {
        for (Iterator it = credoFile.entries(); it.hasNext(); ) {
            final CredoEntry credoEntry = (CredoEntry) it.next();
            System.out.println("Reading '" + credoEntry.getJarEntry().getName()
                               + "'...");
            switch (credoEntry.getType()) {
                case CredoEntry.STRINGS:
                    new StringVirusFinder(new InputStreamReader(
                            credoEntry.getInputStream()), new StringFinder(
                                    scannerConfiguration.getTrie()));
                    break;
                default:
                    throw new CredoException("Unknown CredoEntry-type: "
                                             + credoEntry.getType());
            }
            int verifyLevel = scannerConfiguration.getCredoLevel();
            if (verifyLevel != NO_VERIFY) {
                int credoLevel = CredoVerifier.verify(credoEntry);
                if (credoLevel < verifyLevel) {
                    throw new CredoException("Minimum Credo-level "
                            + verifyLevel + " > actual Credo-level "
                            + credoLevel);
                } else {
                    System.out.println("  verified Credo-level " + credoLevel);
                }
            }
        }
    }
}
