/*
 * $Id: ScannerDaemon.java,v 1.11 2002/04/11 09:54:22 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001 iKu Netzwerkl&ouml;sungen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */ 

package org.openantivirus.scanner;

import java.net.*;
import java.io.*;
import java.util.StringTokenizer;
import org.openantivirus.credo.*;

/**
 * Listens on a port to incoming scan request. Request is of the form
 *
 *  SCAN <filename><newline>
 *
 * Answers with
 *
 *  OK<newline>
 *
 * or
 *
 *  FOUND: <virusname> <virusname> <virusname>...<newline>
 *
 * @author  Kurt Huwig
 * @version $Revision: 1.11 $
 */
public class ScannerDaemon {
    public static final String VERSION =
        "$Id: ScannerDaemon.java,v 1.11 2002/04/11 09:54:22 kurti Exp $";
    
    public static final String BINDNAME = "localhost";
    public static final int PORT = 8127;
    public static final String CREDO_DIRECTORY = "credo";
    
    private static final int MAX_BACKLOG = 50;
    
    private boolean isRunning;
    private final ScannerConfiguration scannerConfiguration =
            new ScannerConfiguration();
    
    public ScannerDaemon(String sSignatureFile, int credoLevel)
                throws CredoException, IOException {
        
        new CredoParser(scannerConfiguration, credoLevel).parse(
                new File(sSignatureFile != null ? sSignatureFile
                                                : CREDO_DIRECTORY));
        final Trie trie = scannerConfiguration.getTrie();
        trie.prepare();
        
        System.out.println("Nodes: " + Node.getInstanceCount());
        int[] depthNodeCount = trie.getNodeDepths();
        for (int i = 0; i < depthNodeCount.length; i++) {
            System.out.println(i + ": " + depthNodeCount[i]);
        }
    }
    
    public void start() {
        ServerSocket serverSocket = null;
        try {
            serverSocket = new ServerSocket(PORT, MAX_BACKLOG,
                                            InetAddress.getByName(BINDNAME));
        } catch (IOException ioe) {
            ioe.printStackTrace();
            return;
        }
        
        isRunning = true;
        while (isRunning) {
            try {
                new Thread(new RequestHandler(serverSocket.accept(),
                                              scannerConfiguration.getTrie()))
                        .start();
            } catch(IOException ioe) {
                ioe.printStackTrace();
            }
        }
    }
    
    public static void main(String[] args) {
        String sVirusSignatures = null;
        int credoLevel = CredoParser.DEFAULT_LEVEL;
        
        for (int i = 0; i < args.length; i++) {
            if (args[i].charAt(0) == '-') {
                final String sParameter = args[i].substring(1);
                if ("nosignature".equals(sParameter)) {
                    credoLevel = CredoParser.NO_VERIFY;
                } else if ("credolevel".equals(sParameter)) {
                    credoLevel = Integer.parseInt(args[++i]);
                } else {
                    System.err.println("Unknown parameter: " + args[i]);
                }
            } else {
                sVirusSignatures = args[i];
            }
        }
        
        try {
            new ScannerDaemon(sVirusSignatures, credoLevel).start();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
