/*
 * $Id: RecursiveFileFilter.java,v 1.4 2002/02/24 16:15:10 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001 iKu Netzwerkl&ouml;sungen
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openantivirus.scanner;

import java.io.*;
import java.util.*;

/**
 * Recursively filters all files
 *
 * @author  Kurt Huwig
 * @version $Revision: 1.4 $
 */
public class RecursiveFileFilter {
    public static final String VERSION =
        "$Id: RecursiveFileFilter.java,v 1.4 2002/02/24 16:15:10 kurti Exp $";

    private StreamFileFilter sff;
    
    /** Utility field holding list of RecursiveFileFilterListeners. */
    private transient ArrayList recursiveFileFilterListenerList;
    
    public RecursiveFileFilter(StreamFileFilter sff) {
        this.sff = sff;
    }
    
    public void filter(File f) throws VirusFoundException, IOException,
                                      ScanningAbortedException {
        filter2(f);
    }
    
    protected void filter2(File f) throws VirusFoundException, IOException,
                                          ScanningAbortedException {
        if (f.isDirectory()) {
            File[] afFiles = f.listFiles();
            for (int i = 0; i < afFiles.length; i++) {
                filter(afFiles[ i ]);
            }
        } else {
            long lStart = System.currentTimeMillis();
            try {
                fireScanning(f);
                sff.filter(new RandomAccessFile(f, "r"));
            } catch (VirusFoundException vfe) {
                fireVirusFound(vfe);
            }
        }
    }
    
    /** Registers RecursiveFileFilterListener to receive events.
     * @param listener The listener to register.
     */
    public synchronized void addRecursiveFileFilterListener(
            RecursiveFileFilterListener listener) {
        if (recursiveFileFilterListenerList == null) {
            recursiveFileFilterListenerList = new java.util.ArrayList();
        }
        recursiveFileFilterListenerList.add(listener);
    }
    
    /** Removes RecursiveFileFilterListener from the list of listeners.
     * @param listener The listener to remove.
     */
    public synchronized void removeRecursiveFileFilterListener(
            RecursiveFileFilterListener listener) {
        if (recursiveFileFilterListenerList != null) {
            recursiveFileFilterListenerList.remove(listener);
        }
    }
    
    /** notifies the listeners, that scanning has started */
    protected synchronized void fireStartingScan() {
        if (recursiveFileFilterListenerList != null) {
            for (Iterator it = recursiveFileFilterListenerList.iterator();
                 it.hasNext(); ) {
                ((RecursiveFileFilterListener) it.next()).startingScan();
            }
        }
    }
    
    /** notifies the listeners, that scanning has started */
    protected synchronized void fireFinishedScan() {
        if (recursiveFileFilterListenerList != null) {
            for (Iterator it = recursiveFileFilterListenerList.iterator();
                 it.hasNext(); ) {
                ((RecursiveFileFilterListener) it.next()).finishedScan();
            }
        }
    }
    
    /** notifies the listeners, that scanning has started */
    protected synchronized void fireScanning(File file)
            throws ScanningAbortedException {
        if (recursiveFileFilterListenerList != null) {
            for (Iterator it = recursiveFileFilterListenerList.iterator();
                 it.hasNext(); ) {
                ((RecursiveFileFilterListener) it.next()).scanning(file);
            }
        }
    }
    
    /** notifies the listeners, that scanning has started */
    protected synchronized void fireVirusFound(VirusFoundException vfe)
            throws VirusFoundException {
        if (recursiveFileFilterListenerList != null) {
            for (Iterator it = recursiveFileFilterListenerList.iterator();
                 it.hasNext(); ) {
                ((RecursiveFileFilterListener) it.next()).virusFound(vfe);
            }
        }
    }
}
