package org.openantivirus.credo;

import java.security.PublicKey;
import java.security.cert.*;
import java.io.*;
import java.util.jar.*;

/**
 * Verifies digital signatures of Credo files
 *
 * Pattern-Roles:
 * @author  Kurt Huwig
 * @version $Id: CredoVerifier.java,v 1.3 2002/04/11 15:34:16 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001 iKu Netzwerkl&ouml;sungen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
public class CredoVerifier {
    public static final String VERSION =
        "$Id: CredoVerifier.java,v 1.3 2002/04/11 15:34:16 kurti Exp $";
    
    /** Number of signature levels */
    public static final int SIGNATURE_LEVELS = 4;
    
    /** Path to the signing certificate */
    private static final String CERTIFICATE_PATH = "/oav.cer";
    
    /** Type of the certificate */
    private static final String CERTIFICATE_TYPE = "X.509";
    
    /**
     * There can be only one! We trust noone besides ourselves :-)
     */
    private static final Certificate[] oavCertificate =
            new Certificate[SIGNATURE_LEVELS];
    
    private static final PublicKey[] oavPublicKey =
            new PublicKey[SIGNATURE_LEVELS];
    
    static {
        for (int level = 0; level < SIGNATURE_LEVELS; level++) {
            try {
                final InputStream is = CredoEntry.class.getResourceAsStream(
                        CERTIFICATE_PATH + ".level" + (level + 1));
                final CertificateFactory cf = CertificateFactory.getInstance(
                        CERTIFICATE_TYPE);
                oavCertificate[level] = cf.generateCertificate(is);
                try {
                    is.close();
                } catch (IOException ioe) {
                    ioe.printStackTrace();
                }
                oavPublicKey[level] = oavCertificate[level].getPublicKey();
            } catch (CertificateException ce) {
                ce.printStackTrace();
            }
        }
    }
    
    /**
     * Verifies the digital signature of this entry; the data of the entry
     * has to be read completely before calling this method.
     *
     * @throws CredoException If the digital signature is invalid
     * @return signature level
     */
    public static int verify(CredoEntry credoEntry) throws CredoException {
        final Certificate[] certificates =
                credoEntry.getJarEntry().getCertificates();
        if (certificates == null) {
            throw new CredoException(
                    "No signature found or entry not fully read");
        }
        int verifiedLevel = -1;
cert:   for (int i = 0; i < certificates.length; i++) {
            for (int level = 0; level < SIGNATURE_LEVELS; level++) {
                if (certificates[i].equals(oavCertificate[level])) {
                    continue cert;
                }
            }
            for (int level = 0; level < SIGNATURE_LEVELS; level++) {
                try {
                    certificates[i].verify(oavPublicKey[level]);
                    System.out.println("  signed by '"
                            + ((X509Certificate)certificates[i])
                              .getSubjectDN() + "'");
                    verifiedLevel = level;
                    //break cert;
                } catch (Exception e) {
                    // we have several certificates; all but one will fail
                }
            }
        }
        if (verifiedLevel == -1) {
            throw new CredoException("No valid signing certificate found");
        }
        return (verifiedLevel + 1);
    }    
}
