/*
 * $Id: VirusHammer.java,v 1.6 2002/04/23 08:54:07 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001-02 iKu Netzwerkl&ouml;sungen
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openantivirus.virushammer;

import java.awt.*;
import java.awt.event.*;
import java.io.*;
import java.text.*;
import java.util.*;
import javax.swing.*;

import java.beans.*;

import org.openantivirus.scanner.*;
import org.openantivirus.credo.*;

/**
 * L10N
 *
 * Pattern-Roles: 
 * @author  Kurt Huwig
 * @version $Revision: 1.6 $
 */
public class VirusHammer {
    public static final String VERSION =
        "$Id: VirusHammer.java,v 1.6 2002/04/23 08:54:07 kurti Exp $";
    
    /** Actions */
    private final Action
            patternFindAction = new PatternFindAction(),
            preferencesAction = new PreferencesAction();
    
    private final StartScanAction startScanAction = new StartScanAction();
    private final StopScanAction  stopScanAction  = new StopScanAction();
    private final ExitAction      exitAction      = new ExitAction();
    
    /** List of all the targets to scan */
    private final ScanTargetList scanTargetList = new ScanTargetList();
    
    private final JFrame frame = new JFrame();
    private final ScannerConfiguration scannerConfiguration =
            new ScannerConfiguration();
    private final ScannerThread scannerThread =
            new ScannerThread(scannerConfiguration);
    private final StatusModel statusModel = new StatusModel();
    private final ScannerOutputPanel scannerOutputPanel =
            new ScannerOutputPanel();
    
    /** Listener */
    private final RecursiveFileFilterListener recursiveFileFilterListener =
            new RecursiveFileFilterListener() {
        private File scanFile = null;
        private File parent   = null;
        
        public void startingScan() {
            scannerOutputPanel.clearList();
        }
        
        public void scanning(File file) {
            scanFile = file;
            if (file.getParentFile() != parent) {
                parent = file.getParentFile();
                statusModel.setText(MessageFormat.format(
                        L10N.getString("Scanning_file"),
                        new Object[] {parent.getAbsolutePath()}));
            }
        }
        
        public void virusFound(VirusFoundException virusFoundException)
                throws VirusFoundException {
            scannerOutputPanel.addFoundVirus(scanFile, 
                    virusFoundException.getVirusName());
        }
        
        public void finishedScan() {
            scanFile = parent = null;
            statusModel.setText(L10N.getString("Idle"));
        }
    };
    
    public VirusHammer() throws CredoException, IOException {
        initScanner();
        initScanTargetList();
        
        frame.setTitle(L10N.getString("VirusHammer") + " "
                       + L10N.getString("VirusHammerVersion"));
        
        frame.setJMenuBar(createMenuBar());
        
        frame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
        frame.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent we) {
                if (exitAction.isEnabled()) {
                    exitAction.actionPerformed(new ActionEvent(we.getSource(),
                            we.getID(), null));
                }
            }
        });
        
        createComponents();
        initActions();
        
        frame.setSize(640, 480);
        frame.setVisible(true);
        statusModel.setText(L10N.getString("Idle"));
    }
    
    protected void initScanner() throws CredoException, IOException {
        new CredoParser(scannerConfiguration).parse(
                this.getClass().getResourceAsStream("/VirusSignatures.credo"));
        scannerConfiguration.getTrie().prepare();
    }
    
    protected JMenuBar createMenuBar() {
        final JMenuBar jmb = new JMenuBar();
        
        JMenu jm;
        
        // File menu
        jm = new JMenu(L10N.getString("File"));
        jm.setMnemonic(L10N.getMnemonic("File"));
        jm.add(preferencesAction).setMnemonic(L10N.getMnemonic("Preferences"));
        jm.addSeparator();
        jm.add(exitAction).setMnemonic(L10N.getMnemonic("Exit"));
        jmb.add(jm);
        
        // Tools menu
        jm = new JMenu(L10N.getString("Tools"));
        jm.setMnemonic(L10N.getMnemonic("Tools"));
        jm.add(patternFindAction).setMnemonic(L10N.getMnemonic("Find_pattern"));
        jmb.add(jm);
        
        return jmb;
    }
    
    protected void createComponents() {
        // JPanel for the center of the frame, so that the toolbar can be
        // moved
        final JPanel jpCenter  = new JPanel(new BorderLayout());
        final JPanel jpExpl    = new JPanel(new BorderLayout());
        final JPanel jpTargets = new JPanel(new BorderLayout());
        final JPanel jpEast    = new JPanel(new GridBagLayout());
        final JSplitPane splitPane = new JSplitPane(JSplitPane.VERTICAL_SPLIT,
                jpExpl, scannerOutputPanel);
        splitPane.setOneTouchExpandable(true);
        splitPane.setDividerLocation(300);
        jpCenter.add(splitPane,         BorderLayout.CENTER);
        jpCenter.add(createStatusBar(), BorderLayout.SOUTH);
        frame.getContentPane().add(jpCenter, BorderLayout.CENTER);
        jpExpl.add(jpTargets, BorderLayout.CENTER);
        jpExpl.add(jpEast,    BorderLayout.EAST);
        
        jpTargets.setBorder(BorderFactory.createTitledBorder(
                L10N.getString("Scan_targets")));
        final ExplorerPanel explorerPanel = new ExplorerPanel();
        explorerPanel.setScanTargetList(scanTargetList);
        jpTargets.add(explorerPanel, BorderLayout.CENTER);
        
        final GridBagConstraints gbcEastButton = new GridBagConstraints();
        gbcEastButton.gridwidth = GridBagConstraints.REMAINDER;
        gbcEastButton.insets    = new Insets(0, 0, 5, 0);
        gbcEastButton.fill      = GridBagConstraints.HORIZONTAL;
        
        JButton jb = new JButton(startScanAction);
        jb.setMnemonic(L10N.getMnemonic("Start_scanning"));
        jpEast.add(jb, gbcEastButton);
        
        jb = new JButton(stopScanAction);
        jb.setMnemonic(L10N.getMnemonic("Start_scanning"));
        jpEast.add(jb, gbcEastButton);
    }
    
    protected JPanel createStatusBar() {
        final JPanel statusBar = new JPanel(new BorderLayout());
        final JLabel status = new JLabel();
        status.setBorder(BorderFactory.createEtchedBorder());
        statusBar.add(status, BorderLayout.CENTER);
        statusModel.addPropertyChangeListener("text",
                                              new PropertyChangeListener() {
            public void propertyChange(PropertyChangeEvent pce) {
                status.setText(statusModel.getText());
            }
        });
        return statusBar;
    }
    
    protected void initActions() {
        new Thread(scannerThread).start();
        scannerThread.setScanTargetList(scanTargetList);
        scannerThread.setTrie(scannerConfiguration.getTrie());
        scannerThread.addRecursiveFileFilterListener(recursiveFileFilterListener);
        startScanAction.setScanTargetList(scanTargetList);
        startScanAction.setScannerThread(scannerThread);
        stopScanAction.setScannerThread(scannerThread);
        exitAction.setScanTargetList(scanTargetList);
    }
    
    /**
     * initializes the list of ScanTargets with the settings from the
     * properties
     */
    protected void initScanTargetList() throws IOException {
        final Properties appProperties = new Properties();
        final File fileProperties = new File(System.getProperty("user.home")
                + File.separatorChar + "VirusHammer.properties");
        if (fileProperties.exists()) {
            final FileInputStream fis = new FileInputStream(fileProperties);
            appProperties.load(fis);
            fis.close();
            final String sTargetList = appProperties.getProperty(
                    "scantargets");
            if (sTargetList != null) {
                final StringTokenizer st = new StringTokenizer(sTargetList,
                        String.valueOf(File.pathSeparatorChar));
                while (st.hasMoreTokens()) {
                    final File target = new File(st.nextToken());
                    scanTargetList.addScanTarget(new ScanTarget(
                            target, target.isDirectory()));
                }
            }
        }
    }
    
    public static void main(String[] args) {
        try {
            L10N.setResourceBundle(ResourceBundle.getBundle(
                    "org/openantivirus/virushammer/VirusHammer"));
            new VirusHammer();
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
}
