/*
 * $Id: StringFinderTest.java,v 1.2 2002/04/22 09:44:24 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001 iKu Netzwerkl&ouml;sungen
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openantivirus.scanner;

import junit.framework.*;

/**
 * StringFinderTest
 *
 * Pattern-Roles:
 * @author  Kurt Huwig
 * @version $Revision: 1.2 $
 */
public class StringFinderTest extends TestCase {
    public static final String VERSION =
        "$Id: StringFinderTest.java,v 1.2 2002/04/22 09:44:24 kurti Exp $";
    
    private Trie         trie;
    private TrieScanner  trieScanner;
    private StringFinder stringFinder;
    
    
    public StringFinderTest(java.lang.String testName) {
        super(testName);
        
        trie = new Trie();
        trieScanner  = new TrieScanner(trie);
        stringFinder = new StringFinder(trie);
        stringFinder.addString("ABCDE".getBytes(), new PositionFoundListener() {
            public void positionFound(PositionFoundEvent pfe)
                    throws VirusFoundException {
                throw new VirusFoundException("Found");
            }
        });
        trie.prepare();        
    }        
    
    public static Test suite() {
        TestSuite suite = new TestSuite(StringFinderTest.class);
        
        return suite;
    }
    
    public void testNormal() {
        String sTest = "1234567890ABCDEabcdefghij";
        try {
            trieScanner.reset();
            trieScanner.update(sTest.getBytes(), 0, sTest.length(), 0, 0);
            fail("Testpattern 1 not found");
        } catch (VirusFoundException vfe) {
            // good!
        }
        
        sTest = "ABCDEabcdefghij";
        try {
            trieScanner.reset();
            trieScanner.update(sTest.getBytes(), 0, sTest.length(), 0, 0);
            fail("Testpattern 2 not found");
        } catch (VirusFoundException vfe) {
            // good!
        }
        
        sTest = "1234567890ABCDE";
        try {
            trieScanner.reset();
            trieScanner.update(sTest.getBytes(), 0, sTest.length(), 0, 0);
            fail("Testpattern 3 not found");
        } catch (VirusFoundException vfe) {
            // good!
        }
        
        sTest = "1234567890ABCDxEabcdefghij";
        try {
            trieScanner.reset();
            trieScanner.update(sTest.getBytes(), 0, sTest.length(), 0, 0);
        } catch (VirusFoundException vfe) {
            fail("Nonexistent pattern 4 found");
        }
    }
    
    /** tests updates across the ringbuffer */
    public void testRingBuffer() {
        byte[] abBuffer = "CDEabcdefghijkAB".getBytes();
        try {
            trieScanner.reset();
            trieScanner.update(abBuffer, 8, 8, 0, 0);
            trieScanner.update(abBuffer, 0, 8, 0, 0);
            
            fail("Ringbuffertest 1");
        } catch (VirusFoundException vfe) {
            // good!
        }
        
        abBuffer = "DEabcdefghijkABC".getBytes();
        try {
            trieScanner.reset();
            trieScanner.update(abBuffer, 8, 8, 0, 4);
            
            fail("Ringbuffertest 2");
        } catch (VirusFoundException vfe) {
            // good!
        }
        
        abBuffer = "abcdefghijkABCDE01234".getBytes();
        try {
            trieScanner.reset();
            trieScanner.update(abBuffer, 11, 4, 0, 0);
            
        } catch (VirusFoundException vfe) {
            fail("Bounds not respected 1");
        }
        
        try {
            trieScanner.reset();
            trieScanner.update(abBuffer, 11, 3, 0, 1);
            
        } catch (VirusFoundException vfe) {
            fail("Bounds not respected 2");
        }
        
        try {
            trieScanner.reset();
            trieScanner.update(abBuffer, 9, 5, 0, 2);
            
            fail("Ringbuffertest 3");
        } catch (VirusFoundException vfe) {
            // good!
        }        
    }
}
