/*
 * $Id: StringVirusFinder.java,v 1.3 2002/01/12 13:48:04 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001 iKu Netzwerkl&ouml;sungen
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openantivirus.scanner;

import java.io.*;
import java.security.*;
import java.security.spec.X509EncodedKeySpec;

/**
 * Detects virii by a simple string
 *
 * Pattern-Roles: 
 * @author  Kurt Huwig
 * @version $Revision: 1.3 $
 */
public class StringVirusFinder {
    public static String VERSION =
        "$Id: StringVirusFinder.java,v 1.3 2002/01/12 13:48:04 kurti Exp $";
    
    public StringVirusFinder(Reader patternReader, StringFinder stringFinder,
                             boolean allowNoSignature)
                throws IOException {
        BufferedReader br = new BufferedReader(patternReader);
        
        // Read public key
        StringBuffer sb = new StringBuffer();
        InputStream is = this.getClass().getResourceAsStream("/publickey.x509");
        int iRead;
        while ((iRead = is.read()) != -1) {
            sb.append((char) iRead);
        }
        is.close();

        try {
            // init signature
            X509EncodedKeySpec pubKeySpec = new X509EncodedKeySpec(
                    sb.toString().getBytes());
            KeyFactory keyFactory = KeyFactory.getInstance("DSA");
            PublicKey pubKey = keyFactory.generatePublic(pubKeySpec);
            Signature signature = Signature.getInstance("SHA1withDSA");
            signature.initVerify(pubKey);
            
            String sLine;
            while ((sLine = br.readLine()) != null) {
                int iPos = sLine.indexOf('=');
                if (iPos == -1) {
                    System.err.println("Malformed pattern line: " + sLine);
                    continue;
                }
                
                // is this the signature?
                if (sLine.startsWith("==")) {
                    if (signature.verify(
                                hexToString(sLine.substring(2)).getBytes())) {
                        System.out.println("Sucessfully verified signature.");
                        return;
                    } else {
                        System.err.println("Invalid signature. Exiting.");
                        System.exit(1);
                    }
                    
                } else {
                    signature.update(sLine.getBytes());
                }
                
                String sVirusName = sLine.substring(0, iPos);
                String sPattern   = sLine.substring(iPos + 1);
                
                try {
                    stringFinder.addString(hexToString(sPattern),
                            new StringVirusFoundListener(sVirusName));
                } catch (Exception e) {
                    System.err.println(sLine);
                    e.printStackTrace();
                }
            }
            br.close();
        } catch (GeneralSecurityException gse) {
            gse.printStackTrace();
            System.exit(1);
        }
        
        if (!allowNoSignature) {
            System.err.println("No signature found. Read the manual on how to "
                    + "continue anyway. Exiting.");
            System.exit(1);
        }
    }
    
    protected String hexToString(String hex) {
        if (hex.length() % 2 != 0) {
            System.err.println("Malformed hexstring: " + hex);
            return null;
        }
        
        StringBuffer sb = new StringBuffer();
        for (int i = 0; i < hex.length(); i += 2) {
            sb.append((char) Integer.parseInt(hex.substring(i, i + 2), 16));
        }
        
        return sb.toString();
    }
            

    
    private class StringVirusFoundListener implements PositionFoundListener {
        private String virusName, pattern;
        
        public StringVirusFoundListener(String virusName) {
            this.virusName = virusName;
        }
        
        public void positionFound(PositionFoundEvent sfe)
                throws VirusFoundException {
            throw new VirusFoundException(virusName);
        }
    }
}
