/*
 * $Id: Trie.java,v 1.2 2001/12/28 16:32:23 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001 iKu Netzwerkl&ouml;sungen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openantivirus.scanner;

import java.util.*;

/**
 * Trie
 *
 * Pattern-Roles:
 * @author  Kurt Huwig
 * @version $Revision: 1.2 $
 */
public class Trie {
    public static final String VERSION =
        "$Id: Trie.java,v 1.2 2001/12/28 16:32:23 kurti Exp $";
    
    public final static int
        MINIMUM_LENGTH = 3;
    
    private Node nRoot = new Node();
    
    private int[] nodeDepthCount = new int[MINIMUM_LENGTH];
    
    public void addString(String sPattern, PositionFoundListener pfl) {
        if (sPattern.length() < MINIMUM_LENGTH) {
            throw new IllegalArgumentException("String too short");
        }
        
        // start at rootnode
        Node nPos = nRoot;
        
        // add nodes into the tree for the prefix with length MINIMUM_LENGTH
        for (int i = 0; i < MINIMUM_LENGTH; i++) {
            int iCharacter = sPattern.charAt(i);
            
            Node next = nPos.getNext(iCharacter);
            if (next == null) {
                next = new Node();
                nPos.setNext(iCharacter, next);
                nodeDepthCount[i]++;
            }
            
            nPos = next;
        }
        nPos.setLastNode(true);
        nPos.addPositionFoundListener(pfl);
    }
    
    protected void createFailureTransitions() {
        nRoot.setFailure(null); // null == failure
        
        List liBFS = new LinkedList();
        for (int i = 0; i < Node.NUM_CHILDS; i++) {
            Node n = nRoot.getNext(i);
            if (n != null) {
                n.setFailure(nRoot);
                liBFS.add(n);
            }
        }
        
        while (!liBFS.isEmpty()) {
            List liBFSNext = new LinkedList();
            
            for (Iterator i = liBFS.iterator(); i.hasNext();) {
                Node q = (Node) i.next();
                for (int c = 0; c < Node.NUM_CHILDS; c++) {
                    Node qs = q.getNext(c);
                    if (qs != null) {
                        liBFSNext.add(qs);
                        Node r = q.getFailure();
                        while (r != null && r.getNext( c ) == null) {
                            r = r.getFailure();
                        }
                        
                        if (r == null) {
                            qs.setFailure(nRoot.getNext(c));
                        } else {
                            qs.setFailure(r.getNext(c));
                        }
                    }
                }
            }
            
            liBFS = liBFSNext;
        }
    }
    
    protected void createFastTrie(Node n) {
        for (int c = 0; c < n.NUM_CHILDS; c++) {
            Node next = n.getNext(c);
            if (next == null) {
                Node q = n;
                do {
                    q = q.getFailure();
                } while (q != null && q.getNext(c) == null);
                n.setTrans(c, q == null ? nRoot : q.getNext(c));
            } else {
                n.setTrans(c, next);
                if (!n.isLastNode()) {
                    createFastTrie(next);
                }
            }
        }
    }
    
    public Node getRootNode() {
        return nRoot;
    }
    
    /**
     * Prepares the Trie for usage. This method has to be called before the
     * trie can be used
     */
    public void prepare() {
        createFailureTransitions();
        createFastTrie(nRoot);
        // XXX: free unused nodes
    }
    
    public int[] getNodeDepths() {
        return nodeDepthCount;
    }
}