/*
 * $Id: SequentialStreamFilter.java,v 1.3 2002/01/04 15:23:09 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001 iKu Netzwerkl&ouml;sungen
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openantivirus.scanner;

import java.io.*;

/**
 * SequentialStreamFilter
 *
 * sends the InputStream sequentially to the StreamScanner
 *
 * Pattern-Roles: 
 * @author  Kurt Huwig
 * @version $Revision: 1.3 $
 */
public class SequentialStreamFilter {
    public static String VERSION =
        "$Id: SequentialStreamFilter.java,v 1.3 2002/01/04 15:23:09 kurti Exp $";
    
    private static final int BLOCK_SIZE = 4096;
    
    private StreamScanner ss;
    
    public SequentialStreamFilter(Trie trie) {
        ss = new TrieScanner(trie);
    }
    
    /**
     * filters the given file unaltered to the scanner
     */
    public void filter(InputStream is, OutputStream os)
            throws VirusFoundException, IOException {
        ss.reset();
        try {
            byte[] abBuffer = new byte[3 * BLOCK_SIZE];
            
            int iLength = fillBuffer(is, abBuffer, 0, abBuffer.length);
            
            if (iLength == -1) {
                return;
                
            } else if (iLength < abBuffer.length) {
                ss.update(abBuffer, 0, iLength, 0, 0);
                if (os != null) {
                    os.write(abBuffer, 0, iLength);
                }
                
            } else  {
                ss.update(abBuffer, 0, 2 * BLOCK_SIZE, 0, BLOCK_SIZE);
                if (os != null) {
                    os.write(abBuffer, 0, 2 * BLOCK_SIZE);
                }

                int iBufferPos    = 2 * BLOCK_SIZE;
                int iReadAheadPos = 0;
                do {
                    iLength = fillBuffer(is, abBuffer, iReadAheadPos,
                                         BLOCK_SIZE);
                    ss.update(abBuffer, iBufferPos, BLOCK_SIZE, BLOCK_SIZE,
                              iLength != -1 ? iLength : 0);
                    if (os != null) {
                        os.write(abBuffer, iBufferPos, BLOCK_SIZE);
                    }
                    iBufferPos    = iReadAheadPos;
                    iReadAheadPos += BLOCK_SIZE;
                    iReadAheadPos %= 3 * BLOCK_SIZE;
                } while (iLength == BLOCK_SIZE);
                
                if (iLength != -1) {
                    ss.update(abBuffer, iBufferPos, iLength, BLOCK_SIZE, 0);
                    if (os != null) {
                        os.write(abBuffer, iBufferPos, iLength);
                    }
                }
            }
        } finally {
            is.close();
        }
    }
    
    /**
     * fills the buffer up to the end unless the end of data is reached, or
     * an IOException occurs
     */
    protected int fillBuffer(InputStream stream, byte[] buffer,
                             int offset, int length) throws IOException {
        int iRead = 0;
        do {
            int iLength = stream.read(buffer, offset + iRead, length - iRead);
            if (iLength == -1) {
                return iRead == 0 ? -1 : iRead;
            }
            iRead += iLength;
        } while (iRead < length);
        return iRead;
    }
}
