/*
 * $Id: RequestHandler.java,v 1.3 2002/01/04 02:31:25 kurti Exp $
 *
 * This file is part of the OpenAntiVirus-Project,
 * see http://www.openantivirus.org/
 * (c) 2001 iKu Netzwerkl&ouml;sungen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

package org.openantivirus.scanner;

import java.io.*;
import java.net.*;
import java.util.*;

/**
 * ScannerThread
 *
 * Handles a single scan request
 *
 * Pattern-Roles:
 * @author  Kurt Huwig
 * @version $Revision: 1.3 $
 */
public class RequestHandler implements Runnable {
    public static String VERSION =
    "$Id: RequestHandler.java,v 1.3 2002/01/04 02:31:25 kurti Exp $";
    
    public static final String OK    = "OK\n",
                               ERROR = "ERROR\n";

    
    /** timeout for data connections in milliseconds; use whole seconds */
    private static final int DATA_TIMEOUT = 10000;
    
    private Socket socket;
    private Trie trie;
    
    public RequestHandler(Socket socket, Trie trie) {
        this.socket = socket;
        this.trie   = trie;
    }
    
    public void run() {
        String sResult = ERROR;
        BufferedReader br = null;
        PrintWriter    pw = null;
        try {
            br = new BufferedReader(new InputStreamReader(
                socket.getInputStream()));
            pw = new PrintWriter(socket.getOutputStream());
            
            StringTokenizer st = new StringTokenizer(br.readLine());
            
            if (!st.hasMoreElements()) {
                System.err.println("No command found");
                return;
            }
            
            String sCommand = st.nextToken().toUpperCase();
            
            try {
                if (sCommand.equals("SCAN")) {
                    if (!st.hasMoreElements()) {
                        System.err.println("Filename not found");
                        return;
                    }
                    new RecursiveFileFilter(new StreamFileFilter(trie))
                            .filter(new File(st.nextToken()));
                    sResult = OK;
                    
                } else if (sCommand.equals("POST")) {
                    ServerSocket dataInServerSocket = new ServerSocket(
                            0, 50,
                            InetAddress.getByName(ScannerDaemon.BINDNAME));
                    
                    dataInServerSocket.setSoTimeout(DATA_TIMEOUT);
                    pw.println("Send data to port '"
                               + dataInServerSocket.getLocalPort()
                               + "' within " + (DATA_TIMEOUT / 1000)
                               + " seconds.");
                    pw.flush();
                    Socket dataInSocket = null;
                    try {
                        dataInSocket  = dataInServerSocket.accept();
                        pw.println("Connected.");
                        pw.flush();
                        new SequentialStreamFilter(trie).filter(
                                dataInSocket.getInputStream(), null);
                        sResult = OK;
                        
                    } catch (InterruptedIOException iie) {
                        // Client failed to connect to data port
                        sResult = ERROR;
                    } finally {
                        if (dataInSocket != null) {
                            dataInSocket.close();
                        }
                        dataInServerSocket.close();
                    }
                    
                } else if (sCommand.equals("FILTER")) {
                    ServerSocket dataInServerSocket = new ServerSocket(
                            0, 50,
                            InetAddress.getByName(ScannerDaemon.BINDNAME));
                    ServerSocket dataOutServerSocket = new ServerSocket(
                            0, 50,
                            InetAddress.getByName(ScannerDaemon.BINDNAME));
                    
                    dataInServerSocket.setSoTimeout(DATA_TIMEOUT);
                    dataOutServerSocket.setSoTimeout(DATA_TIMEOUT);
                    pw.println("Send/receive data to/from port '"
                               + dataInServerSocket.getLocalPort() + "/"
                               + dataOutServerSocket.getLocalPort()
                               + "' within " + (DATA_TIMEOUT / 1000)
                               + " seconds.");
                    pw.flush();
                    Socket dataInSocket = null, dataOutSocket = null;
                    try {
                        dataInSocket  = dataInServerSocket.accept();
                        dataOutSocket = dataOutServerSocket.accept();
                        pw.println("Connected.");
                        pw.flush();
                        new SequentialStreamFilter(trie).filter(
                                dataInSocket.getInputStream(),
                                sCommand.equals("FILTER") ?
                                dataOutSocket.getOutputStream() : null);
                        sResult = OK;
                        
                    } catch (InterruptedIOException iie) {
                        // Client failed to connect to data port
                        sResult = ERROR;
                    } finally {
                        if (dataOutSocket != null) {
                            dataOutSocket.close();
                        }
                        if (dataInSocket != null) {
                            dataInSocket.close();
                        }
                        dataInServerSocket.close();
                        dataOutServerSocket.close();
                    }
                } else {
                    System.err.println("Unknown command '" + sCommand + "'.");
                    sResult = ERROR;
                }
                
            } catch (VirusFoundException vfe) {
                sResult = vfe.getMessage() + "\n";
            }
        } catch (Exception e) {
            e.printStackTrace();
        } finally {
            try {
                if (pw != null ) {
                    pw.write(sResult);
                    pw.close();
                }
                if (br != null) {
                    br.close();
                }
            } catch (IOException ioe) {
                ioe.printStackTrace();
            }
            try {
                socket.close();
            } catch (IOException ioe) {
                ioe.printStackTrace();
            }
        }
    }
}
