(* tstgraph.ml - test de graphique MT pour CAML
   Martial Tarizzo 18/8/95
   Quelques essais pour rire...
   *)

(* le chargement du noyau graphique.
   =================================
   Si CAML tourne sous DOS, valuez (adaptez le chemin !) :
   include "c:\\caml\\frontal\\mtgraph.ml";;
   
   Si CAMLWIN est le noyau de calcul actif :
   include "c:\\caml\\frontal\\wmtgraph.ml";;
   
   Si cela vous plat, compilez le fichier adquat, 
   placez les fichiers .ZO et .ZI dans le rpertoire caml\lib
   (voir le fichier MTGRAPH.TXT pour plus de dtails)
   
   Vous pouvez alors viter la ligne include ci-dessus en faisant :
   load_object "mtgraph";;
   #open "mtgraph";;
   
   ou 
   load_object "wmtgraph";;
   #open "wmtgraph";;
   
   Vous pouvez alors valuer tout ce fichier (F4)...
   ... et essayer les fonctions
   *)
      

(* 28/8/95 - Dmo ajoute : triangle des couleurs *)

(* 19/12/95 : Dmo ajoute : probleme des 4 tortues...*)

(* 13/10/96 : Pour viter des problmes gnants, les caractres
   accentus dans les dfinitions de MTGRAPH.ML et WMTGRAPH.ML ont t
   supprims. Modifi ces fonctions de test en consquence.
   *)

(* chargement de la librairie Windows *)
load_object "wmtgraph";;
(* et ouverture du module *)
#open "wmtgraph";;


(* Section 1 : graphique cartsien standard *)
(********************************************)
(*rayons et botes ... pour faire joli (?) *)
let rayons () =
  graph_init();
  for i= -10 to 10 do
    graph_color (25*i) (20*i) (15*i);
    graph_moveto 0. 0.;
    graph_lineto 2. ((float_of_int i) *. 0.2);
    graph_moveto 0. 0.;
    graph_lineto ((float_of_int i) *. 0.2) 2.;
    graph_moveto 0. 0.;
    graph_lineto (-2.) ((float_of_int i) *. 0.2);
    graph_moveto 0. 0.;
    graph_lineto ((float_of_int i) *. 0.2) (-2.)
  done;
  graph_flush();
  for i = -10 to 10 do
    graph_color (25*i) (20*i) (15*i);
    graph_circle ((float_of_int i) *. 0.1) 0. (abs_float(float_of_int i) *. 0.2)
  done;
  graph_flush();;
  
  (* exemple d'utilisation :
     rayons();;
  *)

let boites () =
  graph_init();
  for i=10 downto 1 do
    graph_color (25*i) (20*i) (15*i);
    graph_moveto ((float_of_int i) *. -0.2) ((float_of_int i) *. -0.2);
    graph_boxfullto ((float_of_int i) *. 0.2) ((float_of_int i) *. 0.2)
  done;
  graph_flush();;

  (* exemple d'utilisation :
     boites();;
  *)

(* trac de fonction. rudimentaire *)
let plot f (xmin, xmax, ymin, ymax) = 
  let x= ref 0.
  and delta_x = (xmax-.xmin)/.100.
  in
    graph_init();
    graph_scale xmin xmax ymin ymax;
    
    (* trac des axes *)
    if ((ymin <. 0.) & (ymax >. 0.)) then
      begin
        graph_moveto xmin 0.;
        graph_lineto xmax 0.
      end;
    if ((xmin <. 0.) & (xmax >. 0.)) then
      begin
        graph_moveto 0. ymin;
        graph_lineto 0. ymax
      end;
    
    (* trac de la fonction *)
    x := xmin;
    graph_moveto xmin (f !x);
    for i=1 to 100 do
      x := !x +. delta_x;
      graph_lineto !x (f !x)
    done;
    graph_flush();; 
    
  (* exemple d'utilisation :
     plot (fun x-> sin (2. *. x) /. x) 
          (-10., 10., -0.6, 2.);
     graph_moveto 2. 1.;
     graph_drawtext "\"(sin 2x)/x\"";
     graph_flush();;
  *)


(* une petite fractale, programme de faon assez imprative.
   Il serait assez facile de rendre cette fonction plus 'symbolique'
   en vitant le codage par des entiers des directions.
   Voir plus loin une version implante par le graphique tortue.
   *)
let creneau profondeur =
  (* les variables locales *)
  (*************************)
  let rec cx = ref 0.        (* les coordonnes du crayon *)
  and cy =ref 0.
  
  (* et les fonctions locales *)
  (****************************)
  and add4 n m =   (* addition modulo 4 : pour tourner...*)
    let res = (n+m) mod 4
    in if res>=0 then res else (res + 4)
  
  and trace_ligne longueur direction =        (* comme son nom l'indique *)
    match direction
      with 0 ->        (*nord*)
        cy := !cy +. longueur;
        graph_lineto !cx !cy
      |    1 ->        (*est*)
        cx := !cx +. longueur;
        graph_lineto !cx !cy
      |    2 ->        (*sud*)
        cy := !cy -. longueur;
        graph_lineto !cx !cy
      |    3 ->        (*ouest*)
        cx := !cx -. longueur;
        graph_lineto !cx !cy
      |    _ -> cx := 1. /. 0.        (*  erreur! *)
  
  and trace_cote n longueur direction =        (* idem!! *)
    if n=0 then trace_ligne longueur direction
    else
      let nouv_long = longueur /. 4.
      and nouv_n = n-1
      in
        trace_cote nouv_n nouv_long direction;
        trace_cote nouv_n nouv_long (add4 direction (-1));
        trace_cote nouv_n nouv_long direction;
        trace_cote nouv_n (nouv_long *. 2.) (add4 direction 1);
        trace_cote nouv_n nouv_long direction;
        trace_cote nouv_n nouv_long (add4 direction (-1));
        trace_cote nouv_n nouv_long direction
  
  (* le corps de la fonction *)
  (***************************)
  in
    graph_init();
    cx:= -1.; cy:= -1.;
    graph_moveto !cx !cy;
    (* les quatre cts du carr *)
    trace_cote profondeur 2. 0;        (*nord*)
    trace_cote profondeur 2. 1; (*est*)
    trace_cote profondeur 2. 2; (*sud*)
    trace_cote profondeur 2. 3; (*ouest*)
    graph_flush();;
    
    (* exemple d'utilisation :
    for i=0 to 3 do 
      print_string (string_of_int i);
      print_newline();
      creneau i 
    done;; 
    *)

(* pour tracer des polygones...*)
let draw_poly =
  let rec draw_lines = 
    fun [] -> ()
    | ((x, y) :: tail) ->
      graph_lineto x y;
      draw_lines tail
  in
    fun [] -> ()
    | ((x0, y0) :: tail) ->
        graph_moveto x0 y0;
        draw_lines tail;
        graph_lineto x0 y0;;
  
  (* exemple :
  graph_init();
  draw_poly 
    (map (fun x -> (cos(x/.deg_par_radian), sin(x/.deg_par_radian)))
         [10.; 45.; 90.; 175.; 230.; 320.]);
  graph_flush();;
  *)

(* ... et les tracer remplis.
  NE MARCHE QU'AVEC LA VERSION DOS : pas de remplissage de surface avec CAMLWIN
  algorithme un peu faible : remplissage  partir du centre d'inertie
  dont rien n'assure videmment qu'il se trouve  l'intrieur !
  *)
let draw_fillpoly l =
  let n = float_of_int (list_length l)
  in
    match (list_it 
            (fun (a, b) (c, d) -> 
              (a+.c, b+.d))
            l 
            (0., 0.))
    with (x, y) -> 
      draw_poly l;
      graph_floodfill (x/.n) (y/.n);;

  (* exemple :
  graph_init();
  graph_clear();
  graph_scale (-500.) 500. (-500.) 500.;
  draw_fillpoly 
    (map (fun x -> (x*.cos(x/.deg_par_radian), x*.sin(x/.deg_par_radian)))
         (let rec make_list l n incr = 
            if n=0 then l
            else make_list (hd l +. incr :: l)  (n-1) incr
          in
            make_list [0.] 36 10.));;
  graph_flush();;
  *)

(* trac du triangle des couleurs
   NE MARCHE QU'AVEC LA VERSION DOS : pas de remplissage de surface avec CAMLWIN
   un triangle est decoup de faon rcursive en petits triangles, chacun 
   tant color en fonction de sa position.
   Positionnement des couleurs :
     B
    ...
   R...V
   La couleur d'un triangle est dtermine par la position de son 
   isobarycentre G. Ce point est lui-mme le barycentre des sommets
   du triangle total, avec des coeffs reprsentant le dosage des
   couleurs (R V B)
   
   Sous CAMLWIN, cette fonction ne fonctionne pas compltement (pas de 
   remplissage des surfaces colores)
   
   *)
let triangle_des_couleurs n = 
  let (xR, yR) = (-255., 0.)
  and (xV, yV) = (255., 0.)
  and (xB, yB) = (0., 255.)
  in
    let rec barycentre (xa, ya) (xb, yb) (xc, yc) = 
      ((xa +. xb +. xc) /. 3., (ya +. yb +. yc) /. 3.)
    and calc_rgb (xg, yg) =
      (int_of_float((xV -. xg -. yg) /. 2.),
       int_of_float((xV +. xg -. yg) /. 2.),
       int_of_float(yg))
    and milieu (xa, ya) (xb, yb) = 
      ((xa +. xb) /. 2., (ya +. yb) /. 2.)
    and dessine_triangle (xa, ya) (xb, yb) (xc, yc) =
      let (xg, yg) = barycentre (xa, ya) (xb, yb) (xc, yc) in
      let (r, g, b) = calc_rgb (xg, yg) in
        graph_color r g b;
        graph_moveto xa ya;
        graph_lineto xb yb;
        graph_lineto xc yc;
        graph_lineto xa ya;
        graph_floodfill xg yg
    and tri_h (xa, ya) (xb, yb) (xc, yc) niveau = 
      if niveau = 0 then
        dessine_triangle (xa, ya) (xb, yb) (xc, yc)
      else
        let (xab, yab) = milieu (xa, ya) (xb, yb)
        and (xbc, ybc) = milieu (xb, yb) (xc, yc) 
        and (xac, yac) = milieu (xa, ya) (xc, yc)
        in
          tri_h (xa, ya) (xab, yab) (xac, yac) (niveau - 1);
          tri_h (xab, yab) (xb, yb) (xbc, ybc) (niveau - 1);
          tri_h (xac, yac) (xbc, ybc) (xc, yc) (niveau - 1);
          tri_b (xac, yac) (xab, yab) (xbc, ybc) (niveau - 1)
    and tri_b (xa, ya) (xb, yb) (xc, yc) niveau = 
      if niveau = 0 then
        dessine_triangle (xa, ya) (xb, yb) (xc, yc)
      else
        let (xab, yab) = milieu (xa, ya) (xb, yb)
        and (xbc, ybc) = milieu (xb, yb) (xc, yc)
        and (xac, yac) = milieu (xa, ya) (xc, yc)
        in
          tri_b (xa, ya) (xab, yab) (xac, yac) (niveau - 1);
          tri_b (xab, yab) (xb, yb) (xbc, ybc) (niveau - 1);
          tri_b (xac, yac) (xbc, ybc) (xc, yc) (niveau - 1);
          tri_h (xab, yab) (xbc, ybc) (xac, yac) (niveau - 1)
    in
      graph_init();
      graph_scale (-260.) 260. (-10.) 260.;
      tri_h (xR, yR) (xV, yV) (xB, yB) n;
      graph_flush();;

  (* Exemple : (joli si plus de 256 couleurs...)
     triangle_des_couleurs 4;;
  *)


(* Section 2 : le monde des tortues ... *)
(****************************************)

(* Exemple 1 : un trac de rosace forme de polygones rguliers 
   * n est le nombre de cts,
   * m le nombre de tours fait par la tortue sur elle-mme au cours d'un 
       trac
   * demi_cote mesure le demi ct du carr du plan centr sur l'origine
     qui sera visible dans la fentre graphique
   *)
let rosace n m demi_cote =
  let t1 = cree_tortue ()
  and deuxpi = 2. *. 3.14159265359
  in
    graph_init();
    graph_scale (-. demi_cote) demi_cote (-. demi_cote) demi_cote;
    for j=1 to n do
      for i=1 to n do
        av t1 1.;
        tg t1 (float_of_int m *. deuxpi /. float_of_int n)
      done;
      graph_flush();
      tg t1 (float_of_int m *. deuxpi /. float_of_int n)
    done;
    graph_flush();;
    
    (* exemple d'utilisation :
    rosace 9 4 1.5;;
    *)


(* Exemple 2 : reprsentation graphique d'un arbre ... par une tortue!
   n est la profondeur,
   l la longueur du tronc
   angle est l'angle entre deux branches
   *)
let arbre n l angle = 
  let t1 = cree_tortue()
  and a = angle /. deg_par_radian
  and a2 = angle /. deg_par_radian /. 2.
  in
    let rec branche n l =
      av t1 l;
      if n <> 0 then
        begin
          tg t1 a2;
          branche (n-1) (l /. 1.8);
          td t1 a;
          branche (n-1) (l /. 1.8);
          tg t1 a2
        end;
      re t1 l
    in
      graph_init();
      graph_scale (-. l) l (-. l) (1.5 *. l);
      lc t1;
      va t1 0. (-. l);
      bc t1;
      fc t1 (90. /. deg_par_radian);
      branche n l;
      graph_flush();;
      
      (* exemple d'utilisation :
      arbre 6 100. 90.;;
      *)

(* Exemple 3 : flocon de Von Koch 
   n est la profondeur
   l la longueur d'un ct du triangle
   *)
let flocon n l =
  let t1 = cree_tortue()
  and a = 60. /. deg_par_radian
  in
    let rec cote n l =
      if n=0 then
        av t1 l
      else
        begin
          cote (n-1) (l /. 3.); tg t1 a;
          cote (n-1) (l /. 3.); td t1 (2. *. a);
          cote (n-1) (l /. 3.); tg t1 a;
          cote (n-1) (l /. 3.)
        end
    in
      graph_init();
      graph_scale (-. l) l (-. l /. 3.) (l +. l /. 3.);
      lc t1; va t1 0. l; bc t1;
      td t1 a; cote n l;
      td t1 (2. *. a); cote n l;
      td t1 (2. *. a); cote n l;
      graph_flush();;
      
      (* exemple d'utilisation :
      flocon 4 5.;
      graph_floodfill 0. 0.;
      graph_flush();;
      *)

(* Exemple 4 : le crneau par une tortue.
   Bien plus simple qu'auparavant... et pratiquement aussi efficace!! *)
let creneau_tortue profondeur =
  let t1 = cree_tortue()
  and a_droit = 90. /. deg_par_radian
  in
    let rec trace_cote n longueur =
      if n=0 then av t1 longueur
      else
        let nouv_long = longueur /. 4.
        and nouv_n = n-1
        in
          trace_cote nouv_n nouv_long; tg t1 a_droit;
          trace_cote nouv_n nouv_long; td t1 a_droit;
          trace_cote nouv_n nouv_long; td t1 a_droit;
          trace_cote nouv_n (nouv_long *. 2.); tg t1 a_droit;
          trace_cote nouv_n nouv_long; tg t1 a_droit;
          trace_cote nouv_n nouv_long; td t1 a_droit;
          trace_cote nouv_n nouv_long
    in
      graph_init();
      lc t1; va t1 (-1.) (-1.); bc t1;
      fc t1 a_droit;
      (* les quatre cts du carr *)
      for i=1 to 4 do
        trace_cote profondeur 2.; td t1 a_droit
      done;
      graph_flush();;
      
      (* exemple d'utilisation :
      creneau_tortue 3;;
      *)

(* un autre plus rgulier *)
let reg_creneau_tortue profondeur =
  let t1 = cree_tortue()
  and a_droit = 90. /. deg_par_radian
  in
    let rec trace_cote n longueur =
      if n=0 then av t1 longueur
      else
        let nouv_long = longueur /. 5.
        and nouv_n = n-1
        in
          trace_cote nouv_n nouv_long; td t1 a_droit;
          trace_cote nouv_n nouv_long; tg t1 a_droit;
          trace_cote nouv_n nouv_long; tg t1 a_droit;
          trace_cote nouv_n nouv_long; td t1 a_droit;
          trace_cote nouv_n nouv_long; td t1 a_droit;
          trace_cote nouv_n nouv_long; tg t1 a_droit;
          trace_cote nouv_n nouv_long; tg t1 a_droit;
          trace_cote nouv_n nouv_long; td t1 a_droit;
          trace_cote nouv_n nouv_long
    in
      graph_init();
      graph_scale (-1.1) 1.1 (-1.1) 1.1;
      lc t1; va t1 (-1.) (-1.); bc t1;
      fc t1 a_droit;
      (* les quatre cts du carr *)
      for i=1 to 4 do
        trace_cote profondeur 2.; td t1 a_droit
      done;
      graph_flush();;
  
  (* Exemple d'utilisation :
     reg_creneau_tortue 2;
     graph_moveto (-0.5) 0.;
     graph_drawtext "Crneau rgulier \" la Tortue\"";
     graph_flush();;
  *)


     
(* Exemple 5 : les 4 tortues ... 
  4 tortues, animes d'intentions plus ou moins avouables, se dirigent
  l'une vers l'autre dans le sens trigo,  vitesse constante.
  But de la fonction : tracer leur trajectoires,  partir de la dfinition
  du problme.
  Cela peut tre la base d'une foule de jolies simulations en physique !
  *)
let pb_4_tortues () =
  let liste_tortues = 
    (* la liste des 4 tortues *)
    [cree_tortue() ; cree_tortue() ; cree_tortue() ; cree_tortue()]
  and Pisur2 = 
    (* C'est lui : l'angle droit ! *)
    2. *. atan(1.)
  (* maintenant, les fonctions de travail *)
  in let rec sqr x =
    (* facile : le carr *) 
    x *. x
  and calc_cap x1 y1 x2 y2 =
    (* l'angle directeur d'un segment M1M2 *) 
    if x1<.x2 then
      atan((y2-.y1)/.(x2-.x1))
    else if x1=.x2 then
      if y1 <. y2 then Pisur2 else -. Pisur2
    else
      2.*.Pisur2 +. atan((y2-.y1)/.(x2-.x1))
  and dpart l = 
    (* le placement initial des tortues *)
    graph_init(); 
    do_list2
      (fun t (x, y, c) -> lc t; va t x y; bc t; fc t (c /. deg_par_radian))
      l
      [(1. , 1., 180.);(-1. , 1. , -90.);(-1. , -1. , 0.);(1. , -1., 90.)]
  and boucle l =
    (* la boucle de dplacement des tortues *)
    let r1= retourne_etat (hd l)
    and r2= retourne_etat (hd (tl l))
    in
      let long = sqrt (sqr(r1.x-.r2.x) +. sqr(r1.y-.r2.y)) /. 20.
      and c = ref 0.
      in
        if long <. 1e-3 then ()
        else
          begin
          c := calc_cap r1.x r1.y r2.x r2.y;
          do_list 
            (fun t -> av t long; fc t !c; c := !c +. Pisur2)
            l;
          boucle l
          end
  in
    (* programme principal *)
    dpart liste_tortues;
    boucle liste_tortues;
    graph_flush();;


  (* Exemple d'utilisation :
     pb_4_tortues ();;
  *)

(*EOF*)
