(* Machine-independent graphic primitives *)



exception Graphic_failure of string;;
        (* Raised by the functions below when they encounter an error. *)


(*** System Environnement *)
type operating_system_type = Unix
                           | Dos
                           | MsWindows
                           | OS2
                           | WindowsNT
                           | MacOs
                           | NextStep
;;
type screen_type = MonochromeScreen
                 | ColorScreen;;

value host_system : unit -> operating_system_type = 1 "gr_host_system"
  and screen_color : unit -> screen_type = 1 "gr_screen_type"
;;




(*** Initializations *)

value open_graph: unit -> unit = 1 "gr_open_graph"
        (* Switch the screen to graphic mode.
           The string argument can be used to specify a graphic mode.
           If the empty string is given, a sensible default mode is
           selected. *)
  and close_graph: unit -> unit = 1 "gr_close_graph"
        (* Switch the screen back to text mode. *)
  and clear_graph : unit -> unit = 1 "gr_clear_graph"
        (* Erase the graphic window. *)
  and screen_width  : unit -> int = 1 "gr_screen_width"
  and screen_height : unit -> int = 1 "gr_screen_height"
        (* Return the size of the graphic screen. Coordinates of the screen
           pixels range over [0 .. size_x() - 1] and [0 .. size_y()-1].
           Drawings outside of this rectangle are clipped, without causing
           an error. The origin (0,0) is at the lower left corner.
           The coordinate system used in the same as in mathematics:
           [y=0] is the bottom line of the window;
           [y=size_y()-1] is the top line;
           angles are measured counterclockwise. *)
  and set_clip_area : int -> int -> int -> int -> unit = 4 "gr_set_clip_area"
  and gr_flush : unit -> unit = 1 "gr_flush"
;;




(*** Windows *)

type window;;
value default_window : unit -> window = 1 "gr_default_window"
;;

value add_window: int -> int -> int -> int -> string -> window = 5
      "gr_add_window"
  and del_window: window -> unit = 1 "gr_del_window"
  and window_pos: window -> int * int = 1 "gr_window_pos"
  and window_width: window -> int = 1 "gr_window_width"
  and window_height: window -> int = 1 "gr_window_height"
  and set_draw_window: window -> unit = 1 "gr_set_draw_window"
  and set_current_window: window -> unit = 1 "gr_set_current_window"
  and get_current_window: unit -> window = 1 "gr_get_current_window"
  and set_window_car : int -> int -> int -> int -> unit = 4 "gr_set_window_car";;




(*** Event *)

type event_type =
    Button_down			(* A mouse button is pressed *)
  | Button_up			(* A mouse button is released *)
  | Key_pressed			(* A key is pressed *)
  | Mouse_motion		(* The mouse is moved *)
  | Poll			(* Don't wait; return immediately *)
  | Redraw
  | Close		        (* The window  is closed *)
  | Size		        (* The window is resize *)
  | Time		        (* Time event *)
;;

type event =
  { win     : window;
    id_event: event_type;
    mouse_x : int;		(* X coordinate of the mouse *)
    mouse_y : int;		(* Y coordinate of the mouse *)
    button  : bool;		(* true if a mouse button is pressed *)
    key     : int                (* the character for the key pressed *)
  }
;;


value F1_key       : int
  and F2_key       : int
  and F3_key       : int
  and F4_key       : int
  and F5_key       : int
  and F6_key       : int
  and F7_key       : int
  and F8_key       : int
  and F9_key       : int
  and F10_key      : int

  and Right_key    : int
  and Left_key     : int
  and Up_key       : int
  and Down_key     : int

  and PageUp_key   : int
  and PageDown_key : int
  and Home_key     : int
  and End_key      : int
  and Insert_key   : int
  and Del_key      : int

  and Copy_key     : int
  and Cut_key      : int
  and Past_key     : int
;;


value get_event: event_type list -> event = 1 "gr_get_event"
  and set_time_event : int -> unit = 1 "gr_set_time_event";;





(*** Colors *)

type color == int
        (* A color is specified by its R, G, B components. Each component
           is in the range [0..255]. The three components are packed in an int:
           0xRRGGBB, where RR are the two hexadecimal digits for the red
           component, and so on. *)
;;

value set_color : color -> unit = 1 "gr_set_color"
        (* Set the current drawing color *)
;;

value rgb: int -> int -> int -> int;;
        (* [rgb r g b] returns the integer encoding the color with red
           component [r], green component [g], and blue component [b].
           [r], [g] and [b] are in the range [0..255]. *)

value black : color
  and white : color
  and red : color
  and green : color
  and blue : color
  and yellow : color
  and cyan : color
  and magenta : color
;;
        (* Some predefined colors *)

value background: color
        (* The color of the screen background (black on the PC ports).
           [clear_graph] paints the screen with this color. *)
  and foreground: color
        (* The initial drawing color (white on the PC ports). *)
;;





(*** Point and line drawing *)

value plot : int -> int -> unit = 2 "gr_plot"
        (* Plot the given point with the current drawing color. *)
  and point_color : int -> int -> color = 2 "gr_point_color"
        (* Return the color of the given point. *)
  and moveto : int -> int -> unit = 2 "gr_moveto"
        (* Position the current point. *)
  and current_point : unit -> int * int = 1 "gr_current_point"
        (* Return the position of the current point. *)
  and lineto : int -> int -> unit = 2 "gr_lineto"
        (* Draw a line with endpoints the current point and the given point,
           and move the current point to the given point. *)
  and draw_arc : int -> int -> int -> int -> int -> int -> unit
  	       = 6 "gr_draw_arc"
        (* [draw_arc x y rx ry a1 a2] draws an elliptical arc with center
           [x,y], horizontal radius [rx], vertical radius [ry], from angle
           [a1] to angle [a2] (in degrees). *)
  and draw_ellipse : int -> int -> int -> int -> unit
        (* [draw_ellipse x y rx ry] draws an ellipse with center
           [x,y], horizontal radius [rx] and vertical radius [ry]. *)
  and draw_circle : int -> int -> int -> unit
        (* [draw_circle x y r] draws a circle with center [x,y] and
           radius [r]. *)
;;






(*** Text drawing *)
type Font_Type = Default_font
               | Fixed_font
               | Proportional_font
;;

type Font_Attributes = Normal_font
                     | Italic
                     | Bold
                     | Underline
;;

type Font_Size = Size1_font
               | Size2_font
               | Size3_font
               | Size4_font
               | Size5_font
               | Size6_font
;;




value draw_char : char -> unit = 1 "gr_draw_char"
  and draw_string : string -> unit = 1 "gr_draw_string"
        (* Draw a character or a character string with lower left corner
           at current position. After drawing, the current position is set
           to the lower right corner of the text drawn. *)
  and set_font : Font_Type -> Font_Attributes -> Font_Size -> unit = 3
                 "gr_set_font"
  and text_size : string -> int * int = 1 "gr_text_size"
        (* Return the dimensions of the given text, if it were drawn with
           the current font and size. *)
;;







(*** Filling *)

value fill_rect : int -> int -> int -> int -> unit = 4 "gr_fill_rect"
        (* [fill_rect x y w h] fills the rectangle with lower left corner
           at [x,y], width [w] and heigth [h], with the current color. *)
  and fill_poly : (int * int) vect -> unit = 1 "gr_fill_poly"
        (* Fill the given polygon with the current color. The vector
           contains the coordinates of the vertices of the polygon. *)
  and fill_arc : int -> int -> int -> int -> int -> int -> unit
  	       = 6 "gr_fill_arc"
        (* Fill an elliptical pieslice with the current color. The
           parameters are the same as for [draw_arc]. *)
  and fill_ellipse : int -> int -> int -> int -> unit
        (* Fill an ellipse with the current color. The
           parameters are the same as for [draw_ellipse]. *)
  and fill_circle : int -> int -> int -> unit
        (* Fill a circle with the current color. The
           parameters are the same as for [draw_circle]. *)
;;




(*** Images *)

type image;;
        (* The abstract type for images, in internal representation.
           Externally, images are represented as matrices of colors. *)

value transp : color;;
        (* In matrices of colors, this color represent a ` `transparent' '
           point: when drawing the corresponding image, all pixels on the
           screen corresponding to a transparent pixel in the image will
           not be modified, while other points will be set to the color
           of the corresponding point in the image. This allows superimposing
           an image over an existing background. *)

value make_image : color vect vect -> image = 1 "gr_make_image"
        (* Convert the given color matrix to an image.
           Each sub-vector represent one horizontal line. All sub-vectors
           must have the same length; otherwise, exception [Graphic_failure]
           is raised. *)
  and dump_image : image -> color vect vect = 1 "gr_dump_image"
        (* Convert an image to a color matrix. *)
  and draw_image : image -> int -> int -> unit = 3 "gr_draw_image"
        (* Draw the given image with lower left corner at the given point. *)
  and get_image : int -> int -> int -> int -> image
        (* Capture the contents of a rectangle on the screen as an image.
           The parameters are the same as for [fill_rectangle]. *)
  and create_image : int -> int -> image = 2 "gr_create_image"
  	(* [create_image w h] allocates a new image with width [w] and 
	   height [h], for use with [copy_image]. *)
  and blit_image : image -> int -> int -> unit = 3 "gr_blit_image"
  	(* [blit_image i x y] copies the contents of a rectangle on
	   the screen into image [i]. The rectangle has lower left
	   corner [(x,y)], and width and height those of image [i]. *)
  and image_size : image -> int * int = 1 "gr_image_size"
  and plot_image : image -> int -> int -> color -> unit = 4 "gr_plot_image"
  and pixel_image : image -> int -> int -> int = 3 "gr_pixel_image"
;;





(*** Sound *)

value sound : int -> int -> unit = 2 "gr_sound"
        (* [sound freq dur] plays a sound at frequency [freq] (in hertz)
           for a duration [dur] (in milliseconds). *)
;;




(*** Directory *)

value read_dir : unit -> string = 1 "gr_read_dir"
  and current_dir : unit -> string = 1 "gr_current_dir"
  and change_dir : string -> unit = 1 "gr_chdir"
;;



(*** Exec *)

value system : string -> int =1 "gr_system"
;;



(*** Copy, Past *)

value gr_copy_text : string -> unit =1 "gr_copy"
  and gr_past_text : unit -> string =1 "gr_past"
;;



(*** Time *)

type time =
  {
    second : int;
    minute : int;
    hour   : int;
    day    : int;
    month  : int;
    year   : int;
    s_1_1_1970 : int
  }
;;


value get_time : unit -> time =1 "mltime"
;;


(*** Time of last file modification *)

value file_modif : string -> time =1 "file_modif_time"
;;