function [A,R] = testmatr(type,n,opts)
%TESTMATR test matrices for PQser.
%   A = testmatr(type,n) returns a similarity test matrix of type "type" and
%   size n.
%      type 1: pre-R matrix with one connected component (irreducible matrix).
%      type 2: reducible matrix with known connected components, in this case 
%              n is a vector containing the sizes of the connected components.
%   A = testmatr(...,opts) optionally passes a set of options.
%   [A,R]=testmatr(...) returns also the ordered matrix R.
%
%   options (only for type 2 matrices):
%   opts.bw	the matrix has bandwidth 2*bw+1 (def. bw=2).
%   opts.spar	if set to 1 (default), the matrix is sparse.
%
%   See also spectrsort, pspectrsort.

%   A. Concas, C. Fenu & G. Rodriguez, University of Cagliari, Italy
%   Email: concas.anna@gmail.com, kate.fenu@gmail.com, rodriguez@unica.it
%
%   Last revised Aug 20, 2017

if nargin<3 || isempty(opts), opts = struct('empty','empty'); end

nam = fieldnames(opts);
% bandwidth of the connected components (for redmatr)
if ~any(strcmpi('bw',nam)), opts.bw = 2; end
% sparsity (for redmatr)
if ~any(strcmpi('spar',nam)), opts.spar = 1; end

switch type
	
	case 1
		Ai = tril(triu(ones(n),-2),2);	% incidence matrix
		di = Ai*ones(n,1);
		Ab = diag(100./di)*Ai;	% abundance matrix, 1 found object
		                        % for each type
		R = msim2(Ab);		% R-matrix, similarity matrix (using
		                        % the Robinson criterion)
		u = randperm(n);
		A = R(u,u);      	% pre-R matrix
		
	case 2
		R = redmatr(n,opts);
		n = size(R,1);
		u = randperm(n);
		A = R(u,u);
		
	otherwise
		error('unknown type');
end


function S = msim2(A)
%MSIM2 similarity matrix using the Robinson criterion

n = size(A,1);
tau = 100*eps;

S = zeros(n);
e = ones(n,1);
for i = 1:n-1
	M = megm(n,i);
	v = 200 - abs(M*A)*e;
	v(v<tau) = 0;
	S(i:n,i) = v(i:n);
	S(i,i+1:n) = v(i+1:n)';
end
S(n,n) = 200;


function M = megm(n,i)
%MEGM	modified elementary Gauss matrix computation

M = diag([zeros(i,1);ones(n-i,1)]);
M(i+1:n,i) = -1;


function A = redmatr(n_c,opts)
%REDMATR reducible test matrix with length(n_c) connected components.
%   n_c vector with dimensions of each component

k = length(n_c);
A = [];

if opts.spar
	for i = 1:k
		A1 = spdiags(ones(n_c(i),2*opts.bw+1), ...
			[-opts.bw:opts.bw],n_c(i),n_c(i));
		A = blkdiag(A,A1);
	end
else
	for i = 1:k
		A1 = tril(triu(ones(n_c(i),n_c(i)),-opts.bw),opts.bw);
		A = blkdiag(A,A1);
	end
end

