function T = spectrsort(A,opts,U)
%SPECTRSORT spectral sort for the seriation problem
%   T = spectrsort(A) constructs a PQ-tree T containing all the set of all
%   permutations that reorder the rows and columns of the
%   similarity/abundancy matrix A.
%   T = spectrsort(A,opts,U) optionally passes a set of options and a
%   vector containing a subset of rows/columns to sort.
%
%   options:
%   opts.tau	tolerance for detecting the multiplicity of the Fiedler
%               value (def. 1e-8).
%   opts.translate translate the matrix substracting the smallest correlation 
%
%   See also pspectrsort, pqtreeplot.

%   A. Concas, C. Fenu & G. Rodriguez, University of Cagliari, Italy
%   Email: concas.anna@gmail.com, kate.fenu@gmail.com, rodriguez@unica.it
%
%   Last revised Aug 20, 2017

n = size(A,1);	% units number

if nargin < 3, U = [1:n]; end
if nargin < 2 || isempty(opts), opts = struct('empty','empty'); end

nam = fieldnames(opts);
% tolerance for the distinct function
if ~any(strcmpi('tau',nam)), opts.tau = 1e-8; end
% matrix translation 
if ~any(strcmpi('translate',nam)), opts.translate = 1; end

if opts.translate
	% compute the value of the smallest correlation
	if n > 1
		alpha = min(min(A));
	else
		alpha = A;
	end

	% translate the matrix substracting the value of the smallest
	% correlation from every correlation 
	if isempty(ver('matlab'))
		if alpha, A = A - alpha*ones(n); end
	else
		if alpha, A = A - alpha; end
	end
end

Ucell = getconcomp(A,opts); % irreducible blocks (connected components)
k = size(Ucell,2);
if k > 1 % separate the irreducible blocks into the children of a P-node
	Tu = struct('type','E','value',cell(1,k)); % preallocation
	for j = 1:k
		Tu(j) = spectrsort(A(Ucell{j},Ucell{j}),opts,U(Ucell{j}));
	end
	T = pnode(Tu);
else
	if n == 1
		T = lnode(U); % leaf-node
	elseif  n == 2
		T = pnode(U); % P-node
	else
		L = lapl(A); % compute the Laplacian matrix
		[V,~,ii,nfied] = fiedvecs(L,opts);
		if nfied == 1 % case of a simple Fiedler value
			x = V(:,ii(2)); % computation of the Fiedler vector
			[y,~,ijj] = distinct(x,opts.tau);
			u = unique(y);
			y = y(ijj);	% reorder as x
			t = length(u);	% number of distinct values in x
			Tv = struct('type','E','value',cell(1,t)); % prealloc
			for j = 1:t
				v = find(y == u(j));
				vl = length(v);
				if vl == 1
					Tv(j) = lnode(U(v));
				else
					Tv(j) = spectrsort(A(v,v),opts,U(v));
				end
			end
			T = qnode(Tv); % Q-node
		else
			warning('Multiple Fiedler value: the input matrix is not pre-R.')
			T = mnode(U); % M-node
		end
	end
end

