function pqtreeplot(T,opts)
%PQTREEPLOT plot a PQ tree.
%   pqtreeplot(T) plot in the current figure the PQ tree whose root is T.
%   If the user clicks on one node with the left mouse button, the
%   corresponding subtree is extracted, it is plotted in a new figure, and
%   it is saved to a variable in the workspace.
%   pqtreeplot(T,opts) optionally passes a set of options.
%
%   options:
%   opts.labelson if set to 1 (default) the values of the leaves are
%                displayed in the plot.
%   opts.fontsize sets the size of the font (default 10)
%   opts.markersize sets the size of the markers (default 8)
%
%   See also spectrsort, pspectrsort, pqtree1perm, pqtreeNperm, pqtreeperms.

%   A. Concas, C. Fenu & G. Rodriguez, University of Cagliari, Italy
%   Email: concas.anna@gmail.com, kate.fenu@gmail.com, rodriguez@unica.it
%
%   Last revised Aug 20, 2017

if T.type == 'L'
	error('Cannot draw a single leaf');
end

if nargin < 2 || isempty(opts), opts = struct('empty','empty'); end

nam = fieldnames(opts);
% display labels in figure
if ~any(strcmpi('labelson',nam)), opts.labelson = 1; end
% font size for figure
if ~any(strcmpi('fontsize',nam)), opts.fontsize = 10; end
% marker size for figure
if ~any(strcmpi('markersize',nam)), opts.markersize = 8; end

[fs,ls] = pqtreenodes(T);
p = find(ls == -1); % P nodes indices
q = find(ls == -2); % Q nodes indices
m = find(ls == -3); % Q nodes indices
l = find(ls > 0); % leaf nodes indices

% compute x and y, coordinates in the unit square, h is the height of the tree
[x,y,h] = treelayout(fs);
x1 = x(p);
y1 = y(p);
x2 = x(q);
y2 = y(q);
x3 = x(m);
y3 = y(m);
x4 = x(l);
y4 = y(l);

x1 = [x1 nan];
y1 = [y1 nan];
x2 = [x2 nan];
y2 = [y2 nan];
x3 = [x3 nan];
y3 = [y3 nan];
x4 = [x4 nan];
y4 = [y4 nan];

f = find(fs~=0);
ff = fs(f);
X = [x(f); x(ff); NaN(size(f))];
Y = [y(f); y(ff); NaN(size(f))];

X = X(:);
Y = Y(:);

n = length(fs);
if n < 10000
	v = plot(X,Y,'r-',x1,y1,'ko',x2,y2,'bs',x3,y3,'cd',x4,y4,'m^');
	set(gca,'fontsize',opts.fontsize)
	set(v(2),'MarkerFaceColor','k')
	set(v(3),'MarkerFaceColor','b')
	set(v(4),'MarkerFaceColor','c')
	set(v(5),'MarkerFaceColor','m')
	if isempty(p) && isempty(m)
		legend(v([3,5]),'Q node','leaf');
	elseif isempty(p) && isempty(q)
		legend(v(4:5),'M node','leaf');
	elseif isempty(q) && isempty(m)
		legend(v([2,5]),'P node','leaf');
	elseif isempty(p)
		legend(v(3:5),'Q node','M node','leaf');
	elseif isempty(q)
		legend(v([2,4,5]),'P node','M node','leaf');
	elseif isempty(m)
		legend(v([2,3,5]),'P node','Q node','leaf');
	else
		legend(v(2:5),'P node','Q node','M node','leaf');
	end
else
	v = plot(X,Y,'r-');
	set(gca,'fontsize',opts.fontsize)
end
	
if opts.labelson
	xoff = 0; yoff = 0.025;
	for i = 1:length(l)
		text(x4(i)-xoff,y4(i)-yoff,num2str(ls(l(i))), ...
			'fontsize',opts.fontsize)
	end
end

for i = 1:length(v)
	set(v(i),'MarkerSize',opts.markersize)
	set(v(i),'ButtonDownFcn',@(src,pnt)pqtreegetnode(src,pnt,fs,ls,x,y,T,opts));
end

xlabel(['height = ' int2str(h)]);

axis tight
axis off
zoom xon
zoom off

