function perm = pqtreeperms(T,pmax)
%PQTREEPERMS extract all admissible permutations from a PQ tree.
%   perm = pqtreeperms(T,perm) extracts all the admissible permutations
%   contained in the PQ tree whose root is T. 
%   To avoid a long execution time and memory consumption, the program runs
%   only if the number of permutations does not exceed pmax (def. 100).
%
%   See also pqtree1perm, pqtreeNperm, pqtreeplot.

%   A. Concas, C. Fenu & G. Rodriguez, University of Cagliari, Italy
%   Email: concas.anna@gmail.com, kate.fenu@gmail.com, rodriguez@unica.it
%
%   Last revised Aug 20, 2017

if nargin < 2, pmax = 120; end

nperm = pqtreeNperm(T);
if nperm > pmax
	error('%d permutations expected, current limit is %d. Increase pmax.', nperm, pmax);
end

switch T.type
	case 'L'
		perm = T.value;
	case {'Q'}
		w = T.value;
		R = pqtreeperms(w(1),pmax);
		for i = 2:length(w)
			S = pqtreeperms(w(i),pmax);
			R = [kron(R,ones(size(S,1),1)) kron(ones(size(R,1),1),S)];
		end
		perm = [R;fliplr(R)];
	case {'P','M'}
		w = T.value;
		nw = length(w);
		W = cell(1,nw); % preallocation
		for i = 1:nw
			W{i} = pqtreeperms(w(i),pmax);
		end
		PS = perms(1:nw);
		for k = 1:size(PS,1)
			p = PS(k,:);
			R = W{p(1)};
			for i = 2:nw
				S = W{p(i)};
				R = [kron(R,ones(size(S,1),1)) kron(ones(size(R,1),1),S)];
			end
			iR = size(R,1);
			if k == 1
				perm = zeros(nperm,size(R,2));
				perm(1:iR,:) = R;
				iperm = iR + 1;
			else
				perm(iperm:iperm+iR-1,:) = R;
				iperm = iperm+iR;
			end
		end
end

