function [V,d,ii,nfied] = fiedvecs(L,opts)
%FIEDVECS compute the Fiedler vectors and values of a Laplacian.
%   [V,d,ivec,nfied] = fiedvecs(L) returns in V and d (a subset of) the
%   eigenvectors and eigenvalue of L. Close eigenvalues are made equal. 
%   The vector ivec gives the order of the sorted eigenvalues in V and d.
%   The scalar nfied is the multiplicity of the Fiedler value.
%   [...] = fiedvecs(L,opts) optionally passes a set of options.
%
%   options:
%   opts.tau	tolerance for detecting the multiplicity of the Fiedler
%               value (def. 1e-8).
%   opts.lrg	large matrix: if set to 1 uses an eigs instead than eig;
%               by default, it is set to 1 if L is sparse.
%   opts.nlarge	L can be considered large only if n>=nlarge (def. 1000).
%   opts.neig	number of eigenvalues computed with eigs (def. 3).
%
%   See also spectrsort, pspectrsort.

%   A. Concas, C. Fenu & G. Rodriguez, University of Cagliari, Italy
%   Email: concas.anna@gmail.com, kate.fenu@gmail.com, rodriguez@unica.it
%
%   Last revised Aug 20, 2017

if (nargin<2) || isempty(opts), opts = struct('empty','empty'); end

%	check options
nam = fieldnames(opts);
% tolerance for the distinct function
if ~any(strcmpi('tau',nam)), opts.tau = 1e-8; end
% sparsity of the Laplacian matrix
if ~any(strcmpi('lrg',nam)), opts.lrg = issparse(L); end
% dimension for a "large" matrix
if ~any(strcmpi('nlarge',nam)), opts.nlarge = 1000; end
% number of eigenvalues
if ~any(strcmpi('neig',nam)), opts.neig = 3; end
% additional eigenvalues
%if ~any(strcmpi('incr',nam)), opts.incr = 5; end

n = size(L,1);
opts.neig = min(opts.neig,n);
opts.lrg = opts.lrg && (n>=opts.nlarge);

if opts.lrg
	[V,D] = eigs(L,opts.neig,'sa');
	d = diag(D);
	[d,ii] = distinct(d,opts.tau);
	nfied = sum(d==d(2));
else
	[V,D] = eig(full(L));
	d = diag(D);
	[d,ii] = distinct(d,opts.tau);
	nfied = sum(d==d(2));
end

