function [EPSINIS,EPSSCA,NSING]=SEAW(EPSINIS,EPSSCA,MAXCOL,TOL,NDIGIT,IFLAG)
%  SEAW   Scalar epsilon-algorithm with Wynn's particular rules
%                                                                               
%  [EPSINIS,EPSSCA,NSING]=SEAW(EPSINIS,EPSSCA,MAXCOL,TOL,NDIGIT,IFLAG)                                    
%                                                                          
%  Arguments:                                                              
%                                                                          
%  EPSINIS  Input/output real value. 
%           In input, before the k-th call of the function, it must contain
%           the new term of the original sequence S_{k-1}.
%           In output, after the k-th call, it contains
%            eps_{k-1}^{(0)}             if k <= MAXCOL+1, k ODD   
%            eps_{k-2}^{(1)}             if k <  MAXCOL+1, k EVEN  
%            eps_{MAXCOL}^{(k-1-MAXCOL)} if k >  MAXCOL+1  
%           Thus, in the successive calls the function returns the sequence
%            eps_0^{(0)}, eps_1^{(0)}, eps_2^{(0)}, eps_2^{(1)} ....
%            eps_MAXCOL^{(0)}, eps_MAXCOL^{(1)}, eps_MAXCOL^{(2)}, .....
%                                                                          
%  EPSSCA   Input/output real vector.  
%           It contains after the k-th call the last computed backward
%           diagonal of the epsilon scheme:      
%           - IF k <= MAXCOL+1 
%               eps_0^{(k-1)}, eps_1^{(k-2)}, ... eps_{k-1}^{(0)}
%           - IF k > MAXCOL+1  
%               eps_0^{(k-1)}, eps_1^{(k-2)}, ... eps_MAXCOL^{(k-1-MAXCOL)}
%           Before the first call its input value must be an empty array.
%                                                                          
%  MAXCOL   Input integer giving the index of the last column of the
%           epsilon scheme that the user wants to compute (lower index of
%           eps_m^{(n}). This value must be a positive even integer number.
%                                                                          
%  TOL      Input real value used in tests for zero. If abs(X) < TOL, then
%           X is considered to be zero
% 
%  NDIGIT   Input integer value. It represents the number of digits that 
%           the user allows to lose due to the cancellation error.
%           If, for any index k and for any n, the relation
%           abs( (eps_k^{(n+1)}-eps_k^{(n)})/eps_k^{(n)} ) < 10^{-NDIGIT}
%           holds, then the particular rules of Wynn shall be used to 
%           compute the value eps_{k+3}^{(n-1)}. 
%           But if, in the next call, we also have 
%           abs((eps_k^{(n+2)}-eps_k^{(n+1)})/eps_k^{(n+1)}) < 10^{-NDIGIT}
%           then the function stops due to a non isolated singularity    
%                                                                                
%  IFLAG    Input integer to be set to zero at the 'first' call of the
%           function for the initializations of the scheme.  
%           If this value is different from zero, or the argument is
%           missing, the function continues in expanding the previous
%           scheme. For a new application of the algorithm, the user must
%           remember to call the function with IFLAG equal to zero
%
%  NSING    Optional output integer value. It represents the total number  
%           of singularities detected in all the calls of the function
%
%  Internal persistent arrays:                                           
%                                                                        
%  INDSAV   Integer vector in which the function saves the lower indexes of
%           the singularities. Needed for detecting, in the next call, the 
%           non isolated singularities. Its dimension do not exceed the 
%           value MAXCOL-3.
%
%  ABVEC    Real array. In the first row the function saves the values of A
%           In the second row, the values of B. A and B are the notation
%           used by Wynn for his particular rules. The number of columns
%           do not exceed the value MAXCOL-3.
%                                                                          
%  REMARKS:                                                                
%                                                                         
%   -  The EPSSCA array must not be modified by the user between
%      two consecutive calls of the function                                    
%   -  To obtain the value eps_{2*j}^{(0)} (2*J <= MAXCOL) the user must
%      call this function (2*J+1) times (including the first call)       
     
%  AUTHORS: Claude Brezinski and Michela Redivo-Zaglia
%  V1.0 - December 2016

persistent INDSAV ABVEC 
persistent INDM LOCINS NEWINS LISTBEG LISTEND IEC IAC IBC ANDIG NS MAXSIN 
persistent Sfsn Sfile Sfs
%
%                        ... EXECUTABLE STATEMENTS
%
%                        ... CHECK THE ARGUMENTS
if nargin < 5               % CHECK the number of input arguments
    error('SEAW - Not enought input arguments ')
elseif nargout < 2          % CHECK the number of output arguments
    error('SEAW - Not enought output arguments ')
elseif (mod(MAXCOL,2) ~= 0) || MAXCOL < 0 % CHECK MAXCOL 
    error('SEAW - MAXCOL must be a positive even integer number')
end

%
%                        ... FIRST CALL OF THE FUNCTION WITH IFLAG == 0 
%                            OR TRIVIAL CASE (MAXCOL == 0)
%
if ((nargin == 6) && (IFLAG == 0)) || (MAXCOL == 0) 
    if ((nargin == 6) && (IFLAG == 0)) 
%                        ... OPEN A NEW LOG FILE FOR SEAW
        if ~isempty(Sfsn)
            Sfsn = Sfsn+1;
        else
            Sfsn = 1;              
        end
        Sfs=['SEAW_',date,'_',num2str(Sfsn),'.log'];
        Sfile = fopen(Sfs,'w');
        fprintf(Sfile,[' === SEAW.log file === \n']);
        fprintf(Sfile,[' === Cycle or Run ',num2str(Sfsn),' === \n']);
    else
%                        ... OPEN THE LOG FILE FOR SEAW
        Sfile = fopen(Sfs,'a');    
        fprintf(Sfile,['\n === MAXCOL = 0 ===']);
    end
%                        ... INITIALIZATIONS
%       INDM is a counter for the calls of the function. The function
%       at each call computes the INDM+1 backward diagonal (not used
%       in the trivial case, that is when MAXCOL == 0)
    INDM   = 0;
%       MAXSIN is the maximum number of singularities that can be stored
%       The particular rules cannot be used when MAXCOL<=2
    MAXSIN = MAXCOL-3;
%
%                        ... SET THE BACKWARD DIAGONAL EQUAL TO eps_0^{(0)}
%                            OR TO eps_0^{(n})(IN THE TRIVIAL CASE) AND
%                            NSING TO ZERO 
%                           
    EPSSCA(1) = EPSINIS;   
    NSING = 0;
    if MAXCOL ~= 0  
%                        ... INITIALIZE THE PERSISTENT VARIABLES
        NS = 0;
        ANDIG  = 10^(-NDIGIT);
        LOCINS = false;
        NEWINS = false;
        IEC    = false;
        IAC    = false;
        IBC    = false;  
        fprintf(Sfile,['\n === Compute the diagonal ', ...
            num2str(INDM+1,'%d'), ' ===']);
    end
    return
%
%                        ... END OF THE FIRST CALL OF THE FUNCTION OR
%                            TRIVIAL CASE (MAXCOL == 0)
end
%                        ... NEXT CALLS OF THE FUNCTION   
%
%                        ... UPDATE INDM (IT REPRESENTS THE BACKWARD  
%                            DIAGONAL TO BE COMPUTED 
INDM = INDM + 1;
fprintf(Sfile,['\n === Compute the diagonal ', num2str(INDM+1,'%d'),' ===']);
%
%                        ... IF INDM > MAXCOL THEN THE ELEMENT 
%                            eps_MAXCOL^{(NUMBER_OF_CALLS-MAXCOL+1)}
%                            IS COMPUTED
IND = min([INDM,MAXCOL]);
%
%                        ... IF THERE IS NO LOCAL INSTABILITY PRESENT
%                            (COMING FROM THE PREVIOUS CALL)
%                            RESET THE INDEXES LISTBEG AND LISTEND AND
%                            THE VECTOR INDSAV.
%
if (~LOCINS)
    LISTBEG = 1;
    LISTEND = 0;
    INDSAV = []; 
end
%
%                        ... INITIALIZATION FOR THE AUXILIARY VARIABLE
%                            AUX1. WE SET IT EQUAL TO eps_{0}^(INDM)
%
AUX1   = EPSINIS;
%
%                        ... STARTING LOOP FOR COMPUTING THE ELEMENTS
%                            OF THE ASCENDING DIAGONAL
%
for INDS = 0:IND - 1
%
%                        ... COMPUTE THE DENOMINATOR OF THE FORMULA
%
    AUX0 = AUX1 - EPSSCA(INDS+1);  
%
%                        ... CONTROL FOR A NEW POINT OF INSTABILITY.
%
    RD = max([abs(AUX1),abs(EPSSCA(INDS+1))]);  
    if (abs(EPSSCA(INDS+1)) > TOL) && (~IEC)  && ... 
            ((RD < TOL) || (abs(AUX0/RD) < ANDIG)) && ...
            (INDS < IND-1) && (INDS < MAXCOL-2) 
        NEWINS  = true;         % there is a new point of instability 
        fprintf(Sfile,['\n New point of instability found in column number ', ...
            num2str(INDS+1,'%d')]);
        NS = NS+1; % increase the TOTAL number of singularities 
        LISTEND = LISTEND+1;    % increase the index of INDSAV 
        if (LISTEND > MAXSIN)   % check if the maximum allowed number
                                % of isolated singularities is reached
            error(['SEAW - Maximum allowed number of isolated ',...
                'singularities is reached.'])  
        end
    end
%
    if (LOCINS || NEWINS)
%
%                        ... WHEN THERE IS A LOCAL INSTABILITY PRESENT
%                            OR A NEW POINT OF INSTABILITY WE USE
%                            THE SINGULAR RULE OF THE ALGORITHM.
%
        if (NEWINS) 
%
%                        ... THERE IS A NEW POINT OF INSTABILITY ...
%
            if (LOCINS) 
%
%                        ... AND THERE IS A LOCAL INSTABILITY PRESENT.
%
%
%                        ... IF INDS COINCIDES WITH THE FIRST ELEMENT
%                            STORED IN INDSAV THEN THERE IS A NON ISOLATED
%                            SINGULARITY AND IT IS IMPOSSIBLE TO CONTINUE.
%
                if (INDS == INDSAV(LISTBEG)) 
                    error(['SEAW - NON isolated singularity.',... 
                        ' Impossible to continue.'])
                end
            end
%
%                        ... LOCAL INSTABILITY PRESENT IS TRUE.
%
            LOCINS = true;
%
%                        ... WE MUST COMPUTE A!
%
            IAC    = true;
%
%                        ... NEW POINT OF INSTABILITY TREATED!
%
            NEWINS = false;
%
%                        ... ADD THE CURRENT INDS TO THE END OF INDSAV.
%
            INDSAV(LISTEND) = INDS;
        end
%
%                        ... WHEN THE CONTROL ON A NEW POINT OF INSTABILITY
%                            IS MADE, THE CONTROL FOR THE COMPUTATION OF E
%                            IS PERFORMED
%
        if (IEC) 
            fprintf(Sfile,['\n PARTICULAR rule for computing E in', ...
                ' column ', num2str(INDS+1,'%d')]);
%
%                        ... WE MUST COMPUTE E AND STORE IT IN AUX0
%
%                        ... DTMP is r = D + B - A
            DTMP  = DTMP + ABVEC(LISTBEG+MAXSIN) - ABVEC(LISTBEG);
            if (abs(EPSSCA(INDS)) < TOL) % check for the denominator
                disp(['Value of C ',num2str(abs(EPSSCA(INDS)),'%15.5e'), ...
                     ' in column ', num2str(INDS+1,'%d')])
                error(['SEAW - Division by a  C < TOL. ', ... 
                  ' Impossible to continue.'])
            end
%                        ... AUX0 is E = r / (1 + r/C)
            AUX0 = DTMP / (1 + DTMP / EPSSCA(INDS));
%
%                        ... E COMPUTED!
%
            IEC     = false;
%
%                        ... INCREASE THE POINTER ON THE VECTOR INDSAV.
%
            LISTBEG = LISTBEG + 1;
%
%                        ... IF, IN THE VECTOR INDSAV, ALL THE STORED
%                            INDEXES HAVE BEEN CONSIDERED, THERE IS NO
%                            LOCAL INSTABILITY PRESENT.
%
            if (LISTBEG > LISTEND), LOCINS = false; end
%
        else      
%                        ... IF WE DON'T COMPUTE E THE NORMAL RULE IS
%                            CONSIDERED.
% 
            fprintf(Sfile,['\n NORMAL rule in column ',num2str(INDS+1,'%d')]);
            if (abs(AUX0) < TOL) % check for the denominator
                disp(['Value of denominator ',num2str(abs(AUX0), ...
                  '%15.5e'), ' in column ', num2str(INDS+1,'%d')])
                error(['SEAW - Division by a value < TOL in ', ... 
                  'the normal rule. Impossible to continue.'])
            end                
            AUX0 = 1 / AUX0;
            if (INDS ~= 0), AUX0 = AUX0 + EPSSCA(INDS); end 
        end
%
        if (LOCINS) 
%
%                        ... THERE IS A LOCAL INSTABILITY PRESENT ...
%
            if (IAC) 
                fprintf(Sfile,['\n PARTICULAR rule for computing A in column ', ...
                    num2str(INDS+1,'%d')]);
%
%                        ... AND WE MUST COMPUTE AND STORE THE VALUE
%                            A = W / (1 - W/C).
%
                if (INDS ~= 0) 
                    if (abs(AUX0) < TOL) % check for the denominator
                        disp(['Value of denominator ', ...
                            num2str(abs(AUX0),'%15.5e'), ...
                            ' in column ', num2str(INDS+1,'%d')])
                        error(['SEAW - Division by a value < TOL. ', ... 
                            'Impossible to continue.'])
                    end
                    ABVEC(LISTEND) = EPSSCA(INDS) ...  
                             / (1 - EPSSCA(INDS) / AUX0);  
                else
                    ABVEC(LISTEND) = 0;
                end
%
%                        ... A COMPUTED!
%
                IAC = false;
%
%                        ... WE MUST COMPUTE ALSO B!
%
                IBC = true;
%
%                        ... IF WE DON'T COMPUTE A ...
%
            elseif (IBC) 
                fprintf(Sfile,['\n PARTICULAR rule for computing B in column ', ...
                    num2str(INDS+1,'%d')]);
%
%                        ... BUT WE MUST COMPUTE AND STORE THE VALUE
%                            B = N / (1 - N/C).
%
                if (abs(AUX1) < TOL) % check for the denominator
                    disp(['Value of denominator ',num2str(RD,'%15.5e'), ...
                        ' in column ', num2str(INDS+1,'%d')])
                    error(['SEAW - Division by a value < TOL. ', ... 
                        'Impossible to continue.'])
                end
                ABVEC(LISTEND+MAXSIN) = EPSSCA(INDS+1) ... 
                          / (1 - EPSSCA(INDS+1) / AUX1); 
%
%                        ... B COMPUTED!
%
                IBC = false;
%
%                        ... IF WE DON'T COMPUTE A AND WE DON'T COMPUTE B ...
%
            elseif (INDS ~= 0) && (INDS == (INDSAV(LISTBEG)+1))  
%
%                        ... AND THE CURRENT INDEX INDS IS THE NEXT
%                            INDEX OF INSTABILITY IN INDSAV WE COMPUTE
%                            D =  S / (1 - S/C) ...
%
                fprintf(Sfile,['\n PARTICULAR rule for computing D in column ', ...
                    num2str(INDS+1,'%d')]);
                if (abs(EPSSCA(INDS+1)) < TOL)  % check for the denominator
                    disp(['Value of denominator ', ...
                        num2str(abs(EPSSCA(INDS+1)),'%15.5e'), ...
                        ' in column ', num2str(INDS+1,'%d')])
                    error(['SEAW - Division by a value < TOL. ', ... 
                        'Impossible to continue.'])
                end           
                DTMP = AUX1 / (1 - AUX1 / EPSSCA(INDS+1)); 
%
%                        ... AND WE MUST COMPUTE E!
%
                IEC = true;
            end
        end
    else
%
%                        ... WHEN THERE IS NO LOCAL INSTABILITY PRESENT
%                            NOR A NEW POINT OF INSTABILITY WE USE
%                            THE NORMAL RULE OF THE ALGORITHM.
%
        fprintf(Sfile,['\n NORMAL rule in column ',num2str(INDS+1,'%d')]);
        if (abs(AUX0) < TOL) % check for the denominator
            disp(['Value of denominator ',num2str(abs(AUX0),'%15.5e'), ...
                    ' in column ', num2str(INDS+1,'%d')])
            error(['SEAW - Division by a value < TOL in ', ... 
                  'the normal rule. Impossible to continue.'])
        end
        AUX0 = 1 / AUX0;
        if (INDS ~= 0), AUX0 = AUX0 + EPSSCA(INDS); end  
    end
%
%                        ... SHIFT THE RHOMBUS.
%
    if (INDS ~= 0), EPSSCA(INDS) = AUX2; end  
    AUX2 = AUX1;
    AUX1 = AUX0;
%
%                        ... END LOOP FOR INDS
%
end
%
%                        ... SAVE FOR THE FINAL VALUES OF THE DIAGONAL.
%
EPSSCA(IND)   = AUX2;  
EPSSCA(IND+1) = AUX1;  
%
%                        ... SET THE RESULT VALUE.
%
if (mod(IND,2) == 0) 
    EPSINIS = EPSSCA(IND+1);  
else
    EPSINIS = EPSSCA(IND);
end
% Set the output parameter counting the numbero of singulatities
NSING = NS;
