function spacetime
% function spacetime
%
%   This is main file for the first example in
% 
%       R. Andreev
%       Space-time discretization of the heat equation
%       Numerical Algorithms, 2014
%       (see README.txt for precise reference)
% 
%   It approximately computes the solution to a heat equation
%   and produces a plot of several temporal snapshots of it.
%   A PDF file of the plot is written in the current directory.

%   R. Andreev, 2013.09.25

%%% INITIALIZE SPATIAL FEM

	disp('Initialize the spatial FEM');
	
	femX_init(true)
	[Mx, Ax] = femX_MA();

	% Note:
	%	femX_* routines share global variables

%%% DEFINE TEMPORAL MESH

	disp('Define the temporal mesh');

	T = 20; K = 100;
	TE = T * sort([0, rand(1, K-1), 1]);

%%% USE THE STABLE MINIMAL RESIDUAL PETROV-GALERKIN ?

	use_mrpg = true;

%%% TEMPORAL FEM

	disp('Assemble temporal FEM matrices');

	[MtFE, CtFE, TF] = femT_assemFE(TE, use_mrpg);
	[MtE, AtE] = femT_assemE(TE);
	MtF = femT_assemF(TF); 

%%% DEFINE SPACE-TIME OPERATORS

	disp('Define space-time operators');

	function Bw = B(w)
		Bw = { Mx * w * CtFE' + Ax * w * MtFE', Mx * w(:,1) };
	end

	function Btv = Bt(v)
		Btv = Mx' * v{1} * CtFE + Ax' * v{1} * MtFE;
		Btv(:,1) = Btv(:,1) + Mx' * v{2};
	end

	function iNd = iN(d)
		iNd = { Ax \ (d{1} / MtF), Mx \ d{2} };
	end

	[VtE, DtE] = eig(full(AtE), full(MtE));
	gamma = sqrt(max(0,diag(DtE)));
	function iMw = iM(w)
		iMw = w * VtE;
		parfor j = 1:length(TE)
			iMw(:,j) = real((Ax + 1i * gamma(j) * Mx) \ iMw(:,j));
		end
		iMw = iMw * VtE';
	end

%%% ASSEMBLE THE SPACE-TIME LOAD VECTOR

	disp('Assemble the space-time load vector');

	b = femTX_assemLoad(TF, @data_f, @data_g, @data_h, QR_Trapz());

	% Note: 
	%   for, say, 4 point Gauss--Legendre quadrature
	%   replace QR_Trapz() by QR_GauLeg(4)

%%% SOLVE

	disp('Solve the linear system');

	tol = 1e-4; maxit = 100;
	u = glsqr(@B, @Bt, b, tol, maxit, @iM, @iN);

	% Note:
	%   for compatibility with B, iN and Bt,
	%   glsqr supports operations on cell arrays

%%% DISPLAY

	disp('Display temporal snapshots of the solution');

	t = 0:5; U = interp1(TE, u', t)';
	for k = 1:size(U,2)
		subplot(1, size(U,2), k); femX_show(U(:,k));
	end

%%% GRAPHICAL POSTPROCESSING

	colormap bone
	AH = findall(gcf, 'type', 'axes')';
	axis(AH, 'square');
	set(AH, 'XTick', [Inf], 'YTick', [Inf], 'ZTick', [Inf]);
	ca = caxis;
	zl = zlim;
	for ah = AH
		ca = max(ca, caxis(ah) .* [-1 1]);
		zl = max(zl, zlim(ah) .* [-1 1]);
	end
	for ah = AH
		caxis(ah, ca .* [-1 1]);
		zlim(ah, zl .* [-1 1]);
	end
	disp('The displayed snapshots are at');
	disp(['t = ' num2str(t)]);
	saveas(gcf, 'result', 'pdf');
end
