function b = femTX_assemLoad(TF, f, g, h, QuadRule)
% function b = femTX_assemLoad(TF, F, G, H, QuadRule)
%
%   Assemble the space-time load vector
%
%   TF is a sorted row vector representing the temporal mesh
%   
%   F and G are function handles accepting at least one argument 
%
%   Assumption:
%       F(t)  is the same as  @(varargin) F(t, varargin{:})
%       G(t)  is the same as  @(varargin) G(t, varargin{:})
%   Note: 
%       femX_b(F(t), G(t)) is invoked for different values of t
%
%   H is passed to femX_b
%
%   femX_b is expected to return a column vector of consistent size
%
%   QuadRule is a function handle that
%       accepts a temporal interval J = [t1 t2] and
%       returns [tI, wI, R]
%   These are R quadrature nodes tI with weights wI on J
%
%   b is a struct where
%
%       b{1} is the space-time source part
%       b{2} is the initial condition part
%       of the space-time load vector
%
%
%   See Sec 7.2 and Sec 6.3 in
%
%       R. Andreev
%       Space-time discretization of the heat equation
%       Numerical Algorithms, 2014
%       (see README.txt for precise reference)

%   R. Andreev, 2012.10.17

	assert(isequal(size(TF), [1 numel(TF)]), 'TF should be a row vector');

	% Assemble the ``initial condition'' part
	% Provide zero Neumann data
	b2 = femX_b(h, @(varargin)0);

	% Assemble the ``space-time source'' part
	% Allocate space for the space-time load vector
	b1 = zeros(length(b2), length(TF)-1);
	% Loop over the temporal elements
	parfor k = 1:size(b1,2)
		% Quadrature on the k-th temporal element
		[tI, wI, R] = QuadRule(TF([k k+1]));
		% Integrate, using that the temporal shape function equals 1
		for r = 1:R
			b1(:,k) = b1(:,k) + wI(r) * femX_b(f(tI(r)), g(tI(r)));
		end
	end

	b = {b1, b2};
end
