/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D.                                 */
/*                                                                        */
/*        This file contains the function computeWThreaded, which is a    */
/*        threaded version of the function computeW. It is used in        */
/*        exactly the same way as computeW and has the same output as     */
/*        computeW. This function however is optimized for system with    */
/*        multiple cores and distributes the work load equally among all  */
/*        cores. For this purpose it uses threads.h on the one hand and   */
/*        POSIX threads on the other hand.                                */
/*                                                                        */
/*        This file contains only the implementation. For extensive       */
/*        documentation consult the corresponding header file.            */
/*                                                                        */
/*        Beware: As POSIX threads may not be available, this file is     */
/*        only compiled if the pre-processor flag HILTHREADS is set.      */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '10, '12                                    */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#ifdef HILTHREADS

#include "threadedW.h"

void computeWThreaded(double* matrix, const double* coordinates,
    const double* elements, int nC, int nE, double eta) {
  Matrix* m = newMatrix(nC, nC, matrix);
  genericBoss(m,
    newCompWorkerSharedDataVLike(coordinates, elements, nC, nE, eta),
    computeWThreadedWorker, 0, nE);
}

void* computeWThreadedWorker(void* data) {
  int i=0, j=0;
  int aidx=0, bidx=0, cidx=0, didx=0;
  double a0=0.0, a1=0.0, b0=0.0, b1=0.0, c0=0.0, c1=0.0, d0=0.0, d1=0.0;
  const double *coordinates = NULL, *elements = NULL;
  int nC = 0, nE = 0;
  double eta = 0.0;
  double tmp = 0.0;
  CompWorkerData* cwdata = (CompWorkerData*) data;

  cwdata->status = HILTHR_COMPWORKER_STATUS_WORKING;
  coordinates = cwdata->sharedData.VLike->coordinates;
  elements    = cwdata->sharedData.VLike->elements;
  nC          = cwdata->sharedData.VLike->nC;
  nE          = cwdata->sharedData.VLike->nE;
  eta         = cwdata->sharedData.VLike->eta;

  for (i = cwdata->first_col; i <= cwdata->last_col; ++i) {
    MatrixColumn *aidxColumn = NULL, *bidxColumn = NULL;
    double *aidxValues = NULL, *bidxValues = NULL;

    aidx = (int) elements[i] - 1;
    a0 = coordinates[aidx];
    a1 = coordinates[aidx+nC];

    bidx = (int) elements[i+nE] - 1;
    b0 = coordinates[bidx];
    b1 = coordinates[bidx+nC];

    aidxColumn  = newMatrixColumn(aidx, nC);
    aidxValues = aidxColumn->values;
    bidxColumn  = newMatrixColumn(bidx, nC);
    bidxValues = bidxColumn->values;

    for (j = 0; j < nE; ++j) {
      cidx = (int) elements[j] - 1;
      c0 = coordinates[cidx];
      c1 = coordinates[cidx+nC];

      didx = (int) elements[j+nE] - 1;
      d0 = coordinates[didx];
      d1 = coordinates[didx+nC];

      tmp = computeWij(c0, c1, d0, d1, b0, b1, a0, a1, eta);

      aidxValues[cidx] += tmp;
      aidxValues[didx] -= tmp;
      bidxValues[cidx] -= tmp;
      bidxValues[didx] += tmp;
    }
  
    matrixColumnQueueAppend(cwdata->storage.queue, aidxColumn);
    matrixColumnQueueAppend(cwdata->storage.queue, bidxColumn);
  }

  cwdata->status = HILTHR_COMPWORKER_STATUS_FINISHED;
  return NULL;
}

#endif

