/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D.                                 */
/*                                                                        */
/*        This file contains the function computeVThreaded, which is a    */
/*        threaded version of the function computeV. It is used in        */
/*        exactly the same way as computeV and has the same output as     */
/*        computeV. This function however is optimized for system with    */
/*        multiple cores and distributes the work load equally among all  */
/*        cores. For this purpose it uses threads.h on the one hand and   */
/*        POSIX threads on the other hand.                                */
/*                                                                        */
/*        This file contains only the implementation. For extensive       */
/*        documentation consult the corresponding header file.            */
/*                                                                        */
/*        Beware: As POSIX threads may not be available, this file is     */
/*        only compiled if the pre-processor flag HILTHREADS is set.      */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '10, '12                                    */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#ifdef HILTHREADS

#include "threadedV.h"

void computeVThreaded(double* V, int nC, int nE, const double* coordinates,
    const double* elements, double eta) {
  int i = 0, firstRow = 0, lastRow = -1;
  CompWorker* threads[MAX_NUMOF_VWORKERS];
  int actualNumberOfThreads = MAX_NUMOF_VWORKERS;
  int targetSize = nE*(nE+1) / (2*actualNumberOfThreads);
  Matrix* matrix = newMatrix(nE,nE,V);
  CompWorkerSharedData shared =
      newCompWorkerSharedDataVLike(coordinates, elements, nC, nE, eta);

  if (targetSize < HILTHR_V_MINROW_PER_WORKER) {
    actualNumberOfThreads = (int) nE*(nE+1)/2 / HILTHR_V_MINROW_PER_WORKER;
    if (actualNumberOfThreads < 1) {
      actualNumberOfThreads = 1;
      targetSize = nE;
    }
    else {
      targetSize = HILTHR_V_MINROW_PER_WORKER;
    }
  }

#ifdef DIAGNOSTICS
  fprintf(DIAGNOSTICS_FH,
    "matrixSize (nE)      : %d\n"
    "targetSize           : %d\n"
    "actualNumberOfThreads: %d\n",
    nE, targetSize, actualNumberOfThreads);
#endif

  for (i=0; i < actualNumberOfThreads; ++i) {
    firstRow = lastRow+1;
    if (i == actualNumberOfThreads-1) {
      lastRow = nE-1;
    }
    else {
      lastRow = (int) sqrt(1+firstRow*(firstRow+1)+2*targetSize);
    }

    threads[i] = newCompWorkerSimple(computeVThreadedWorker, matrix,
                                      firstRow, lastRow, shared);
    #ifdef DIAGNOSTICS
    fprintf(DIAGNOSTICS_FH,
      "Created thread %d for columns %d to %d\n", i, firstRow, lastRow);
    #endif
  }

  for (i=0; i < actualNumberOfThreads; ++i) {
    pthread_join(threads[i]->thread, NULL);
    free(threads[i]);
  }

  freeCompWorkerSharedDataVLike(shared.VLike);
}

void* computeVThreadedWorker(void* data) {
  int nC = 0, nE = 0, firstRow = 0, lastRow = 0;
  double* V = NULL;
  const double* coordinates;
  const double* elements;
  double eta = 0.;
  int i = 0, j = 0, aidx = 0, bidx = 0, cidx = 0, didx = 0;
  double a0 = 0., a1 = 0., b0 = 0., b1 = 0.,
          c0 = 0., c1 = 0., d0 = 0., d1 = 0.;
  CompWorkerData* wdata = (CompWorkerData*) data;

  nC          = wdata->sharedData.VLike->nC;
  nE          = wdata->sharedData.VLike->nE;
  V           = wdata->storage.matrix->storage;
  coordinates = wdata->sharedData.VLike->coordinates;
  elements    = wdata->sharedData.VLike->elements;
  eta         = wdata->sharedData.VLike->eta;
  firstRow    = wdata->first_col;
  lastRow     = wdata->last_col;

  for (i=firstRow; i<=lastRow; ++i) {
    aidx = (int) elements[i]-1;        /* 1st node of element T_i = [A,B] */
    a0 = coordinates[aidx];
    a1 = coordinates[aidx+nC];

    bidx = (int) elements[i+nE]-1;     /* 2nd node of element T_i = [A,B] */
    b0 = coordinates[bidx];
    b1 = coordinates[bidx+nC];

    for (j=0; j<=i; ++j) {
      cidx = (int) elements[j]-1;      /* 1st node of element T_j = [C,D] */
      c0 = coordinates[cidx];
      c1 = coordinates[cidx+nC];

      didx = (int) elements[j+nE]-1;   /* 2nd node of element T_j = [C,D] */
      d0 = coordinates[didx];
      d1 = coordinates[didx+nC];

      V[i + j*nE] = computeVij(a0,a1,b0,b1, c0,c1,d0,d1,eta);
      V[j + i*nE] = V[i + j*nE];
    }
  }

  return NULL;
}

#endif
