/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D.                                 */
/*                                                                        */
/*        This file contains the function computeKThreaded, which is a    */
/*        threaded version of the function computeK. It is used in        */
/*        exactly the same way as computeK and has the same output as     */
/*        computeK. This function however is optimized for system with    */
/*        multiple cores and distributes the work load equally among all  */
/*        cores. For this purpose it uses threads.h on the one hand and   */
/*        POSIX threads on the other hand.                                */
/*                                                                        */
/*        This file contains only the implementation. For extensive       */
/*        documentation consult the corresponding header file.            */
/*                                                                        */
/*        Beware: As POSIX threads may not be available, this file is     */
/*        only compiled if the pre-processor flag HILTHREADS is set.      */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '10, '12                                    */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#ifdef HILTHREADS

#include "threadedK.h"

void computeKThreaded(double* matrix, const double* coordinates,
    const double* elements, int nC, int nE, double eta) {
  Matrix* m = newMatrix(nC, nE, matrix);
  genericBoss(m,
    newCompWorkerSharedDataVLike(coordinates, elements, nC, nE, eta),
    computeKThreadedWorker, 0, nE);
}

void* computeKThreadedWorker(void* data) {
  int i=0, j=0;
  int aidx=0, bidx=0, cidx=0, didx=0;
  double a0=0.0, a1=0.0, b0=0.0, b1=0.0, c0=0.0, c1=0.0, d0=0.0, d1=0.0;
  double linetest1=0.0, linetest2=0.0;
  double I0=0.0, I1=0.0;
  const double *coordinates = NULL, *elements = NULL;
  int nC = 0, nE = 0;
  double eta = 0.0;
  CompWorkerData* cwdata = (CompWorkerData*) data;

  cwdata->status = HILTHR_COMPWORKER_STATUS_WORKING;
  coordinates = cwdata->sharedData.VLike->coordinates;
  elements    = cwdata->sharedData.VLike->elements;
  nC          = cwdata->sharedData.VLike->nC;
  nE          = cwdata->sharedData.VLike->nE;
  eta         = cwdata->sharedData.VLike->eta;
  
  for (j=cwdata->first_col; j<=cwdata->last_col; ++j) {
    MatrixColumn *aidxColumn = NULL, *bidxColumn = NULL;
    double *aidxValues = NULL, *bidxValues = NULL;

    aidx = (int) elements[j]-1;                 /* 1st node of Ti = [A,B] */
    a0 = coordinates[aidx];
    a1 = coordinates[aidx+nC];

    bidx = (int) elements[j+nE]-1;              /* 2nd node of Ti = [A,B] */
    b0 = coordinates[bidx];
    b1 = coordinates[bidx+nC];

    aidxColumn  = newMatrixColumn(aidx, nC);
    aidxValues = aidxColumn->values;
    bidxColumn  = newMatrixColumn(bidx, nC);
    bidxValues = bidxColumn->values;
   
    for (i=0;i<nE;++i) {
      cidx = (int) elements[i]-1;               /* 1st node of Tj = [C,D] */
      c0 = coordinates[cidx];
      c1 = coordinates[cidx+nC];

      didx = (int) elements[i+nE]-1;            /* 2nd node of Tj = [C,D] */
      d0 = coordinates[didx];
      d1 = coordinates[didx+nC];
      
      linetest1 = fabs( (a0-c0)*(b1-a1) - (a1-c1)*(b0-a0) );
      linetest2 = fabs( (a0-d0)*(b1-a1) - (a1-d1)*(b0-a0) );

      if( linetest1>EPS*( sqrt((a0-c0)*(a0-c0)+(a1-c1)*(a1-c1)) ) ||
          linetest2>EPS*( sqrt((a0-d0)*(a0-d0)+(a1-d1)*(a1-d1)) ) ) {
        computeKij(&I0,&I1,eta,c0,c1,d0,d1,a0,a1,b0,b1);
        aidxValues[i] = I0-I1;
        bidxValues[i] = I0+I1;
      }
    }

    matrixColumnQueueAppend(cwdata->storage.queue, aidxColumn);
    matrixColumnQueueAppend(cwdata->storage.queue, bidxColumn);
  }

  cwdata->status = HILTHR_COMPWORKER_STATUS_FINISHED;
  return NULL;
}

#endif

