/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D. It provides functions to        */
/*        compute single layer potential (slp) type integrals.            */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '09, '10, '12                               */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#ifndef _SINGLELAYERPOTENTIAL_H_GUARD_
#define _SINGLELAYERPOTENTIAL_H_GUARD_

#include <math.h>
#include <assert.h>
#include "geometry.h"
#include "gaussQuadrature.h"

/********************************** slp ***********************************/
/* INPUT: k>=0 is an integer, u and v represent 2-dimensional vectors.    */
/* OUTPUT: A double that is the value of the integral int_{-1}^{+1} s^k   */
/*        log|s*u+v|^2 ds.                                                */
/* DESCRIPTION: This function calculates the integral as stated above.    */
/*        Internally, it uses the slpIterative-function and returns the   */
/*        last element of the returned array after freeing the memory     */
/*        that is used by that array.                                     */
/**************************************************************************/
double slp(int k, double u[2], double v[2]);

/****************************** slpIterative ******************************/
/* INPUT: k>=0 is an integer, u and v are vectors in \R^2.                */
/* OUTPUT: An array r of length k+1 that is given by r[i] = int_{-1}^{+1} */
/*        s^i log|s*u+v|^2 ds.                                            */
/* DESCRIPTION:                                                           */
/**************************************************************************/
double* slpIterative(int k, double u[2], double v[2]);

/******************************* doubleSlp ********************************/
/* INPUT: k,l >= 0 are integers. u, v and w are vectors in \R^2           */
/* OUTPUT: Value of the integral int_{-1}^{+1} int_{-1}^{+1} s^k t^l      */
/*        log|su+tv+w| dt ds                                              */
/* DESCRIPTION: This function calculates the integral as stated above in  */
/*        a purely analytical way.                                        */
/**************************************************************************/
double doubleSlp(int k, int l, double u[2], double v[2], double w[2]);

/******************************* computeVij *******************************/
/* INPUT: double a0, a1, b0, b1, c0, c1, d0, d1 are the coordinates of    */
/*        the                                                             */
/*        points A, B, C, D. double eta is the admissible constant. It is */
/*        a number between 0 and 1.                                       */
/* OUTPUT: double, the value of the Galerkin integral -1/(2pi) \int_Tj    */
/*        \int_Ti log |x-y| dsy dsx where T_i = [A,B] and T_j = [C,D].    */
/* DESCRIPTION: This function uses computeWij and multiplies the result   */
/*        with |Ti| * |Tj| to obtain the result.                          */
/**************************************************************************/
double computeVij(double a0, double a1, double b0, double b1,
                  double c0, double c1, double d0, double d1, double eta);

/******************************* computeWij *******************************/
/* INPUT: double a0, a1, b0, b1, c0, c1, d0, d1 are the coordinates of    */
/*        the                                                             */
/*        points A, B, C, D. double eta is the admissible constant. It is */
/*        a number between 0 and 1.                                       */
/* OUTPUT: double, the value of the Galerkin integral -1/(2pi) 1/|Ti| *   */
/*        1/|Tj| * \int_Tj \int_Ti log |x-y| dsy dsx where T_i = [A,B]    */
/*        and T_j = [C,D].                                                */
/* DESCRIPTION: This function checks whether the boundary elements T_i    */
/*        and T_j are admissible, this means that \dist(T_i, T_j) > eta   */
/*        \min\{\diam(T_i), \diam(T_j). In case, T_i and T_j are          */
/*        admissible computeWijSemianalytic is called. Otherwise          */
/*        computeWijAnalytic is called.                                   */
/**************************************************************************/
double computeWij(double a0, double a1, double b0, double b1,
                  double c0, double c1, double d0, double d1, double eta);

/*************************** computeWijAnalytic ***************************/
/* INPUT: double a0,a1,b0,b1,c0,c1,d0,d1 are the coordinates of the       */
/*        points A, B, C, D.                                              */
/* OUTPUT: Returns a double, the value of the Galerkin integral -1/(2pi)  */
/*        * 1/|Ti| * 1/|Tj| * int_Tj int_Ti log |x-y| dsy dsx, whereas Ti */
/*        = [A,B] and Tj = [C,D].                                         */
/* DESCRIPTION: This function calculates the double integral in a purely  */
/*        analytical way. It is called in case that Ti and Tj are not     */
/*        admissible.                                                     */
/**************************************************************************/
double computeWijAnalytic(double a0, double a1, double b0, double b1,
                           double c0, double c1, double d0, double d1);

/************************* computeWijSemiAnalytic *************************/
/* INPUT: double a0,a1,b0,b1,c0,c1,d0,d1 are the coordinates of the       */
/*        points A, B, C, D. The segments Ti = [A,B], Tj = [C,D] are be   */
/*        admissible.                                                     */
/* OUTPUT: Returns a double, the value of the Galerkin integral -1/(2pi)  */
/*        * 1/|Ti| * 1/|Tj| * int_Tj int_Ti log |x-y| dsy dsx, whereas Ti */
/*        = [A,B] and Tj = [C,D].                                         */
/* DESCRIPTION: This function replaces the outer integral with gauss      */
/*        quadrature to avoid cancellation effects. It must not get       */
/*        called whenever the segments Ti and Tj are not admissible.      */
/**************************************************************************/
double computeWijSemianalytic(double a0, double a1, double b0, double b1,
                               double c0, double c1, double d0, double d1);

#endif
