/**************************************************************************/
/* DESCRIPTION: This file is part of the HILBERT program package for the  */
/*        numerical solution of the Laplace equation with mixed boundary  */
/*        conditions by use of BEM in 2D. It provides functions to        */
/*        compute the newton potential.                                   */
/**************************************************************************/
/* VERSION: 3.1                                                           */
/**************************************************************************/
/* (C) 2009-2013 HILBERT-Team '10, '12                                    */
/* support + bug report:  hilbert@asc.tuwien.ac.at                        */
/**************************************************************************/
#ifndef _NEWTONPOTENTIAL_H_GUARD
#define _NEWTONPOTENTIAL_H_GUARD

#include <math.h>
#include "singleLayerPotential.h"
#include "gaussQuadrature.h"
#include "geometry.h"

/******************************** computeN ********************************/
/* INPUT: N is a nE x 3*nT matrix. nC and nE are the number of            */
/*        coordinates and elements, respectively, of the boundary \Gamma. */
/*        coordinates and elements contain the coordinates and elements   */
/*        of the boundary mesh. nV and nT are the number of vertices and  */
/*        triangles of the domain mesh.                                   */
/* OUTPUT: None, this function computes the entries of the matrix N       */
/* DESCRIPTION: The function computeNkj is called for the computation of  */
/*        the corresponding matrix entry.                                 */
/**************************************************************************/
void computeN(double* N, const double *coordinates, const double *elements,
    const double* vertices, const double* triangles, int nC,int nE,
    int nV, int nT, double eta);

/******************************* computeNkj *******************************/
/* INPUT: a, b are the points A,B of the line segment E_k. n1,n2,n3 are   */
/*        the nodes of the triangle T_\ell of the domain mesh. eta is an  */
/*        admissibility parameter.                                        */
/* OUTPUT: The function returns the value of the discrete Newton          */
/*        potential,                                                      */
/*                                                                        */
/*        -1/(2pi) \int_{E_k} \int_{T_\ell} log|x-y| dy ds_x.             */
/*                                                                        */
/* DESCRIPTION: First, the line segment E_k = [a,b] and the triangle      */
/*        [n1,n2,n3] are tested for admissibility. If they are            */
/*        admissible, the entry N_{k,j} is computed semi-analytically.    */
/*        Otherwise, the entry is computed analytically.                  */
/**************************************************************************/
double computeNkj(double a[2], double b[2], double n1[2],
                  double n2[2], double n3[2], double eta);

/********************* computeNkjSemiAnalyticSegment **********************/
/* INPUT: a, b are the points A,B of the line segment E. n1,n2,n3 are the */
/*        nodes of the triangle T of the domain mesh.                     */
/* OUTPUT: The function returns the value                                 */
/*                                                                        */
/*        -1/(2pi) \int_{E} \int_{T} log|x-y| ds_x dy.                    */
/*                                                                        */
/* DESCRIPTION: The entry N_{k,j} is computed SEMI-ANALYTIC. This means   */
/*        the function uses the function newtonPotential to compute the   */
/*        inner integral analytically. A Gauss quadrature rule is used to */
/*        perform the outer integration over the line segment.            */
/**************************************************************************/
double computeNkjSemiAnalyticSegment(double a[2], double b[2], double n1[2],
    double n2[2], double n3[2]);

/*************************** computeNkjAnalytic ***************************/
/* INPUT: a, b are the points A,B of the line segment E_k. n1,n2,n3 are   */
/*        the nodes of the triangle T_\ell of the domain mesh.            */
/* OUTPUT: The function returns the entry N_{k,j} of the discrete Newton  */
/*        potential                                                       */
/*                                                                        */
/*        N_{k,j} = -1/(2pi) \int_{E_k} \int_{T_\ell} log|x-y| dy ds_x    */
/*                                                                        */
/* DESCRIPTION: The entry N_{k,j} is computed "ANALYTIC". This means that */
/*        the two integrals are computed analytically up to an integral   */
/*        of a smooth function. This remaining integral is evaluated by   */
/*        use of a Gaussian quadrature rule. This, clearly is not a fully */
/*        analytic computation. Besides the above mentioned integral over */
/*        a smooth function, we have to compute double-integrals of the   */
/*        type \int_{-1}^{+1} \int_{-1}^{+1} s^k t^l log|su+tv+w| dt ds,  */
/*        which is done by calling the function doubleSlp.                */
/**************************************************************************/
double computeNkjAnalytic(double a[2], double b[2], double n1[2],
    double n2[2], double n3[2]);

/**************************** newtonPotential *****************************/
/* INPUT: n1, n2, n3 are the nodes of a triangle T, given in counter      */
/*        clockwise order. x is a point in R^2.                           */
/* OUTPUT: The function returns the newton-potential                      */
/*                                                                        */
/*        \int_{T} log(|x-t|) dt                                          */
/*                                                                        */
/* DESCRIPTION: The result is computed analytically up to a quadrature of */
/*        a smooth function.                                              */
/**************************************************************************/
double newtonPotential(double n1[2], double n2[2],
                        double n3[2], double x[2]);

/****************************** evalAtanInt *******************************/
/* INPUT: n1, n2 and n3 are vertices of the triangle T.                   */
/*        x is a point in \R^2.                                           */
/* OUTPUT: The function returns the value of the integral                 */
/*                                                                        */
/*        \int_0^1 \int_0^(1-\xi) \Delta / (c + \eta b + \eta^2) d \eta   */
/*        d\xi dx                                                         */
/*                                                                        */
/*        where \Delta, a, b, c are defined as in the documentation for   */
/*        the function innerAtanInt. This integral arises as part of the  */
/*        computation of integrateAtanInt, but it is also used from       */
/*        within the function newtonPotential.                            */
/* DESCRIPTION: We use a Gauss rule for the outer integral whereas the    */
/*        inner integral is computed analytically using the function      */
/*        innerAtanInt.                                                   */
/**************************************************************************/
double evalAtanInt(double n1[2], double n2[2], double n3[2], double x[2]);

/****************************** innerAtanInt ******************************/
/* INPUT: xi is a value between 0 and 1. a, b, c are defined by a :=      */
/*        |u|^2 b := 2u \dot (w+\xi v), c := |w+\xi v|^2 where u, v and w */
/*        are defined by u := -n2+n1, v := -n3+n1, w := x-n1.             */
/* OUTPUT: The function returns the value of the integral                 */
/*                                                                        */
/*        \int_0^\xi \Delta / (c+\eta b+\eta^2 a) d \eta.                 */
/*                                                                        */
/*        where \Delta is defined by \Delta := 4*a*c-b^2.                 */
/* DESCRIPTION: Computes the integral and returns its value.              */
/**************************************************************************/
double innerAtanInt(double a, double b, double c, double xi);

/**************************** integrateAtanInt ****************************/
/* INPUT: a and b are the end points of a boundary mesh element E_j and   */
/*        n1,n2 n3 are the corner points of the volume mesh element T_k.  */
/* OUTPUT: The function returns the value of the integral                 */
/*                                                                        */
/*        \int_E_j \int_0^1 \int_0^1-\xi \Delta / (c+\eta b+\eta^2 a) d   */
/*        \eta d \xi d x,                                                 */
/*                                                                        */
/*        where \Delta, a, b and c are defined as in the documentation    */
/*        for the function innerAtanInt.                                  */
/* DESCRIPTION: Computes the integral by transforming it to an integral   */
/*        over the reference interval (-1,1) and the usage of a gauss     */
/*        quadrature formular. The inner integrals are evaluated using    */
/*        the evalAtanInt function.                                       */
/**************************************************************************/
double integrateAtanInt(double a[2], double b[2],
                        double n1[2], double n2[2], double n3[2]);
#endif

