function answer = ...
  configureQuestion(question, symbol, defval, type, varargin)
%CONFIGUREQUESTION   Configuration of HILBERT Toolbox.
%   CONFIGUREQUESTION(question, symbol, defval, type, varargin) provides a 
%   way to ask the user about HILBERT's configuration.
%
%   This file is part of the building system for HILBERT. HILBERT's building
%   system is a recursive, dependency-based system similar to "make".
%
%   CONFIGUREQUESTION provides a way to ask the user about HILBERT's
%   configuration. It is aware of certain types ("number","integer", 
%   "matrix", "boolean") and one may provide a list of default values. 
%   CONFIGUREQUESTION checks the user input and asks the user the same 
%   question until it has got a valid answer.
%
%   In the signature above, question is a text representation of the
%   question, the function will ask the user. symbol is the (public) name of
%   the variable that is set. defval is the default value, type is the input
%   type and may be either "number","integer", "matrix" or "bool". 
%   Furthermore, one may provide a list of possible values as further 
%   parameters. If the list is empty, any value of the given type is a valid
%   option.

% (C) 2009-2013 HILBERT-Team '09, '10, '12, '13
% support + bug report:  hilbert@asc.tuwien.ac.at
%
% Version: 3.1

  disp(sprintf('%s? [%s = %s]', question, symbol, defval)); 
   
  if (nargin > 5)
    options = '';
    for i=6:nargin
      options = [options, varargin{i-5}, ' | '];
    end
    options = [options, varargin{nargin-4}];
    disp(sprintf('(%s)', options));
  end
  
  answer = input([symbol ' = '], 's');
  if (strcmp(answer, ''))
    answer = defval;
  end
  
  if (nargin > 5)
    okay = false;
    for i=5:nargin
      if (strcmp(answer, varargin(i-4)))
        okay = true;
        break;
      end
    end
  
    if (okay == false)
      disp('Your answer is not one of the possible values.');
      answer = ...
          configureQuestion(question, symbol, defval, type,  varargin{1:length(varargin)});
      return;
    end
  end
  
  switch (type)
    case 'num'
      [x, s] = str2num(answer);
      if (s ~= true || sum(size(x) ~= [1 1]))
        disp('Invalid type, must be a number.');
        answer = ...
            configureQuestion(question, symbol, defval, type, varargin{1:length(varargin)});
        return;
      end
    case 'mat'
      [x, s] = str2num(answer);
      if (s ~= true)
        disp('Invalid type, must be a matrix.');
        answer = ...
            configureQuestion(question, symbol, defval, type, varargin{1:length(varargin)});
        return;
      end
    case 'int'
      [x, s] = str2num(answer);
      if (s ~= true || sum(size(x) ~= [1 1]) || x - round(x) ~= 0)
        disp('Invalid type, must be an integer.');
        answer = ...
            configureQuestion(question, symbol, defval, type, varargin{1:length(varargin)});
        return;
      end
    case 'bool'
      answer = lower(answer);
      if (strcmp(answer, 'true') || strcmp(answer, 'yes') ...
            || strcmp(answer, 'y'))
        answer = 'yes';
      elseif (strcmp(answer, 'false') || strcmp(answer, 'no') ...
            || strcmp(answer, 'n'))
        answer = 'no';
      else
        disp('Invalid type, must be a boolean (yes | no)');
        answer = ...
            configureQuestion(question, symbol, defval, type, varargin{1:length(varargin)});
        return;
      end
    case 'str'
    otherwise
      error(['Unknown type ' type ' in configureQuestion!']);
  end
end